<?php
// app/Controllers/ApiToken.php

namespace Controllers;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Security\ApiAuth;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Error\Logger;
use Models\Entity\Korisnik;

class ApiToken extends BaseController
{
    private ApiAuth $apiAuth;
    
    public function __construct(
        ApiAuth $apiAuth,
        Authentication $authentication,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->apiAuth = $apiAuth;
    }
    
    /**
     * API Tokens management view
     */
    public function index(Request $request): Response
    {
        if (!$this->authentication->isLoggedIn()) {
            return Response::redirect('login');
        }
        
        return $this->response()->render(
            'ApiToken/index.html.php',
            [
                'title' => 'API Token Management'
            ],
            'API Token Management'
        );
    }
    
    /**
     * List API tokens for the current user
     */
    public function list(Request $request): Response
    {
        if (!$this->authentication->isLoggedIn()) {
            return Response::json([
                'error' => 'Unauthorized',
                'message' => 'You must be logged in to list API tokens'
            ], 401);
        }
        
        $user = $this->authentication->getUser();
        $tokens = $this->apiAuth->getUserTokens($user->id);
        
        return Response::json([
            'success' => true,
            'tokens' => $tokens
        ]);
    }
    
    /**
     * Generate a new API token
     */
    public function generate(Request $request): Response
    {
        if (!$this->authentication->isLoggedIn()) {
            return Response::json([
                'error' => 'Unauthorized',
                'message' => 'You must be logged in to generate API tokens'
            ], 401);
        }
        
        // Validate description
        $description = $request->getPost('description');
        if (!$description) {
            return Response::json([
                'error' => 'Validation Error',
                'message' => 'Description is required'
            ], 422);
        }
        
        // Get scopes from request
        $scopes = $request->getPost('scopes');
        if (!empty($scopes)) {
            if (is_string($scopes)) {
                $scopes = explode(',', $scopes);
            }
            // Trim whitespace from scopes
            $scopes = array_map('trim', $scopes);
            // Filter out empty scopes
            $scopes = array_filter($scopes);
        } else {
            $scopes = null; // Will use default scopes
        }
        
        // Generate token with personal access token method
        $token = $this->apiAuth->generatePersonalAccessToken($description, $scopes);
        
        if (!$token) {
            return Response::json([
                'error' => 'Token Generation Failed',
                'message' => 'Failed to generate token'
            ], 500);
        }
        
        $this->logger->security('API token generated', [
            'user_id' => $this->authentication->getUser()->id,
            'token_id' => $token['id'],
            'scopes' => implode(',', $token['scopes']),
            'ip' => $request->getIp()
        ]);
        
        return Response::json([
            'success' => true,
            'token' => $token
        ]);
    }
    
    /**
     * Revoke an API token
     */
    public function revoke(Request $request): Response
    {
        if (!$this->authentication->isLoggedIn()) {
            return Response::json([
                'error' => 'Unauthorized',
                'message' => 'You must be logged in to revoke API tokens'
            ], 401);
        }
        
        $tokenId = $request->getPost('token_id');
        if (!$tokenId) {
            return Response::json([
                'error' => 'Validation Error',
                'message' => 'Token ID is required'
            ], 422);
        }
        
        $user = $this->authentication->getUser();
        
        // Check if current user is admin or master
        $canManageOtherTokens = ($user->user_permissions & (Korisnik::KORISNIK_ADMIN | Korisnik::KORISNIK_MASTER)) > 0;
        
        // Get user ID parameter if provided and user has admin rights
        $userId = $canManageOtherTokens ? ($request->getPost('user_id') ?: $user->id) : $user->id;
        
        // Revoke the token for the specified user (or current user)
        $success = $this->apiAuth->revokeToken($tokenId, $userId);
        
        if (!$success) {
            return Response::json([
                'error' => 'Revocation Failed',
                'message' => 'Failed to revoke token or token not found'
            ], 404);
        }
        
        $this->logger->security('API token revoked', [
            'user_id' => $user->id,
            'token_id' => $tokenId,
            'target_user_id' => $userId,
            'ip' => $request->getIp()
        ]);
        
        return Response::json([
            'success' => true,
            'message' => 'Token revoked successfully'
        ]);
    }
    
    /**
     * Revoke all tokens for a user
     */
    public function revokeAll(Request $request): Response
    {
        if (!$this->authentication->isLoggedIn()) {
            return Response::json([
                'error' => 'Unauthorized',
                'message' => 'You must be logged in to revoke API tokens'
            ], 401);
        }
        
        $user = $this->authentication->getUser();
        
        // Check if current user is admin or master
        $canManageOtherTokens = ($user->user_permissions & (Korisnik::KORISNIK_ADMIN | Korisnik::KORISNIK_MASTER)) > 0;
        
        // Get user ID parameter if provided and user has admin rights
        $userId = $canManageOtherTokens ? ($request->getPost('user_id') ?: $user->id) : $user->id;
        
        // Revoke all tokens for the specified user (or current user)
        $count = $this->apiAuth->revokeAllUserTokens($userId);
        
        $this->logger->security('All API tokens revoked', [
            'user_id' => $user->id,
            'target_user_id' => $userId,
            'count' => $count,
            'ip' => $request->getIp()
        ]);
        
        return Response::json([
            'success' => true,
            'message' => 'All tokens revoked successfully',
            'count' => $count
        ]);
    }
}