<?php

namespace Controllers;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Database\DatabaseConnection;
use baseKRIZAN\BORNA\BornaManager;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Session\SessionManager;

/**
 * Controller for the BORNA Security Module dashboard
 */
class BORNA extends BaseController
{
    protected ?BornaManager $securityManager = null;
    protected ?Container $container;

    /**
     * Constructor
     */
    public function __construct(Container $container)
    {
        $this->container = $container;
        $logger = $container->get('logger');
        $sessionManager = $container->get('sessionManager');
        $authentication = $container->has('authentication') ? $container->get('authentication') : null;
        
        parent::__construct($logger, $sessionManager, $authentication, $container);
        
        // Safely get the bornaManager if it exists
        if ($container->has('bornaManager')) {
            $this->securityManager = $container->get('bornaManager');
        } else {
            $this->securityManager = null;
        }
    }

    /**
     * Main security dashboard action
     */
    public function index(Request $request): Response
    {
        // Check if manager exists first, then check if it's enabled
        if ($this->securityManager === null || !$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Get security statistics
        $securityStats = $this->securityManager->getSecurityStats();
        
        // Render the dashboard using BaseController's render method
        return $this->response()->render('BORNA/index.html.php', [
            'securityStats' => $securityStats,
            'active_page' => 'dashboard'
        ], 'BORNA Security Dashboard');
    }

    /**
     * Security overview page
     */
    public function overview(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Get security statistics
        $securityStats = $this->securityManager->getSecurityStats();
        
        // Get blocked IPs
        $blockedIPs = $this->securityManager->getBlockedIPs();
        
        // Get recent events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        $recentEvents = $securityEvents->getRecentEvents(10);
        
        // Get time period for event counts
        $from = new \DateTime('-24 hours');
        $to = new \DateTime();
        $eventCounts = $securityEvents->getEventCounts($from, $to);
        
        // Get most active IPs
        $activeIPs = $securityEvents->getMostActiveIPs(5, $from, $to);
        
        // Get most attacked paths
        $attackedPaths = $securityEvents->getMostAttackedPaths(5, $from, $to);
        
        return $this->response()->render('BORNA/overview.html.php', [
            'securityStats' => $securityStats,
            'blockedIPs' => $blockedIPs,
            'recentEvents' => $recentEvents,
            'eventCounts' => $eventCounts,
            'activeIPs' => $activeIPs,
            'attackedPaths' => $attackedPaths,
            'active_page' => 'overview'
        ], 'BORNA - Security Overview');
    }

    /**
     * Rules management page
     */
    public function rules(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Get firewall rules
        $firewallRules = $this->container->get('bornaFirewallRules');
        $rules = $firewallRules->getAllRules();
        
        // Check for rule action (add, edit, delete)
        $action = $request->getQuery()['action'] ?? '';
        $ruleId = $request->getQuery()['id'] ?? '';
        
        // Handle rule actions
        if ($request->getMethod() === 'POST') {
            if ($action === 'add') {
                $this->handleAddRule($request, $firewallRules);
            } elseif ($action === 'edit' && !empty($ruleId)) {
                $this->handleEditRule($request, $firewallRules, $ruleId);
            } elseif ($action === 'delete' && !empty($ruleId)) {
                $firewallRules->deleteRule($ruleId);
                return new Response('', 302, [
                    'Location' => 'rules?deleted=1'
                ]);
            } else if ($action === 'toggle' && !empty($ruleId)) {
                $post = $request->getPost();
                $enabled = ($post['enabled'] ?? '0') === '1';
                $firewallRules->setRuleEnabled($ruleId, $enabled);
                return new Response('', 302, [
                    'Location' => 'rules?toggled=1'
                ]);
            }
        }
        
        // Get rule for editing
        $editRule = null;
        if ($action === 'edit' && !empty($ruleId)) {
            $editRule = $firewallRules->getRule($ruleId);
        }
        
        // Render the rules template
        return $this->response()->render('BORNA/rules.html.php', [
            'rules' => $rules,
            'action' => $action,
            'editRule' => $editRule,
            'config' => $this->securityManager->getConfig(),
            'success' => ($request->getQuery()['success'] ?? '0') === '1',
            'deleted' => ($request->getQuery()['deleted'] ?? '0') === '1',
            'toggled' => ($request->getQuery()['toggled'] ?? '0') === '1',
            'active_page' => 'rules'
        ], 'BORNA - Firewall Rules');
    }
    
    /**
     * IP management page
     */
    public function ips(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Get blocked IPs
        $blockedIPs = $this->securityManager->getBlockedIPs();
        
        // Handle IP actions
        if ($request->getMethod() === 'POST') {
            $post = $request->getPost();
            $action = $post['action'] ?? '';
            
            if ($action === 'block') {
                $post = $request->getPost();
                $ip = $post['ip'] ?? '';
                $reason = $post['reason'] ?? 'Manually blocked';
                $duration = (int)($post['duration'] ?? 24);
                
                if (!empty($ip)) {
                    $this->securityManager->blockIP($ip, $reason, $duration);
                    return new Response('', 302, [
                        'Location' => 'ips?blocked=1'
                    ]);
                }
            } elseif ($action === 'unblock') {
                $post = $request->getPost();
                $ip = $post['ip'] ?? '';
                
                if (!empty($ip)) {
                    $this->securityManager->unblockIP($ip);
                    return new Response('', 302, [
                        'Location' => 'ips?unblocked=1'
                    ]);
                }
            }
        }
        
        // Render the IPs template
        return $this->response()->render('BORNA/ips.html.php', [
            'blockedIPs' => $blockedIPs,
            'config' => $this->securityManager->getConfig(),
            'blocked' => ($request->getQuery()['blocked'] ?? '0') === '1',
            'unblocked' => ($request->getQuery()['unblocked'] ?? '0') === '1',
            'active_page' => 'ips'
        ], 'BORNA - Blocked IPs');
    }
    
    /**
     * Events page
     */
    public function events(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Get security events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        
        // Handle filter parameters
        $fromDate = $request->getQuery()['from'] ?? '-24 hours';
        $toDate = $request->getQuery()['to'] ?? 'now';
        $eventType = $request->getQuery()['type'] ?? '';
        
        // Convert to DateTime objects
        $from = new \DateTime($fromDate);
        $to = new \DateTime($toDate);
        
        // Get filtered events
        $events = $securityEvents->getEvents($from, $to, empty($eventType) ? null : $eventType);
        
        // Get event counts
        $eventCounts = $securityEvents->getEventCounts($from, $to);
        
        // Handle export action
        if (($request->getQuery()['export'] ?? '') === '1') {
            $format = $request->getQuery()['format'] ?? 'json';
            $exportPath = $securityEvents->exportEvents($format);
            
            // Create a download response
            $filename = basename($exportPath);
            $contentType = $format === 'json' ? 'application/json' : 'text/csv';
            
            return new Response(file_get_contents($exportPath), 200, [
                'Content-Type' => $contentType,
                'Content-Disposition' => 'attachment; filename="' . $filename . '"'
            ]);
        }
        
        // Render the events template
        return $this->response()->render('BORNA/events.html.php', [
            'events' => $events,
            'eventCounts' => $eventCounts,
            'from' => $from->format('Y-m-d H:i:s'),
            'to' => $to->format('Y-m-d H:i:s'),
            'eventType' => $eventType,
            'config' => $this->securityManager->getConfig(),
            'active_page' => 'events'
        ], 'BORNA - Security Events');
    }
    
    /**
     * Settings page
     */
    public function settings(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Handle settings update
        if ($request->getMethod() === 'POST') {
            $post = $request->getPost();
            $securityLevel = (int)($post['security_level'] ?? 1);
            
            // Update security level
            $this->securityManager->setSecurityLevel($securityLevel);
            
            // Update geo protection settings
            if ($this->container->has('bornaGeoAnalyzer')) {
                $geoAnalyzer = $this->container->get('bornaGeoAnalyzer');
                
                $allowedCountries = explode(',', $post['allowed_countries'] ?? '');
                $blockedCountries = explode(',', $post['blocked_countries'] ?? '');
                
                // Filter and clean country codes
                $allowedCountries = array_filter(array_map('trim', $allowedCountries));
                $blockedCountries = array_filter(array_map('trim', $blockedCountries));
                
                $geoAnalyzer->setAllowedCountries($allowedCountries);
                $geoAnalyzer->setBlockedCountries($blockedCountries);
            }
            
            // Update pattern analyzer settings
            if ($this->container->has('bornaPatternAnalyzer')) {
                $patternAnalyzer = $this->container->get('bornaPatternAnalyzer');
                
                $xssProtection = ($post['xss_protection'] ?? '0') === '1';
                $sqliProtection = ($post['sqli_protection'] ?? '0') === '1';
                
                $patternAnalyzer->setXssProtection($xssProtection);
                $patternAnalyzer->setSqliProtection($sqliProtection);
            }
            
            // Redirect to prevent resubmission
            return new Response('', 302, [
                'Location' => 'settings?updated=1'
            ]);
        }
        
        // Get current settings
        $currentSettings = [
            'security_level' => $this->securityManager->getSecurityLevel(),
            'allowed_countries' => [],
            'blocked_countries' => [],
            'xss_protection' => true,
            'sqli_protection' => true
        ];
        
        // Get geolocation settings
        if ($this->container->has('bornaGeoAnalyzer')) {
            $geoAnalyzer = $this->container->get('bornaGeoAnalyzer');
            $currentSettings['allowed_countries'] = $geoAnalyzer->getAllowedCountries();
            $currentSettings['blocked_countries'] = $geoAnalyzer->getBlockedCountries();
        }
        
        // Get pattern analyzer settings
        if ($this->container->has('bornaPatternAnalyzer')) {
            $patternAnalyzer = $this->container->get('bornaPatternAnalyzer');
            $currentSettings['xss_protection'] = $patternAnalyzer->isXssProtectionEnabled();
            $currentSettings['sqli_protection'] = $patternAnalyzer->isSqliProtectionEnabled();
        }
        
        // Render the settings template
        return $this->response()->render('BORNA/settings.html.php', [
            'settings' => $currentSettings,
            'config' => $this->securityManager->getConfig(),
            'updated' => ($request->getQuery()['updated'] ?? '0') === '1',
            'active_page' => 'settings'
        ], 'BORNA - Security Settings');
    }
    
    /**
     * Reports page
     */
    public function reports(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return $this->response()->renderError('BORNA Security Module is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Handle report generation
        $reportType = $request->getQuery()['type'] ?? 'security';
        $fromDate = $request->getQuery()['from'] ?? '-7 days';
        $toDate = $request->getQuery()['to'] ?? 'now';
        
        // Convert to DateTime objects
        $from = new \DateTime($fromDate);
        $to = new \DateTime($toDate);
        
        // Get security events for the period
        $securityEvents = $this->container->get('bornaSecurityEvents');
        $events = $securityEvents->getEvents($from, $to);
        
        // Generate report data based on type
        $reportData = $this->generateReportData($reportType, $events, $from, $to);
        
        // Render the reports template
        return $this->response()->render('BORNA/reports.html.php', [
            'reportType' => $reportType,
            'reportData' => $reportData,
            'from' => $from->format('Y-m-d'),
            'to' => $to->format('Y-m-d'),
            'config' => $this->securityManager->getConfig(),
            'active_page' => 'reports'
        ], 'BORNA - Security Reports');
    }
    
    /**
     * API endpoint to get security state
     */
    public function getState(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get security statistics
        $securityStats = $this->securityManager->getSecurityStats();
        
        // Return JSON response
        return Response::json([
            'success' => true,
            'state' => $securityStats
        ]);
    }
    
    /**
     * API endpoint to get attack data
     */
    public function getAttacks(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get period from request
        $period = $request->getQuery()['period'] ?? '24h';
        
        // Convert period to DateTime
        switch ($period) {
            case '24h':
                $from = new \DateTime('-24 hours');
                break;
            case '7d':
                $from = new \DateTime('-7 days');
                break;
            case '30d':
                $from = new \DateTime('-30 days');
                break;
            default:
                $from = new \DateTime('-24 hours');
        }
        
        $to = new \DateTime();
        
        // Get security events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        $attacks = $securityEvents->getAttackEvents($from, $to);
        
        // Group attacks by hour
        $hourlyData = $this->groupAttacksByHour($attacks);
        
        // Return JSON response
        return Response::json([
            'success' => true,
            'period' => $period,
            'total_attacks' => count($attacks),
            'hourly_data' => $hourlyData
        ]);
    }

    /**
     * Group attacks by hour for charts
     */
    private function groupAttacksByHour(array $attacks): array
    {
        $hourlyData = array_fill(0, 24, 0);
        $currentHour = (int)date('G');
        
        foreach ($attacks as $attack) {
            if (isset($attack['timestamp'])) {
                $attackHour = (int)date('G', strtotime($attack['timestamp']));
                $hourOffset = ($currentHour - $attackHour + 24) % 24;
                $hourlyData[$hourOffset]++;
            }
        }
        
        return $hourlyData;
    }

    /**
     * Generate report data based on type
     */
    private function generateReportData(string $reportType, array $events, \DateTime $from, \DateTime $to): array
    {
        $reportData = [];
        
        switch ($reportType) {
            case 'security':
                // Security overview report
                $attackEvents = array_filter($events, function($event) {
                    return $event['event_type'] === 'attack_detected';
                });
                
                $blockedEvents = array_filter($events, function($event) {
                    return $event['event_type'] === 'ip_blocked';
                });
                
                $reportData = [
                    'total_events' => count($events),
                    'attack_events' => count($attackEvents),
                    'blocked_events' => count($blockedEvents),
                    'period_days' => $from->diff($to)->days + 1,
                    'attacks_by_day' => $this->groupEventsByDay($attackEvents),
                    'blocks_by_day' => $this->groupEventsByDay($blockedEvents)
                ];
                break;
                
            case 'geographic':
                // Geographic distribution report
                $reportData = $this->analyzeGeographicDistribution($events);
                break;
                
            case 'threats':
                // Threat analysis report
                $reportData = $this->analyzeThreatTypes($events);
                break;
        }
        
        return $reportData;
    }
    
    /**
     * Group events by day
     */
    private function groupEventsByDay(array $events): array
    {
        $byDay = [];
        
        foreach ($events as $event) {
            $date = substr($event['timestamp'], 0, 10); // Get YYYY-MM-DD part
            
            if (!isset($byDay[$date])) {
                $byDay[$date] = 0;
            }
            
            $byDay[$date]++;
        }
        
        // Sort by date
        ksort($byDay);
        
        return $byDay;
    }
    
    /**
     * Analyze geographic distribution of attacks
     */
    private function analyzeGeographicDistribution(array $events): array
    {
        $countries = [];
        $ips = [];
        
        foreach ($events as $event) {
            // Get country from event data if available
            $country = $event['data']['country'] ?? $event['data']['country_code'] ?? null;
            $ip = $event['data']['ip'] ?? null;
            
            if ($country) {
                if (!isset($countries[$country])) {
                    $countries[$country] = 0;
                }
                
                $countries[$country]++;
            }
            
            if ($ip) {
                if (!isset($ips[$ip])) {
                    $ips[$ip] = 0;
                }
                
                $ips[$ip]++;
            }
        }
        
        // Sort by count descending
        arsort($countries);
        arsort($ips);
        
        return [
            'countries' => $countries,
            'ips' => $ips
        ];
    }
    
    /**
     * Analyze threat types from attack events
     */
    private function analyzeThreatTypes(array $events): array
    {
        $threatTypes = [];
        $paths = [];
        
        foreach ($events as $event) {
            if ($event['event_type'] === 'attack_detected' && isset($event['data']['threats'])) {
                foreach ($event['data']['threats'] as $threat) {
                    $type = $threat['type'] ?? 'unknown';
                    
                    if (!isset($threatTypes[$type])) {
                        $threatTypes[$type] = 0;
                    }
                    
                    $threatTypes[$type]++;
                }
            }
            
            // Track attacked paths
            if (isset($event['data']['path'])) {
                $path = $event['data']['path'];
                
                if (!isset($paths[$path])) {
                    $paths[$path] = 0;
                }
                
                $paths[$path]++;
            }
        }
        
        // Sort by count descending
        arsort($threatTypes);
        arsort($paths);
        
        return [
            'threat_types' => $threatTypes,
            'paths' => $paths
        ];
    }
    
    /**
     * Handle adding a new rule
     */
    private function handleAddRule(Request $request, $firewallRules): void
    {
        try {
            $post = $request->getPost();
            $rule = [
                'name' => $post['name'] ?? '',
                'type' => $post['type'] ?? '',
                'pattern' => $post['pattern'] ?? '',
                'action' => $post['action'] ?? '',
                'enabled' => ($post['enabled'] ?? '0') === '1',
                'description' => $post['description'] ?? ''
            ];
            
            // Add rule
            $firewallRules->addRule($rule);
            
            // Redirect with success
            header('Location: rules?success=1');
            exit;
        } catch (\Exception $e) {
            $this->logger->error('Failed to add rule', [
                'error' => $e->getMessage()
            ]);
            
            // Redirect with error
            header('Location: rules?error=' . urlencode($e->getMessage()));
            exit;
        }
    }
    
    /**
     * Handle editing a rule
     */
    private function handleEditRule(Request $request, $firewallRules, string $ruleId): void
    {
        try {
            $post = $request->getPost();
            $rule = [
                'name' => $post['name'] ?? '',
                'type' => $post['type'] ?? '',
                'pattern' => $post['pattern'] ?? '',
                'action' => $post['action'] ?? '',
                'enabled' => ($post['enabled'] ?? '0') === '1',
                'description' => $post['description'] ?? ''
            ];
            
            // Update rule
            $firewallRules->updateRule($ruleId, $rule);
            
            // Redirect with success
            header('Location: rules?success=1');
            exit;
        } catch (\Exception $e) {
            $this->logger->error('Failed to update rule', [
                'error' => $e->getMessage(),
                'rule_id' => $ruleId
            ]);
            
            // Redirect with error
            header('Location: rules?error=' . urlencode($e->getMessage()));
            exit;
        }
    }



    /**
     * API endpoint to search IPs
     */
    public function searchIPs(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get search query
        $searchQuery = $request->getQuery()['q'] ?? '';
        
        // Get blocked IPs
        $blockedIPs = $this->securityManager->getBlockedIPs();
        
        // Filter IPs
        $filteredIPs = array_filter($blockedIPs, function($ip) use ($searchQuery) {
            if (empty($searchQuery)) return true;
            
            $searchLower = strtolower($searchQuery);
            return 
                stripos($ip['ip'], $searchLower) !== false || 
                stripos($ip['reason'] ?? '', $searchLower) !== false;
        });
        
        // Prepare response
        $responseData = [
            'success' => true,
            'ips' => array_values($filteredIPs)
        ];
        
        return Response::json($responseData);
    }

    /**
     * API endpoint to get IP information
     */
    public function getIPInfo(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get IP from query
        $ip = $request->getQuery()['ip'] ?? '';
        
        if (empty($ip)) {
            return Response::json([
                'success' => false,
                'message' => 'No IP address provided'
            ], 400);
        }
        
        // Use geolocation service (you'll need to implement this)
        $geoInfo = $this->getIPGeolocation($ip);
        
        // Get previous blocks for this IP
        $blockedIPs = $this->securityManager->getBlockedIPs();
        $previousBlocks = isset($blockedIPs[$ip]) ? 1 : 0;
        
        // Prepare response
        $responseData = [
            'success' => true,
            'ip' => $ip,
            'country' => $geoInfo['country'] ?? 'Unknown',
            'city' => $geoInfo['city'] ?? null,
            'isp' => $geoInfo['isp'] ?? null,
            'threat_score' => $geoInfo['threat_score'] ?? null,
            'previous_blocks' => $previousBlocks,
            'threat_info' => $geoInfo['threat_info'] ?? null
        ];
        
        return Response::json($responseData);
    }

    /**
     * API endpoint to search rules
     */
    public function searchRules(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get search query
        $searchQuery = $request->getQuery()['q'] ?? '';
        
        // Get firewall rules
        $firewallRules = $this->container->get('bornaFirewallRules');
        $allRules = $firewallRules->getAllRules();
        
        // Filter rules
        $filteredRules = array_filter($allRules, function($rule) use ($searchQuery) {
            if (empty($searchQuery)) return true;
            
            $searchLower = strtolower($searchQuery);
            return 
                stripos($rule['name'], $searchLower) !== false || 
                stripos($rule['description'] ?? '', $searchLower) !== false || 
                stripos($rule['type'], $searchLower) !== false || 
                stripos($rule['pattern'], $searchLower) !== false;
        });
        
        // Prepare response
        $responseData = [
            'success' => true,
            'rules' => array_values($filteredRules)
        ];
        
        return Response::json($responseData);
    }

    /**
     * API endpoint to search events
     */
    public function searchEvents(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get search query
        $searchQuery = $request->getQuery()['q'] ?? '';
        
        // Get security events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        
        // Use last 30 days as default range
        $from = new \DateTime('-30 days');
        $to = new \DateTime();
        
        $events = $securityEvents->getEvents($from, $to);
        
        // Filter events
        $filteredEvents = array_filter($events, function($event) use ($searchQuery) {
            if (empty($searchQuery)) return true;
            
            $searchLower = strtolower($searchQuery);
            
            // Check various fields for match
            return 
                stripos($event['event_type'], $searchLower) !== false ||
                (isset($event['data']['ip']) && stripos($event['data']['ip'], $searchLower) !== false) ||
                (isset($event['data']['path']) && stripos($event['data']['path'], $searchLower) !== false) ||
                (isset($event['data']['country']) && stripos($event['data']['country'], $searchLower) !== false);
        });
        
        // Prepare response
        $responseData = [
            'success' => true,
            'events' => array_values($filteredEvents)
        ];
        
        return Response::json($responseData);
    }

    /**
     * API endpoint to get specific event details
     */
    public function getEventDetails(Request $request, string $eventId): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get security events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        
        // Use last 30 days as default range
        $from = new \DateTime('-30 days');
        $to = new \DateTime();
        
        $events = $securityEvents->getEvents($from, $to);
        
        // Find specific event
        $event = null;
        foreach ($events as $e) {
            if ($e['id'] === $eventId) {
                $event = $e;
                break;
            }
        }
        
        if (!$event) {
            return Response::json([
                'success' => false,
                'message' => 'Event not found'
            ], 404);
        }
        
        // Prepare response
        $responseData = [
            'success' => true,
            'event' => $event
        ];
        
        return Response::json($responseData);
    }

    /**
     * API endpoint to test connection (for settings)
     */
    public function testConnection(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get connection details from request body
        $data = json_decode($request->getBody(), true);
        $type = $data['type'] ?? '';
        
        try {
            if ($type === 'redis') {
                // Test Redis connection
                $redisConfig = [
                    'host' => $data['host'] ?? '127.0.0.1',
                    'port' => $data['port'] ?? 6379,
                    'password' => $data['password'] ?? null,
                    'database' => $data['database'] ?? 0
                ];
                
                $redis = new \Redis();
                $connected = $redis->connect($redisConfig['host'], $redisConfig['port']);
                
                if (!$connected) {
                    return Response::json([
                        'success' => false,
                        'message' => 'Failed to connect to Redis'
                    ], 400);
                }
                
                // Authenticate if password provided
                if (!empty($redisConfig['password'])) {
                    $authenticated = $redis->auth($redisConfig['password']);
                    if (!$authenticated) {
                        return Response::json([
                            'success' => false,
                            'message' => 'Redis authentication failed'
                        ], 400);
                    }
                }
                
                // Select database
                $redis->select($redisConfig['database']);
                
                return Response::json([
                    'success' => true,
                    'message' => 'Redis connection successful'
                ]);
            } else {
                // Default database connection test
                $dbConnection = DatabaseConnection::getInstance();
                $result = $dbConnection->querySingleValue('SELECT 1');

                // Provjera je li upit uspješno izvršen
                if ($result !== null && $result == 1) {
                    return Response::json([
                        'success' => true,
                        'message' => 'Database connection successful'
                    ]);
                } else {
                    return Response::json([
                        'success' => false,
                        'message' => 'Database connection failed'
                    ], 500);
                }
            }
        } catch (\Throwable $e) {
            return Response::json([
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * API endpoint to reset settings
     */
    public function resetSettings(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        try {
            // Reset to default security level
            $this->securityManager->setSecurityLevel(\baseKRIZAN\BORNA\Enum\SecurityLevel::MEDIUM);
            
            // Reset geolocation settings
            if ($this->container->has('bornaGeoAnalyzer')) {
                $geoAnalyzer = $this->container->get('bornaGeoAnalyzer');
                $geoAnalyzer->setAllowedCountries([]);
                $geoAnalyzer->setBlockedCountries([]);
            }
            
            // Reset pattern analyzer settings
            if ($this->container->has('bornaPatternAnalyzer')) {
                $patternAnalyzer = $this->container->get('bornaPatternAnalyzer');
                $patternAnalyzer->setXssProtection(true);
                $patternAnalyzer->setSqliProtection(true);
            }
            
            return Response::json([
                'success' => true,
                'message' => 'Settings reset to default values'
            ]);
        } catch (\Throwable $e) {
            return Response::json([
                'success' => false,
                'message' => 'Failed to reset settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper method to get IP geolocation (You'll need to implement this)
     */
    private function getIPGeolocation(string $ip): array
    {
        // Use a third-party IP geolocation service
        try {
            // This is just an example. You'd replace with an actual API call
            $geoApiUrl = "https://ipapi.co/{$ip}/json/";
            
            $context = stream_context_create([
                'http' => [
                    'timeout' => 5 // 5 seconds
                ]
            ]);
            
            $response = @file_get_contents($geoApiUrl, false, $context);
            
            if ($response === false) {
                return [
                    'country' => 'Unknown',
                    'city' => null,
                    'isp' => null,
                    'threat_score' => rand(0, 100),
                    'threat_info' => null
                ];
            }
            
            $geoData = json_decode($response, true);
            
            // Estimate threat score based on geolocation data
            $threatScore = 0;
            
            // Example threat scoring logic
            $highRiskCountries = ['RU', 'CN', 'KP', 'IR'];
            if (in_array($geoData['country_code'] ?? '', $highRiskCountries)) {
                $threatScore += 50;
            }
            
            return [
                'country' => $geoData['country_name'] ?? 'Unknown',
                'city' => $geoData['city'] ?? null,
                'isp' => $geoData['org'] ?? null,
                'threat_score' => min(100, max(0, $threatScore)),
                'threat_info' => $threatScore > 50 ? 'High-risk geolocation' : null
            ];
        } catch (\Throwable $e) {
            // Log the error
            $this->logger->error('IP Geolocation lookup failed', [
                'ip' => $ip,
                'error' => $e->getMessage()
            ]);
            
            return [
                'country' => 'Unknown',
                'city' => null,
                'isp' => null,
                'threat_score' => rand(0, 100),
                'threat_info' => null
            ];
        }
    }

    /**
     * API endpoint to generate report
     */
    public function generateReport(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get report parameters from request body
        $data = json_decode($request->getBody(), true);
        $reportType = $data['report_type'] ?? 'security';
        $fromDate = $data['from_date'] ?? '-7 days';
        $toDate = $data['to_date'] ?? 'now';
        
        // Convert to DateTime objects
        $from = new \DateTime($fromDate);
        $to = new \DateTime($toDate);
        
        // Get security events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        $events = $securityEvents->getEvents($from, $to);
        
        // Generate report data
        $reportData = $this->generateReportData($reportType, $events, $from, $to);
        
        // Prepare response
        $responseData = [
            'success' => true,
            'report_type' => $reportType,
            'from_date' => $from->format('Y-m-d'),
            'to_date' => $to->format('Y-m-d'),
            'report_data' => $reportData
        ];
        
        return Response::json($responseData);
    }

    /**
     * API endpoint to export report
     */
    public function exportReport(Request $request): Response
    {
        // Check if BORNA is enabled
        if (!$this->securityManager->isEnabled()) {
            return Response::json(['error' => 'BORNA Security Module is disabled'], 503);
        }
        
        // Get report parameters from request body
        $data = json_decode($request->getBody(), true);
        $reportType = $data['report_type'] ?? 'security';
        $fromDate = $data['from_date'] ?? '-7 days';
        $toDate = $data['to_date'] ?? 'now';
        
        // Convert to DateTime objects
        $from = new \DateTime($fromDate);
        $to = new \DateTime($toDate);
        
        // Get security events
        $securityEvents = $this->container->get('bornaSecurityEvents');
        $events = $securityEvents->getEvents($from, $to);
        
        // Generate report data
        $reportData = $this->generateReportData($reportType, $events, $from, $to);
        
        // Create export data
        $exportData = [
            'report_type' => $reportType,
            'date_range' => [
                'from' => $from->format('Y-m-d'),
                'to' => $to->format('Y-m-d')
            ],
            'generated_at' => date('Y-m-d H:i:s'),
            'data' => $reportData
        ];
        
        // Generate temporary filename
        $filename = 'borna_security_report_' . $reportType . '_' . date('Y-m-d_H-i-s') . '.json';
        $exportPath = \baseKRIZAN\Config\Config::get('paths.logs') . '/exports/' . $filename;
        
        // Ensure export directory exists
        if (!is_dir(dirname($exportPath))) {
            mkdir(dirname($exportPath), 0755, true);
        }
        
        // Write JSON file
        file_put_contents($exportPath, json_encode($exportData, JSON_PRETTY_PRINT));
        
        // Return file for download
        return new Response(
            file_get_contents($exportPath),
            200,
            [
                'Content-Type' => 'application/json',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"'
            ]
        );
    }
}