<?php

namespace Controllers;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Validation\ValidationResult;
use baseKRIZAN\Config\Config;

/**
 * Base Controller
 * 
 * Provides common functionality for all application controllers
 */
abstract class BaseController
{
    /**
     * Logger instance
     */
    protected Logger $logger;
    
    /**
     * Session manager instance
     */
    protected ?SessionManager $sessionManager = null;
    
    /**
     * Authentication service
     */
    protected ?Authentication $authentication = null;
    
    /**
     * Dependency injection container
     */
    protected ?Container $container = null;

    /**
     * BaseController constructor
     */
    public function __construct(
        Logger $logger, 
        ?SessionManager $sessionManager = null,
        ?Authentication $authentication = null,
        ?Container $container = null
    ) {
        $this->logger = $logger;
        $this->sessionManager = $sessionManager;
        $this->authentication = $authentication;
        $this->container = $container;
    }
    
    /**
     * Creates and configures a Response instance
     * 
     * @return Response Configured Response instance
     */
    protected function response(): Response
    {
        $response = new Response();
        $response->setLogger($this->logger);
        return $response;
    }
    
    /**
     * Get config service
     */
    protected function getConfig(): Config
    {
        return $this->container?->has('config') 
            ? $this->container->get('config') 
            : new Config();
    }
    
    /**
     * Create an edit token with separate limits for validation attempts and saves
     */
    protected function createEditToken(int $id, string $type, int $expiry = 3600): string 
    {
        if (!$this->sessionManager) {
            throw new \RuntimeException('SessionManager is not available');
        }
        
        $token = bin2hex(random_bytes(32));
        $editTokens = $this->sessionManager->get('edit_tokens', []);
        
        $editTokens[$token] = [
            'id' => $id,
            'type' => $type,
            'timestamp' => time(),
            'expiry' => $expiry,
            'validation_attempts' => 0,    // Broj pokušaja validacije (može biti neograničeno)
            'save_attempts' => 0,          // Broj pokušaja spremanja (ograničeno na 1)
            'last_used' => null,
            'consumed' => false            // Je li token "potrošen" uspješnim spremanjem
        ];
        
        $this->sessionManager->set('edit_tokens', $editTokens);
        
        $this->logger->security('Edit token created', [
            'type' => $type,
            'id' => $id,
            'expiry' => $expiry
        ]);
        
        return $token;
    }

    /**
     * Validate edit token for form validation (non-consuming)
     */
    protected function validateEditTokenForValidation(Request $request, string $expectedType): ?array 
    {
        if (!$this->sessionManager) {
            throw new \RuntimeException('SessionManager is not available');
        }
        
        $editToken = $request->getPost('edit_token');
        if (!$editToken) {
            return ['error' => ['No edit token provided']];
        }
        
        $editTokens = $this->sessionManager->get('edit_tokens', []);
        
        if (!isset($editTokens[$editToken])) {
            $this->logger->security('Invalid edit token used for validation', [
                'token' => substr($editToken, 0, 8) . '...',
                'ip' => $request->getIp(),
                'user_id' => $this->authentication->getUser()->id ?? null
            ]);
            return ['error' => ['Invalid edit token']];
        }
        
        $tokenData = $editTokens[$editToken];
        
        // Provjeri je li token već potrošen
        if ($tokenData['consumed']) {
            $this->logger->security('Consumed edit token reuse attempt', [
                'token' => substr($editToken, 0, 8) . '...',
                'ip' => $request->getIp()
            ]);
            return ['error' => ['Edit session expired']];
        }
        
        // Provjeri tip tokena
        if ($tokenData['type'] !== $expectedType) {
            $this->logger->security('Token type mismatch', [
                'expected' => $expectedType,
                'actual' => $tokenData['type'],
                'ip' => $request->getIp()
            ]);
            return ['error' => ['Invalid token type']];
        }
        
        // Provjeri je li token istekao
        $expiry = $tokenData['expiry'] ?? 3600;
        if (time() - $tokenData['timestamp'] > $expiry) {
            unset($editTokens[$editToken]);
            $this->sessionManager->set('edit_tokens', $editTokens);
            
            $this->logger->security('Expired edit token used', [
                'token' => substr($editToken, 0, 8) . '...',
                'age' => time() - $tokenData['timestamp'],
                'ip' => $request->getIp()
            ]);
            return ['error' => ['Edit session expired']];
        }
        
        // Ažuriraj broj validacijskih pokušaja (ali NE troši token)
        $editTokens[$editToken]['validation_attempts']++;
        $editTokens[$editToken]['last_used'] = time();
        $this->sessionManager->set('edit_tokens', $editTokens);
        
        $this->logger->debug('Edit token validated for form validation', [
            'type' => $expectedType,
            'id' => $tokenData['id'],
            'validation_attempts' => $editTokens[$editToken]['validation_attempts']
        ]);
        
        return ['id' => $tokenData['id'], 'token' => $editToken];
    }

    /**
     * Consume edit token for successful save (one-time use)
     */
    protected function consumeEditToken(string $token): bool 
    {
        if (!$this->sessionManager) {
            return false;
        }
        
        $editTokens = $this->sessionManager->get('edit_tokens', []);
        
        if (!isset($editTokens[$token])) {
            return false;
        }
        
        // Označi token kao potrošen
        $editTokens[$token]['consumed'] = true;
        $editTokens[$token]['save_attempts']++;
        $this->sessionManager->set('edit_tokens', $editTokens);
        
        $this->logger->security('Edit token consumed for successful save', [
            'token' => substr($token, 0, 8) . '...',
            'save_attempts' => $editTokens[$token]['save_attempts']
        ]);
        
        return true;
    }

    /**
     * Refresh edit token - create new token with same ID
     */
    protected function refreshEditToken(string $oldToken, string $type): ?string 
    {
        if (!$this->sessionManager) {
            return null;
        }
        
        $editTokens = $this->sessionManager->get('edit_tokens', []);
        
        if (!isset($editTokens[$oldToken])) {
            return null;
        }
        
        $oldTokenData = $editTokens[$oldToken];
        
        // Ukloni stari token
        unset($editTokens[$oldToken]);
        $this->sessionManager->set('edit_tokens', $editTokens);
        
        // Stvori novi token s istim ID-om
        $newToken = $this->createEditToken(
            $oldTokenData['id'], 
            $type, 
            $oldTokenData['expiry']
        );
        
        $this->logger->security('Edit token refreshed', [
            'old_token' => substr($oldToken, 0, 8) . '...',
            'new_token' => substr($newToken, 0, 8) . '...',
            'id' => $oldTokenData['id'],
            'old_validation_attempts' => $oldTokenData['validation_attempts']
        ]);
        
        return $newToken;
    }
    
    /**
     * Validate request and render form on failure
     */
    protected function validateForm(
        Request $request, 
        array $rules, 
        string $template, 
        string $title, 
        array $messages = []
    ): ValidationResult|Response {
        $validation = $this->validate($request, $rules, $messages);
        
        if ($validation->fails()) {
            $formData = $request->all();
            
            return $this->response()->renderForm(
                $formData, 
                $validation->errors(), 
                $template, 
                $title, 
                400, 
                $this->getCommonViewData()
            );
        }
        
        return $validation;
    }
    
    /**
     * Validate request data
     */
    protected function validate(
        Request $request, 
        array $rules, 
        array $messages = []
    ): ValidationResult {
        return $request->validate($rules, $messages);
    }
    
    /**
     * Get common data for all views
     */
    protected function getCommonViewData(): array
    {
        // Get authentication information
        $loggedIn = false;
        $user = null;
        
        if ($this->authentication !== null) {
            $loggedIn = $this->authentication->isLoggedIn();
            $user = $this->authentication->getUser();
        }
        
        // Return common view data
        return [
            'loggedIn' => $loggedIn,
            'user' => $user,
            'currentYear' => date('Y'),
            'appName' => $this->getConfig()->get('appname'),
            'environment' => $this->getConfig()->get('environment')
        ];
    }
    
    /**
     * Get entity ID from request, checking both route parameters and POST data
     */
    protected function getEntityId(Request $request, string $paramName = 'id'): ?int
    {
        // Check route parameters first
        $id = $request->getParam($paramName);
        
        // Then check POST data
        if ($id === null) {
            $id = $request->getPost($paramName);
        }
        
        // Convert to integer if found
        return $id !== null ? (int)$id : null;
    }
}