<?php

namespace Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Validation\Validator;
use Models\DatabaseTable;
use baseKRIZAN\Database\DatabaseConnection;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class DB extends BaseController
{
    protected DatabaseTable $korisniciTable;
    protected DatabaseConnection $databaseConnection;
    protected Validator $validator;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        DatabaseConnection $databaseConnection,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->databaseConnection = $databaseConnection;
        $this->validator = $validator;
    }

    // IMPORT/EXPORT JEDNE TABLICE
    public function importExport(Request $request): Response 
    {
        if ($request->getMethod() === 'POST') {
            $this->logger->database('ImportExport POST request', [
                'post' => $request->getPost(),
                'files' => $request->getFile('file'),
                'method' => $request->getMethod()
            ]);

            if ($request->getPost('import')) {
                $this->logger->database('Handling import');
                return $this->handleImport($request);
            }
            
            if ($request->getPost('export')) {
                $this->logger->database('Handling CSV export');
                return $this->handleExportCSV($request);
            }
            
            if ($request->getPost('exportsql')) {
                $this->logger->database('Handling SQL export');
                return $this->handleExportSQL($request);
            }

            $this->logger->database('No valid action found in POST request');
        }

        return $this->response()->render(
            'DB/baze_importexport.html.php',
            ['conn' => $this->databaseConnection->getPDO()],
            'Database import/export'
        );
    }

    // IMPORT/EXPORT CIJELE BAZE
    public function importExportFull(Request $request): Response 
    {
        if ($request->getMethod() === 'POST') {
            if ($request->getPost('import')) {
                return $this->handleFullDatabaseImport($request);
            }
            
            if ($request->getPost('exportsql')) {
                return $this->handleFullDatabaseExport();
            }
        }

        return $this->response()->render(
            'DB/baza_importexport.html.php',
            ['conn' => $this->databaseConnection->getPDO()],
            'Full Database import/export'
        );
    }

    // UPRAVLJANJE TABLICAMA
    public function listAndEdit(Request $request): Response 
    {
        $pdo = $this->databaseConnection->getPDO();
        $data = ['conn' => $pdo];

        if ($request->getMethod() === 'POST') {
            $selectedTable = $request->getPost('selected_table');
            
            if ($selectedTable) {
                if ($request->getPost('delete_table')) {
                    return $this->handleDeleteTable($selectedTable);
                }
                
                if ($request->getPost('new_table_name')) {
                    return $this->handleRenameTable($selectedTable, $request->getPost('new_table_name'));
                }
                
                if ($request->getPost('new_table_type')) {
                    return $this->handleChangeTableType($selectedTable, $request->getPost('new_table_type'));
                }
                
                if ($request->getPost('new_table_collation')) {
                    return $this->handleChangeCollation($selectedTable, $request->getPost('new_table_collation'));
                }
                
                if ($request->getPost('column_name') && $request->getPost('new_column_name')) {
                    return $this->handleRenameColumn(
                        $selectedTable, 
                        $request->getPost('column_name'), 
                        $request->getPost('new_column_name')
                    );
                }
            }
        }

        return $this->response()->render(
            'DB/baze_listandedit.html.php',
            $data,
            'Database Management'
        );
    }

    // HELPER METODE ZA IMPORT/EXPORT JEDNE TABLICE
    private function handleImport(Request $request): Response 
    {
        $file = $request->getFile('file');
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            return Response::redirect('databases/importexport');
        }

        $sqlContent = file_get_contents($file['tmp_name']);
        if (strpos($sqlContent, 'INSERT INTO') === false) {
            return Response::redirect('databases/importexport');
        }

        try {
            $this->databaseConnection->getPDO()->exec($sqlContent);
            return Response::redirect('databases/importexport');
        } catch (\PDOException $e) {
            $this->logger->error('SQL import error', ['error' => $e->getMessage()]);
            return Response::redirect('databases/importexport');
        }
    }

    private function handleExportCSV(Request $request): Response 
    {
        $table = $request->getPost('table');
        if (!$table) {
            return Response::redirect('databases/import-export');
        }

        $pdo = $this->databaseConnection->getPDO();
        $query = $pdo->query("SELECT * FROM $table");
        $rows = $query->fetchAll(\PDO::FETCH_ASSOC);

        $response = new Response();
        $response->setHeader('Content-Type', 'text/csv; charset=utf-8');
        $response->setHeader('Content-Disposition', 'attachment; filename=' . $table . '.csv');

        ob_start();
        $fp = fopen('php://output', 'w');
        
        $header = array_map(function ($column) {
            return iconv('UTF-8', 'Windows-1250//TRANSLIT', $column);
        }, array_keys($rows[0]));
        fputcsv($fp, $header, ';');

        foreach ($rows as $row) {
            array_walk($row, function (&$value) {
                if (!is_null($value)) {
                    $value = iconv('UTF-8', 'Windows-1250//TRANSLIT', $value);
                }
            });
            fputcsv($fp, $row, ';');
        }
        
        fclose($fp);
        $content = ob_get_clean();
        
        return $response->setContent($content);
    }

    private function handleExportSQL(Request $request): Response 
    {
        $table = $request->getPost('table');
        if (!$table) {
            return Response::redirect('databases/import-export');
        }

        $pdo = $this->databaseConnection->getPDO();
        $query = $pdo->query("SHOW CREATE TABLE $table");
        $structure = $query->fetch(\PDO::FETCH_NUM);
        
        $query = $pdo->query("SELECT * FROM $table");
        $rows = $query->fetchAll(\PDO::FETCH_ASSOC);

        $response = new Response();
        $response->setHeader('Content-Type', 'application/sql');
        $response->setHeader('Content-Disposition', 'attachment; filename=' . $table . '.sql');

        $output = $this->generateSQLDump($table, $structure[1], $rows);
        return $response->setContent($output);
    }

    // HELPER METODE ZA IMPORT/EXPORT CIJELE BAZE
    private function handleFullDatabaseImport(Request $request): Response 
    {
        $file = $request->getFile('file');
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            return Response::redirect('database/importexportfull');
        }

        $sqlContent = file_get_contents($file['tmp_name']);
        try {
            $this->databaseConnection->getPDO()->exec($sqlContent);
            return Response::redirect('database/importexportfull');
        } catch (\PDOException $e) {
            $this->logger->error('Full database import error', ['error' => $e->getMessage()]);
            return Response::redirect('database/importexportfull');
        }
    }

    private function handleFullDatabaseExport(): Response 
    {
        $pdo = $this->databaseConnection->getPDO();
        $database = $pdo->query("SELECT DATABASE()")->fetchColumn();
        $filename = $database . "_backup_" . date("Y-m-d_H-i-s") . ".sql";

        $response = new Response();
        $response->setHeader('Content-Type', 'application/sql');
        $response->setHeader('Content-Disposition', 'attachment; filename=' . $filename);

        $output = "-- SQL Dump for database `$database`\n";
        $output .= "-- Generated on " . date('Y-m-d H:i:s') . "\n";
        $output .= "-- PHP Version: " . phpversion() . "\n\n";

        $tables = $pdo->query("SHOW TABLES")->fetchAll(\PDO::FETCH_COLUMN);
        foreach ($tables as $table) {
            $createTable = $pdo->query("SHOW CREATE TABLE `$table`")->fetch(\PDO::FETCH_NUM);
            $rows = $pdo->query("SELECT * FROM `$table`")->fetchAll(\PDO::FETCH_ASSOC);
            
            $output .= "DROP TABLE IF EXISTS `$table`;\n";
            $output .= $createTable[1] . ";\n\n";
            $output .= $this->generateInsertStatements($table, $rows);
        }

        return $response->setContent($output);
    }

    // HELPER METODE ZA UPRAVLJANJE TABLICAMA
    private function handleDeleteTable(string $table): Response 
    {
        try {
            $sql = "DROP TABLE `$table`";
            $this->databaseConnection->getPDO()->exec($sql);
            return Response::redirect('database/listandedit');
        } catch (\PDOException $e) {
            $this->logger->error('Table delete error', ['error' => $e->getMessage()]);
            return Response::redirect('database/listandedit');
        }
    }

    private function handleRenameTable(string $oldName, string $newName): Response 
    {
        try {
            $sql = "RENAME TABLE `$oldName` TO `$newName`";
            $this->databaseConnection->getPDO()->exec($sql);
            return Response::redirect('database/listandedit');
        } catch (\PDOException $e) {
            $this->logger->error('Table rename error', ['error' => $e->getMessage()]);
            return Response::redirect('database/listandedit');
        }
    }

    private function handleChangeTableType(string $table, string $newType): Response 
    {
        try {
            $sql = "ALTER TABLE `$table` ENGINE=$newType";
            $this->databaseConnection->getPDO()->exec($sql);
            return Response::redirect('database/listandedit');
        } catch (\PDOException $e) {
            $this->logger->error('Table type change error', ['error' => $e->getMessage()]);
            return Response::redirect('database/listandedit');
        }
    }

    private function handleChangeCollation(string $table, string $newCollation): Response 
    {
        try {
            $sql = "ALTER TABLE `$table` COLLATE $newCollation";
            $this->databaseConnection->getPDO()->exec($sql);
            return Response::redirect('database/listandedit');
        } catch (\PDOException $e) {
            $this->logger->error('Table collation change error', ['error' => $e->getMessage()]);
            return Response::redirect('database/listandedit');
        }
    }

    private function handleRenameColumn(string $table, string $oldName, string $newName): Response 
    {
        try {
            $sql = "ALTER TABLE `$table` CHANGE `$oldName` `$newName` VARCHAR(255)";
            $this->databaseConnection->getPDO()->exec($sql);
            return Response::redirect('database/listandedit');
        } catch (\PDOException $e) {
            $this->logger->error('Column rename error', ['error' => $e->getMessage()]);
            return Response::redirect('database/listandedit');
        }
    }

    // UTILITY METODE
    private function generateSQLDump(string $table, string $structure, array $rows): string 
    {
        $output = "--\n-- SQL Dump\n--\n";
        $output .= "DROP TABLE IF EXISTS `$table`;\n";
        $output .= $structure . ";\n\n";
        $output .= $this->generateInsertStatements($table, $rows);
        return $output;
    }

    private function generateInsertStatements(string $table, array $rows): string 
    {
        if (empty($rows)) {
            return '';
        }

        $output = '';
        $chunkSize = 500;
        $rowCount = count($rows);
        
        for ($i = 0; $i < $rowCount; $i += $chunkSize) {
            $chunk = array_slice($rows, $i, $chunkSize);
            $output .= "INSERT INTO `$table` (" . implode(",", array_keys($chunk[0])) . ") VALUES ";
            
            foreach ($chunk as $row) {
                $values = array_map(function($value) {
                    return $value === null ? 'NULL' : $this->databaseConnection->getPDO()->quote($value);
                }, $row);
                
                $output .= "(" . implode(",", $values) . "),";
            }
            
            $output = rtrim($output, ",") . ";\n\n";
        }

        return $output;
    }
}