<?php

namespace Controllers;

use baseKRIZAN\Services\FCMService;
use baseKRIZAN\Services\FirebaseAuthService;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;

/**
 * FCMController - Controller for FCM operations
 * 
 * Handles HTTP requests related to Firebase Cloud Messaging
 */
class FCMController extends BaseController
{
    protected ?FCMService $fcmService;
    protected ?FirebaseAuthService $firebaseAuthService;
    protected bool $fcmEnabled;

    /**
     * Constructor
     */
    public function __construct(
        ?FCMService $fcmService,
        ?FirebaseAuthService $firebaseAuthService,
        Authentication $authentication,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->fcmService = $fcmService;
        $this->firebaseAuthService = $firebaseAuthService;
        $this->fcmEnabled = (bool)\baseKRIZAN\Config\Config::get('NOTIFICATIONS_FCM_ENABLED', false);
        
        $this->logger->notification('FCMController initialized', [
            'fcm_enabled' => $this->fcmEnabled,
            'fcm_service_available' => $this->fcmService !== null,
            'firebase_auth_service_available' => $this->firebaseAuthService !== null
        ]);
    }

    /**
     * Save push notification token
     */
    public function saveToken(Request $request): Response
    {
        // Check if FCM is enabled
        if (!$this->fcmEnabled || !$this->fcmService) {
            return Response::json([
                'success' => false, 
                'message' => 'FCM functionality is disabled'
            ], 403);
        }
        
        try {
            $data = $request->getPost();
            $this->logger->notification('Token save request received', [
                'data_keys' => array_keys($data),
                'device_type_present' => isset($data['deviceType']),
                'device_type' => $data['deviceType'] ?? 'not_set'
            ]);

            if (!isset($data['token']) || !isset($data['userId']) || !isset($data['email'])) {
                return Response::json([
                    'success' => false, 
                    'message' => 'Missing required parameters'
                ], 400);
            }

            $deviceType = $data['deviceType'] ?? 'unknown';
            $this->logger->notification('Saving user FCM token', [
                'user_id' => $data['userId'],
                'device_type' => $deviceType
            ]);

            $result = $this->fcmService->saveUserToken(
                (int)$data['userId'], 
                $data['email'], 
                $data['token'],
                $deviceType
            );

            return Response::json([
                'success' => $result,
                'message' => $result ? 'Token successfully saved' : 'Failed to save token'
            ]);

        } catch (\Exception $e) {
            $this->logger->error('FCM token save error', [
                'error' => $e->getMessage()
            ]);
            return Response::json([
                'success' => false,
                'message' => 'Internal server error'
            ], 500);
        }
    }
    
    /**
     * Get Firebase auth token
     */
    public function getToken(Request $request): Response
    {
        try {
            // Check if user is authenticated
            if (!$this->authentication->isLoggedIn()) {
                $this->logger->notification('Unauthorized access to Firebase token', [
                    'ip' => $request->getIP()
                ]);
                return Response::json(['error' => 'Unauthorized'], 401);
            }

            $user = $this->authentication->getUser();
            
            // Check if user has admin permissions
            $isAdmin = method_exists($user, 'hasMinimumPermission') && 
                      $user->hasMinimumPermission(Korisnik::KORISNIK_ADMIN);
            
            // Generate Firebase custom token
            $token = $this->firebaseAuthService->generateAuthToken(
                (int)$user->id,
                $user->user_email,
                $isAdmin
            );
            
            if (!$token) {
                throw new \Exception('Failed to generate Firebase token');
            }
            
            return Response::json([
                'success' => true,
                'token' => $token
            ]);
            
        } catch (\Exception $e) {
            $this->logger->error('Error generating Firebase token', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Error generating Firebase token'
            ], 500);
        }
    }
    
    /**
     * Get Firebase configuration
     */
    public function getFirebaseConfig(Request $request): Response
    {
        try {
            // Check if user is authenticated
            if (!$this->authentication->isLoggedIn()) {
                $this->logger->notification('Unauthorized access to Firebase config', [
                    'ip' => $request->getIP()
                ]);
                return Response::json(['error' => 'Unauthorized'], 401);
            }

            $user = $this->authentication->getUser();
        
            $this->logger->notification('Firebase configuration requested', [
                'user_id' => $user->id,
                'user_email' => $user->user_email,
                'fcm_enabled' => $this->fcmEnabled
            ]);

            // Return only basic data needed for Firebase initialization
            return Response::json([
                'success' => true,
                'config' => [
                    'apiKey' => \baseKRIZAN\Config\Config::get('FIREBASE_API_KEY'),
                    'projectId' => \baseKRIZAN\Config\Config::get('FIREBASE_PROJECT_ID'),
                    'messagingSenderId' => \baseKRIZAN\Config\Config::get('FIREBASE_MESSAGING_SENDER_ID'),
                    'appId' => \baseKRIZAN\Config\Config::get('FIREBASE_APP_ID'),
                    'vapidKey' => \baseKRIZAN\Config\Config::get('FIREBASE_SERVER_KEY'),
                    'databaseURL' => \baseKRIZAN\Config\Config::get('FIREBASE_DATABASE_URL')
                ],
                'features' => [
                    'fcm_enabled' => $this->fcmEnabled,
                    'realtime_db_enabled' => (bool)\baseKRIZAN\Config\Config::get('NOTIFICATIONS_REALTIME_DB_ENABLED', false)
                ]
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error providing Firebase configuration', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Internal server error'
            ], 500);
        }
    }
    
    /**
     * Send a test notification
     */
    public function sendTestNotification(Request $request): Response
    {
        try {
            // Check if user is authenticated
            if (!$this->authentication->isLoggedIn()) {
                return Response::json(['error' => 'Unauthorized'], 401);
            }

            $user = $this->authentication->getUser();
            
            // Check if user has admin permissions
            $isAdmin = method_exists($user, 'hasMinimumPermission') && 
                      $user->hasMinimumPermission(Korisnik::KORISNIK_ADMIN);
                      
            if (!$isAdmin) {
                return Response::json(['error' => 'Permission denied'], 403);
            }
            
            $data = json_decode(file_get_contents("php://input"), true);
            
            $userId = (int)($data['user_id'] ?? $user->id);
            $title = $data['title'] ?? 'Test Notification';
            $body = $data['body'] ?? 'This is a test push notification from Firebase Cloud Messaging.';
            $notificationType = $data['type'] ?? 'test';
            
            // Additional data for notification
            $notificationData = [
                'type' => $notificationType,
                'route' => $data['route'] ?? 'notification/home',
                'timestamp' => time() * 1000
            ];
            
            // Send notification
            $result = $this->fcmService->sendUserNotification(
                $userId,
                $title,
                $body,
                $notificationData
            );
            
            return Response::json([
                'success' => $result,
                'message' => $result ? 'Test notification sent successfully' : 'Failed to send test notification'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error sending test notification', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Internal server error'
            ], 500);
        }
    }
}