<?php

namespace Controllers;

use baseKRIZAN\Services\FirebaseRealtimeService;
use baseKRIZAN\Services\NotificationService;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;

/**
 * FirebaseRealtimeController - Controller for Firebase Realtime Database operations
 * 
 * Handles HTTP requests related to Firebase Realtime Database
 */
class FirebaseRealtimeController extends BaseController
{
    protected ?FirebaseRealtimeService $firebaseRealtimeService;
    protected NotificationService $notificationService;
    protected bool $realtimeDbEnabled;

    /**
     * Constructor
     */
    public function __construct(
        ?FirebaseRealtimeService $firebaseRealtimeService,
        NotificationService $notificationService,
        Authentication $authentication,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->firebaseRealtimeService = $firebaseRealtimeService;
        $this->notificationService = $notificationService;
        $this->realtimeDbEnabled = (bool)\baseKRIZAN\Config\Config::get('NOTIFICATIONS_REALTIME_DB_ENABLED', false);
        
        $this->logger->notification('FirebaseRealtimeController initialized', [
            'realtime_db_enabled' => $this->realtimeDbEnabled,
            'realtime_db_service_available' => $this->firebaseRealtimeService !== null
        ]);
    }

    /**
     * API endpoint for full Firebase synchronization
     */
    public function syncFirebase(Request $request): Response
    {
        if (!$this->realtimeDbEnabled || !$this->firebaseRealtimeService) {
            return Response::json([
                'success' => false, 
                'message' => 'Firebase Realtime Database functionality is disabled'
            ], 403);
        }
        
        try {
            // Ensure session is started and consistent if sessionManager is available
            if ($this->sessionManager !== null) {
                $this->sessionManager->ensureSessionIntegrity();
            }

            if (!$this->authentication->isLoggedIn()) {
                $this->logger->notification('Unauthorized access to Firebase sync', [
                    'ip' => $request->getIP()
                ]);
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            
            // Check if user has admin permissions (optional)
            $isAdmin = method_exists($user, 'hasMinimumPermission') && 
                    $user->hasMinimumPermission(Korisnik::KORISNIK_ADMIN);
                    
            if (!$isAdmin) {
                $this->logger->notification('Non-admin user tried to access Firebase sync', [
                    'user_id' => $user->id,
                    'user_email' => $user->user_email,
                    'ip' => $request->getIP()
                ]);
                return Response::json(['error' => 'Permission denied'], 403);
            }
            
            // Get user ID from request or use current user ID
            $userId = (int)($request->getQuery('user_id') ?? $user->id);
            
            $this->logger->notification('Manual Firebase sync requested', [
                'requested_by' => $user->id,
                'for_user_id' => $userId
            ]);
            
            // Get all notifications for this user
            $notifications = $this->notificationService->getUserNotifications($userId, 1000, 'all');
            
            // Perform full sync with cleanup
            $result = $this->firebaseRealtimeService->syncNotifications($userId, $notifications, true);
            
            return Response::json([
                'success' => $result,
                'message' => $result ? 'Firebase sync completed successfully' : 'Firebase sync failed'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error during manual Firebase sync', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'An error occurred during Firebase synchronization'
            ], 500);
        }
    }
    
    /**
     * Update notification read status in Firebase
     */
    public function updateReadStatus(Request $request): Response
    {
        if (!$this->realtimeDbEnabled || !$this->firebaseRealtimeService) {
            return Response::json([
                'success' => false, 
                'message' => 'Firebase Realtime Database functionality is disabled'
            ], 403);
        }
        
        try {
            if (!$this->authentication->isLoggedIn()) {
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            $userId = (int)$user->id;
            
            // Get single notification ID if provided
            $data = json_decode(file_get_contents("php://input"), true);
            $notificationId = isset($data['notification_id']) ? (int)$data['notification_id'] : null;
            
            // Sync read status to Firebase
            $result = $this->firebaseRealtimeService->syncReadStatus($userId, $notificationId);
            
            return Response::json([
                'success' => $result,
                'message' => $result ? 'Read status updated in Firebase' : 'Failed to update read status'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error updating read status in Firebase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'An error occurred during Firebase read status update'
            ], 500);
        }
    }
    
    /**
     * Update notification sticky status in Firebase
     */
    public function updateStickyStatus(Request $request): Response
    {
        if (!$this->realtimeDbEnabled || !$this->firebaseRealtimeService) {
            return Response::json([
                'success' => false, 
                'message' => 'Firebase Realtime Database functionality is disabled'
            ], 403);
        }

        try {
            if (!$this->authentication->isLoggedIn()) {
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            $userId = (int)$user->id;
            
            $data = json_decode(file_get_contents("php://input"), true);
            
            if (!isset($data['notification_id']) || !isset($data['is_sticky'])) {
                return Response::json([
                    'success' => false,
                    'error' => 'Missing required parameters'
                ], 400);
            }
            
            $notificationId = (int)$data['notification_id'];
            $isSticky = (bool)$data['is_sticky'];
            
            // Update notification in Firebase
            $result = $this->firebaseRealtimeService->updateNotification(
                $userId, 
                (string)$notificationId, 
                [
                    'is_sticky' => $isSticky,
                    'updated_at' => (int)round(microtime(true) * 1000)
                ]
            );
            
            return Response::json([
                'success' => $result,
                'message' => $result ? 'Sticky status updated in Firebase' : 'Failed to update sticky status'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error updating sticky status in Firebase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'An error occurred during Firebase sticky status update'
            ], 500);
        }
    }
}