<?php

namespace Controllers;

use baseKRIZAN\Services\NotificationService;
use baseKRIZAN\Services\NotificationTypeRegistry;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;

/**
 * InitialDataController
 * 
 * Handles batched API requests for initial data loading
 */
class InitialDataController extends BaseController
{
    protected NotificationService $notificationService;
    protected FCMController $fcmController;

    /**
     * Constructor
     */
    public function __construct(
        NotificationService $notificationService,
        FCMController $fcmController,
        Authentication $authentication,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->notificationService = $notificationService;
        $this->fcmController = $fcmController;
    }

    /**
     * Get all initial data in a single request
     */
    public function getInitialData(Request $request): Response
    {
        try {
            // Ensure session integrity
            if ($this->sessionManager !== null) {
                $this->sessionManager->ensureSessionIntegrity();
            }

            if (!$this->authentication->isLoggedIn()) {
                $this->logger->notification('Unauthenticated access to initial data');
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            
            if (!$user) {
                $this->logger->error('User object not available');
                return Response::json([
                    'success' => false,
                    'error' => 'User information not available'
                ], 400);
            }
            
            $userId = (int)$user->id;
            
            // Get all required data in one request
            $initialData = [
                'notification_types' => NotificationTypeRegistry::getNotificationTypesData(),
                'firebase_config' => $this->getFirebaseConfig(),
                'unread_notifications' => $this->notificationService->getUserNotifications($userId, 10, 'unread'),
                'user_data' => [
                    'id' => $userId,
                    'email' => $user->user_email,
                    'name' => $user->user_firstandlastname ?? '',
                    'is_admin' => method_exists($user, 'hasPermission') ? $user->hasPermission(Korisnik::KORISNIK_ADMIN) : false
                ]
            ];
            
            $this->logger->notification('Initial data fetched for user', [
                'user_id' => $userId
            ]);

            return Response::json([
                'success' => true,
                'data' => $initialData,
                'timestamp' => time()
            ]);

        } catch (\Exception $e) {
            $this->logger->error('Error fetching initial data', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return Response::json([
                'success' => false,
                'error' => 'Failed to fetch initial data'
            ], 500);
        }
    }
    
    /**
     * Get Firebase configuration data
     */
    private function getFirebaseConfig(): array
    {
        // Get configuration data from FCMController's method
        $fcmConfigResponse = $this->fcmController->getFirebaseConfig(new Request());
        
        // Parse the JSON response
        $fcmConfigData = json_decode($fcmConfigResponse->getBody(), true);
        
        // Return only the configuration part
        return $fcmConfigData['success'] ? [
            'config' => $fcmConfigData['config'] ?? [],
            'features' => $fcmConfigData['features'] ?? []
        ] : [];
    }
}