<?php

namespace Controllers;

use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\LUKA\LukaManager;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Template\TemplateRenderer;

/**
 * Controller for the LUKA Digital Twin admin dashboard
 */
class LUKA extends BaseController
{
    protected ?LukaManager $lukaManager = null;
    protected ?Container $container;

    /**
     * Constructor
     */
    public function __construct(Container $container)
    {
        $this->container = $container;
        $logger = $container->get('logger');
        $sessionManager = $container->get('sessionManager');
        $authentication = $container->has('authentication') ? $container->get('authentication') : null;
        
        parent::__construct($logger, $sessionManager, $authentication, $container);
        
        // Safely get the lukaManager if it exists
        if ($container->has('lukaManager')) {
            $this->lukaManager = $container->get('lukaManager');
        } else {
            $this->lukaManager = null;
        }
    }

    /**
     * Dashboard home page
     */
    public function index(Request $request): Response
    {
        // Check if LUKA exists and is enabled
        if ($this->lukaManager === null || !$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled. Enable it in your configuration.', 'Module Disabled', 503);
        }
        
        // Get application state for summary display
        $appState = $this->lukaManager->getApplicationState();
        
        // Render the dashboard using BaseController's render method
        return $this->response()->render('LUKA/index.html.php', [
            'appState' => $appState,
            'active_page' => 'dashboard'
        ], 'LUKA Digital Twin');
    }

    /**
     * Overview page - shows summary of system metrics
     */
    public function overview(Request $request): Response
    {
        if (!$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled', 'Module Disabled', 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Get metrics storage from container
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Get metrics summary
        $metricsSummary = $metricsStorage->getMetricsSummary();
        
        // Recent data for charts (last 24 hours)
        $from = new \DateTime('-24 hours');
        $to = new \DateTime();
        
        $performanceData = $metricsStorage->getMetrics('performance', $from, $to);
        $memoryData = $metricsStorage->getMetrics('memory', $from, $to);
        
        return $this->response()->render('LUKA/overview.html.php', [
            'appState' => $appState,
            'metricsSummary' => $metricsSummary,
            'performanceData' => $performanceData,
            'memoryData' => $memoryData,
            'active_page' => 'overview'
        ], 'LUKA - Overview');
    }

    /**
     * Performance page - detailed performance metrics
     */
    public function performance(Request $request): Response
    {
        if (!$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled', 'Module Disabled', 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Get metrics data from last 7 days by default
        $from = new \DateTime('-7 days');
        $to = new \DateTime();
        
        // Get metrics storage
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Get detailed performance data - using getMetrics instead of getDetailedMetrics
        $performanceData = $metricsStorage->getMetrics('performance', $from, $to);
        
        return $this->response()->render('LUKA/performance.html.php', [
            'appState' => $appState,
            'performanceData' => $performanceData,
            'active_page' => 'performance'
        ], 'LUKA - Performance');
    }

    /**
     * Requests page - request analysis
     */
    public function requests(Request $request): Response
    {
        if (!$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled', 'Module Disabled', 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Get metrics storage
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Get request data from last 24 hours by default
        $from = new \DateTime('-24 hours');
        $to = new \DateTime();
        
        $requestData = $metricsStorage->getMetrics('requests', $from, $to);
        
        return $this->response()->render('LUKA/requests.html.php', [
            'appState' => $appState,
            'requestData' => $requestData,
            'active_page' => 'requests'
        ], 'LUKA - Request Analysis');
    }

    /**
     * Database page - database performance
     */
    public function database(Request $request): Response
    {
        if (!$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled', 'Module Disabled', 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Get metrics storage
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Get database metrics
        $databaseData = $metricsStorage->getMetrics('database', new \DateTime('-24 hours'), new \DateTime());
        
        return $this->response()->render('LUKA/database.html.php', [
            'appState' => $appState,
            'databaseData' => $databaseData,
            'active_page' => 'database'
        ], 'LUKA - Database Performance');
    }

    /**
     * Errors page - error analysis
     */
    public function errors(Request $request): Response
    {
        if (!$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled', 'Module Disabled', 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Get metrics storage
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Get error data
        $errorData = $metricsStorage->getMetrics('errors', new \DateTime('-24 hours'), new \DateTime());
        
        return $this->response()->render('LUKA/errors.html.php', [
            'appState' => $appState,
            'errorData' => $errorData,
            'active_page' => 'errors'
        ], 'LUKA - Error Analysis');
    }

    /**
     * System page - system information
     */
    public function system(Request $request): Response
    {
        if (!$this->lukaManager->isEnabled()) {
            return $this->response()->renderError('LUKA Digital Twin is disabled', 'Module Disabled', 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Get metrics storage
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Get config
        $config = $this->lukaManager->getConfig();
        
        // Check if WebSocket is enabled
        $websocketEnabled = $config['websocket_enabled'] ?? false;
        
        return $this->response()->render('LUKA/system.html.php', [
            'appState' => $appState,
            'config' => $config,
            'websocketEnabled' => $websocketEnabled,
            'active_page' => 'system'
        ], 'LUKA - System Information');
    }

    /**
     * API endpoint za dohvat metrika
     */
    public function getMetrics(Request $request): Response
    {
        // Provjeri je li LUKA uključena
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        $metrics = $request->getQuery();
        $metricType = $metrics['type'] ?? 'performance';
        
        // Parsiraj vremenski raspon
        $fromStr = $metrics['from'] ?? '-24 hours';
        $toStr = $metrics['to'] ?? 'now';
        
        $from = new \DateTime($fromStr);
        $to = new \DateTime($toStr);
        
        // Dohvati metriku
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Dohvat metrike iz storage-a
        $metricsData = $metricsStorage->getMetrics($metricType, $from, $to);
        
        // Provjeri podatke za pojedine tipove metrika i formatiraj ih ako je potrebno
        switch ($metricType) {
            case 'performance':
                // Standardiziraj format response-a za frontend
                $metricsData = $this->standardizePerformanceData($metricsData);
                break;
                
            case 'memory':
                // Standardiziraj format response-a za frontend
                $metricsData = $this->standardizeMemoryData($metricsData);
                break;
        }
        
        // Vrati JSON response
        return Response::json([
            'success' => true,
            'metric_type' => $metricType,
            'from' => $from->format('Y-m-d H:i:s'),
            'to' => $to->format('Y-m-d H:i:s'),
            'data' => $metricsData
        ]);
    }

    /**
     * Standardiziraj format podataka o performansama
     * 
     * @param array $data Podaci o performansama
     * @return array Standardizirani podaci
     */
    private function standardizePerformanceData(array $data): array
    {
        $result = [];
        
        foreach ($data as $item) {
            // Osiguraj da svaki item ima hour/timestamp polje
            if (!isset($item['hour']) && !isset($item['timestamp'])) {
                // Ako nema ni hour ni timestamp, dodaj current time
                $item['timestamp'] = date('Y-m-d H:i:s');
            }
            
            // Osiguraj da postoji avg_response_time polje
            if (!isset($item['avg_response_time']) && isset($item['max_time'])) {
                $item['avg_response_time'] = $item['max_time'];
            } elseif (!isset($item['avg_response_time'])) {
                $item['avg_response_time'] = 0;
            }
            
            $result[] = $item;
        }
        
        return $result;
    }

    /**
     * Standardiziraj format podataka o memoriji
     * 
     * @param array $data Podaci o memoriji
     * @return array Standardizirani podaci
     */
    private function standardizeMemoryData(array $data): array
    {
        $result = [];
        
        foreach ($data as $item) {
            // Osiguraj da svaki item ima hour/timestamp polje
            if (!isset($item['hour']) && !isset($item['timestamp'])) {
                // Ako nema ni hour ni timestamp, dodaj current time
                $item['timestamp'] = date('Y-m-d H:i:s');
            }
            
            // Osiguraj da postoji avg_memory polje
            if (!isset($item['avg_memory']) && isset($item['peak_memory'])) {
                $item['avg_memory'] = $item['peak_memory'];
            } elseif (!isset($item['avg_memory'])) {
                $item['avg_memory'] = 0;
            }
            
            $result[] = $item;
        }
        
        return $result;
    }
    
    /**
     * API endpoint za dohvat metrika po komponentama
     */
    public function getComponentMetrics(Request $request): Response
    {
        // Provjeri je li LUKA uključena
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        try {
            // Dohvati metriku
            $metricsStorage = $this->container->get('lukaMetricsStorage');
            
            // Dohvati parametre
            $fromStr = $request->getQuery()['from'] ?? '-24 hours';
            $toStr = $request->getQuery()['to'] ?? 'now';
            
            $from = new \DateTime($fromStr);
            $to = new \DateTime($toStr);
            
            // Dohvati podatke iz requests tablice
            $requests = $metricsStorage->getMetrics('requests', $from, $to);
            
            // Struktura za pohranu podataka o komponentama
            $components = [];
            $totalTime = 0;
            
            // Prođi kroz sve zahtjeve i prikupi podatke o komponentama
            foreach ($requests as $request) {
                if (isset($request['components_data']) && !empty($request['components_data'])) {
                    $componentsData = is_string($request['components_data']) 
                        ? json_decode($request['components_data'], true) 
                        : $request['components_data'];
                    
                    // Ako komponente nisu prazan array, obradi ih
                    if (is_array($componentsData) && !empty($componentsData)) {
                        foreach ($componentsData as $name => $data) {
                            if (!isset($components[$name])) {
                                $components[$name] = [
                                    'name' => $name,
                                    'total_time' => 0,
                                    'count' => 0,
                                    'max_time' => 0
                                ];
                            }
                            
                            // Dodaj vrijeme izvršavanja
                            $components[$name]['total_time'] += $data['total_time'] ?? 0;
                            $components[$name]['count'] += $data['count'] ?? 0;
                            
                            // Ažuriraj maksimalno vrijeme ako je potrebno
                            $maxTime = 0;
                            if (isset($data['instances']) && is_array($data['instances'])) {
                                foreach ($data['instances'] as $instance) {
                                    $executionTime = $instance['execution_time'] ?? 0;
                                    if ($executionTime > $maxTime) {
                                        $maxTime = $executionTime;
                                    }
                                }
                            }
                            
                            if ($maxTime > $components[$name]['max_time']) {
                                $components[$name]['max_time'] = $maxTime;
                            }
                        }
                    }
                }
            }
            
            // Izračunaj prosječno vrijeme po komponenti
            $result = [];
            foreach ($components as $name => $data) {
                if ($data['count'] > 0) {
                    $avgTime = $data['total_time'] / $data['count'];
                    
                    $result[] = [
                        'name' => $name,
                        'avg_time' => $avgTime,
                        'max_time' => $data['max_time'],
                        'count' => $data['count']
                    ];
                    
                    $totalTime += $avgTime;
                }
            }
            
            // Ako nema podataka o komponentama, vrati prazan array
            if (empty($result)) {
                return Response::json([
                    'success' => true,
                    'components' => [],
                    'avg_request_time' => 0
                ]);
            }
            
            // Vrati rezultate
            return Response::json([
                'success' => true,
                'components' => $result,
                'avg_request_time' => $totalTime
            ]);
            
        } catch (\Exception $e) {
            $this->logger->error('Failed to get component metrics', [
                'error' => $e->getMessage()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Failed to get component metrics'
            ], 500);
        }
    }

    /**
     * API endpoint za dohvat SQL upita
     */
    public function getDatabaseQueries(Request $request): Response
    {
        // Provjeri je li LUKA uključena
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        try {
            // Dohvati parametre
            $fromStr = $request->getQuery()['from'] ?? '-24 hours';
            $toStr = $request->getQuery()['to'] ?? 'now';
            $limit = (int)($request->getQuery()['limit'] ?? 50);
            
            $from = new \DateTime($fromStr);
            $to = new \DateTime($toStr);
            
            // Dohvati metriku
            $metricsStorage = $this->container->get('lukaMetricsStorage');
            
            // Dohvati SQL upite
            $databaseMetrics = $metricsStorage->getMetrics('database', $from, $to);
            
            // Ograniči broj rezultata
            if (count($databaseMetrics) > $limit) {
                // Sortiramo po prosječnom vremenu izvršavanja (silazno)
                usort($databaseMetrics, function($a, $b) {
                    $aTime = $a['avg_execution_time'] ?? $a['avg_time'] ?? 0;
                    $bTime = $b['avg_execution_time'] ?? $b['avg_time'] ?? 0;
                    return $bTime <=> $aTime;
                });
                
                // Uzmimo samo prvih $limit zapisa
                $databaseMetrics = array_slice($databaseMetrics, 0, $limit);
            }
            
            // Vrati JSON odgovor
            return Response::json([
                'success' => true,
                'from' => $from->format('Y-m-d H:i:s'),
                'to' => $to->format('Y-m-d H:i:s'),
                'data' => $databaseMetrics
            ]);
            
        } catch (\Exception $e) {
            $this->logger->error('Failed to get database queries', [
                'error' => $e->getMessage()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Failed to retrieve database queries'
            ], 500);
        }
    }

    /**
     * API endpoint za dohvat sistemskih logova
     */
    public function getSystemLogs(Request $request): Response
    {
        // Provjeri je li LUKA uključena
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        // Dohvati parametre
        $limit = (int)($request->getQuery()['limit'] ?? 50);
        $logType = $request->getQuery()['type'] ?? 'all';
        
        // Dohvati metriku
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Dohvati datumski raspon
        $from = new \DateTime('-24 hours');
        $to = new \DateTime();
        
        // Dohvati logove
        $systemLogs = $metricsStorage->getMetrics('system_logs', $from, $to);
        
        // Filtriraj po tipu ako je potrebno
        if ($logType !== 'all') {
            $systemLogs = array_filter($systemLogs, function($log) use ($logType) {
                return $log['log_type'] === $logType;
            });
        }
        
        // Vrati JSON odgovor
        return Response::json([
            'success' => true,
            'data' => array_values($systemLogs) // array_values resetira indekse
        ]);
    }

    /**
     * API endpoint za dohvat WebSocket metrika
     */
    public function getWebsocketMetrics(Request $request): Response
    {
        // Provjeri je li LUKA uključena
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        // Provjeri je li WebSocket uključen
        if (!($this->lukaManager->getConfig()['websocket_enabled'] ?? false)) {
            return Response::json(['error' => 'WebSocket is not enabled'], 400);
        }
        
        // Dohvati metriku
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Dohvati datumski raspon
        $from = new \DateTime('-24 hours');
        $to = new \DateTime();
        
        // Dohvati WebSocket metrike
        $websocketMetrics = $metricsStorage->getMetrics('websocket', $from, $to);
        
        // Vrati JSON odgovor
        return Response::json([
            'success' => true,
            'data' => $websocketMetrics
        ]);
    }
    
    /**
     * API endpoint for getting application state
     */
    public function getState(Request $request): Response
    {
        // Check if LUKA is enabled
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        // Get application state
        $appState = $this->lukaManager->getApplicationState();
        
        // Return JSON response
        return Response::json([
            'success' => true,
            'state' => $appState
        ]);
    }
    
    /**
     * API endpoint for clearing metrics data
     */
    public function clearMetrics(Request $request): Response
    {
        // Check if LUKA is enabled
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        // Get metrics storage
        $metricsStorage = $this->container->get('lukaMetricsStorage');
        
        // Clear old data (set retention to 0 to clear all)
        $success = $metricsStorage->cleanupOldData(0);
        
        // Log the action
        $this->logger->luka('Metrics data cleared', [
            'user_id' => $this->authentication->getUser()->id ?? 'unknown',
            'success' => $success
        ]);
        
        // Return JSON response
        return Response::json([
            'success' => $success,
            'message' => $success ? 'Metrics data cleared successfully' : 'Failed to clear metrics data'
        ], $success ? 200 : 500);
    }
    
    /**
     * API endpoint for restarting WebSocket server
     */
    public function restartWebSocket(Request $request): Response
    {
        // Check if LUKA is enabled
        if (!$this->lukaManager->isEnabled()) {
            return Response::json(['error' => 'LUKA Digital Twin is disabled'], 503);
        }
        
        // Check if WebSocket is enabled
        $websocketEnabled = $this->lukaManager->getConfig()['websocket_enabled'] ?? false;
        
        if (!$websocketEnabled || !$this->container->has('lukaEventPublisher')) {
            return Response::json([
                'success' => false,
                'message' => 'WebSocket is not enabled'
            ], 400);
        }
        
        // Get publisher
        $publisher = $this->container->get('lukaEventPublisher');
        
        // Restart WebSocket
        $success = $publisher->restart();
        
        // Log the action
        $this->logger->luka('WebSocket server restart attempted', [
            'user_id' => $this->authentication->getUser()->id ?? 'unknown',
            'success' => $success
        ]);
        
        // Return JSON response
        return Response::json([
            'success' => $success,
            'message' => $success ? 'WebSocket server restarted successfully' : 'Failed to restart WebSocket server'
        ], $success ? 200 : 500);
    }
}