<?php
/**
 * Login kontroler
 * 
 * Rukuje autentikacijom korisnika (prijava, odjava)
 * 
 * @package Controllers
 */
namespace Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Validation\Validator;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Login extends BaseController
{
    /**
     * Tablica korisnika
     * 
     * @var DatabaseTable
     */
    protected DatabaseTable $korisniciTable;
    
    /**
     * Validator
     * 
     * @var Validator
     */
    protected Validator $validator;

    /**
     * Konstruktor
     *
     * @param Authentication $authentication Servis za autentikaciju
     * @param DatabaseTable $korisniciTable Tablica korisnika
     * @param Validator $validator Validator
     * @param Logger $logger Logger instanca
     * @param SessionManager|null $sessionManager Session manager
     */
    public function __construct(
        Authentication $authentication,
        DatabaseTable $korisniciTable,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->validator = $validator;
    }

    /**
     * Prikazuje obrazac za prijavu
     *
     * @return Response
     */
    public function loginForm(): Response
    {
        return $this->response()->render(
            'Login/login.html.php',
            [],
            'Log In'
        );
    }

    /**
     * Obrađuje zahtjev za prijavu
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    public function processLogin(Request $request): Response
    {
        try {
            $rules = [
                'email' => 'required|email',
                'password' => 'required'
            ];

            // Koristi novu metodu za validaciju
            $validation = $this->validateForm(
                $request, 
                $rules, 
                'Login/login.html.php', 
                'Log In'
            );
            
            // Ako je validacija vratila response, to znači da su greške
            if ($validation instanceof Response) {
                return $validation;
            }

            $email = strtolower($request->getPost('email'));
        
            // Prije pokušaja logina, očistimo sve istekle sesije za korisnika
            $user = $this->korisniciTable->find('user_email', $email);
            if (!empty($user)) {
                $userId = $user[0]->id;
                
                // Prvo dohvatimo session handler preko session managera
                $sessionHandler = $this->sessionManager->getSessionHandler();
                if ($sessionHandler instanceof \baseKRIZAN\Session\DatabaseSessionHandler) {
                    $sessionHandler->cleanExpiredSessionsForUser($userId);
                }
            }
            
            if ($this->authentication->login($email, $request->getPost('password'))) {
                // Dohvati korisnika i provjeri verifikaciju
                $user = $this->korisniciTable->find('user_email', $email);
                
                if (!empty($user) && $user[0]->email_verified == 0) {
                    $this->logger->security('Login attempt - unverified email', [
                        'email' => $email,
                        'ip' => $request->getIp()
                    ]);
                    
                    return $this->response()->renderForm(
                        ['email' => $email],
                        ['auth' => ['Please verify your email before logging in. Check your inbox for the verification link.']],
                        'Login/login.html.php',
                        'Log In'
                    );
                }
                
                $this->logger->security('Successful login', [
                    'email' => $email,
                    'ip' => $request->getIp()
                ]);
                return Response::redirect('login/success');
            }

            $this->logger->security('Failed login attempt - invalid credentials', [
                'email' => $email,
                'ip' => $request->getIp()
            ]);

            return $this->response()->renderForm(
                ['email' => $email],
                ['auth' => ['Invalid email/password combination']],
                'Login/login.html.php',
                'Log In'
            );

        } catch (\Throwable $e) {
            $this->logger->error('Login error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'email' => $request->getPost('email') ?? null
            ]);

            return $this->response()->renderForm(
                ['email' => $request->getPost('email') ?? null],
                ['system' => ['Login failed. Please try again later.']],
                'Login/login.html.php',
                'Log In'
            );
        }
    }

    /**
     * Prikazuje stranicu nakon uspješne prijave
     * 
     * @return Response
     */
    public function loginSuccess(): Response
    {
        try {
            $user = $this->authentication->getUser();
            if (!$user) {
                return Response::redirect('login');
            }

            $currentDate = (new \DateTime($this->getConfig()->get('defaulttimezone')))->format('Y-m-d H:i:s');

            $userData = [
                'id' => $user->id,
                'user_lastlogin' => $currentDate
            ];

            $this->korisniciTable->save($userData);

            return $this->response()->render('Login/loginsuccess.html.php', [], 'Login Successful');
        } catch (\Throwable $e) {
            $this->logger->error('Error updating last login', [
                'message' => $e->getMessage(),
                'user_id' => $user->id ?? null
            ]);
            // Nastaviti na success page čak i ako update user_lastlogin ne uspije
            return $this->response()->render('Login/loginsuccess.html.php', [], 'Login Successful');
        }
    }

    /**
     * Odjavljuje korisnika
     * 
     * @return Response
     */
    public function logout(): Response
    {
        try {
            $user = $this->authentication->getUser();
            $this->authentication->logout();
            
            if ($user) {
                $this->logger->security('User logged out', [
                    'user_id' => $user->id,
                    'email' => $user->user_email ?? null
                ]);
            }

            return $this->response()->render('Login/logout.html.php', [], 'You have been logged out');
        } catch (\Throwable $e) {
            $this->logger->error('Logout error', [
                'message' => $e->getMessage(),
                'user_id' => $user->id ?? null
            ]);
            return Response::redirect('login');
        }
    }

    /**
     * Prikazuje stranicu s greškom za neprijavljen pristup
     * 
     * @return Response
     */
    public function loginError(): Response
    {
        return $this->response()->render(
            'Login/loginerror.html.php',
            [],
            'You are not logged in'
        );
    }
}