<?php

namespace Controllers;

use baseKRIZAN\Services\NotificationService;
use baseKRIZAN\Services\NotificationTypeRegistry;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;

/**
 * NotificationController
 * 
 * Handles all standard notification-related HTTP requests
 */
class NotificationController extends BaseController
{
    protected DatabaseTable $korisniciTable;
    protected NotificationService $notificationService;

    /**
     * Constructor
     */
    public function __construct(
        DatabaseTable $korisniciTable,
        NotificationService $notificationService,
        Authentication $authentication,
        Logger $logger,
        ?SessionManager $sessionManager = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->notificationService = $notificationService;
    }

    /**
     * Notification homepage
     */
    public function home(): Response
    {
        $currentUser = $this->authentication->getUser();
        $users = $this->korisniciTable->findAll('user_lastlogin desc');

        return $this->response()->render(
            'Notification/notification_home.php',
            [
                'currentUser' => $currentUser,
                'users' => $users
            ],
            'Notification'
        );
    }

    /**
     * Create a notification
     * This endpoint can be used by other controllers to create notifications
     */
    public function createNotification(Request $request): Response
    {
        try {
            if (!$this->authentication->isLoggedIn()) {
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $data = json_decode(file_get_contents("php://input"), true);
            
            if (!isset($data['title']) || !isset($data['message'])) {
                return Response::json([
                    'success' => false,
                    'error' => 'Missing required parameters'
                ], 400);
            }
            
            $currentUser = $this->authentication->getUser();
            
            // Prepare notification data
            $notificationData = [
                'from_id' => $currentUser->id,
                'title' => $data['title'],
                'message' => $data['message'],
                'type' => $data['type'] ?? 'general',
                'route' => $data['route'] ?? '',
                'related_id' => $data['related_id'] ?? null,
                'is_sticky' => $data['is_sticky'] ?? false,
                'is_read' => false,
                'send_push' => $data['send_push'] ?? true
            ];
            
            // Get recipients
            $recipients = $data['recipients'] ?? [];
            
            // Create notification
            $notificationId = $this->notificationService->createNotification(
                $notificationData,
                $recipients
            );
            
            return Response::json([
                'success' => $notificationId !== null,
                'notification_id' => $notificationId
            ]);
            
        } catch (\Exception $e) {
            $this->logger->error('Error creating notification', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Failed to create notification'
            ], 500);
        }
    }

    /**
     * Get user notifications
     */
    public function getNotifications(Request $request): Response
    {
        try {
            // Ensure session is started and consistent if sessionManager is available
            if ($this->sessionManager !== null) {
                $this->sessionManager->ensureSessionIntegrity();
            }

            if (!$this->authentication->isLoggedIn()) {
                $this->logger->notification('Notification request from unauthenticated user');
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            
            if (!$user) {
                $this->logger->notification('Unable to retrieve user object for notifications');
                return Response::json([
                    'success' => false,
                    'error' => 'User information not available'
                ], 400);
            }
            
            $userId = (int)$user->id;
            
            if ($userId <= 0) {
                $this->logger->notification('Invalid user ID for notifications', ['id' => $userId]);
                return Response::json([
                    'success' => false,
                    'error' => 'Invalid user ID'
                ], 400);
            }
            
            $limit = min(max((int)($request->getQuery('limit') ?? 10), 1), 100);
            $filter = $request->getQuery('filter') ?? 'default';
            
            $this->logger->notification('Fetching notifications', [
                'user_id' => $userId,
                'limit' => $limit,
                'filter' => $filter
            ]);

            // Get notifications for display based on requested filter
            $notifications = $this->notificationService->getUserNotifications($userId, $limit, $filter);

            return Response::json([
                'success' => true,
                'notifications' => $notifications
            ]);

        } catch (\Exception $e) {
            $this->logger->error('Error fetching notifications', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $userId ?? 'unknown'
            ]);

            return Response::json([
                'success' => false,
                'error' => 'Failed to fetch notifications'
            ], 500);
        }
    }

    /**
     * Mark notifications as read
     */
    public function markNotificationsAsRead(Request $request): Response
    {
        try {
            if (!$this->authentication->isLoggedIn()) {
                return Response::json(['error' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            
            // Check if user object exists and has id property
            if (!$user || !isset($user->id)) {
                return Response::json([
                    'success' => false,
                    'error' => 'User information not available'
                ], 400);
            }
            
            $userId = (int)$user->id;
            
            // Get single notification ID if provided
            $data = json_decode(file_get_contents("php://input"), true);
            $notificationId = isset($data['notification_id']) ? (int)$data['notification_id'] : null;
            
            // Mark specific notification or all as read
            $result = ($notificationId !== null) 
                ? $this->notificationService->markNotificationAsRead($userId, $notificationId)
                : $this->notificationService->markNotificationsAsRead($userId);

            return Response::json(['success' => $result]);
        } catch (\Exception $e) {
            $this->logger->error('Error marking notifications as read', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Failed to mark notifications as read'
            ], 500);
        }
    }

    /**
     * Toggle sticky status for notification
     */
    public function toggleSticky(Request $request): Response
    {
        try {
            $data = json_decode(file_get_contents("php://input"), true);
            $notificationId = (int)($data['notification_id'] ?? 0);
            
            if ($notificationId <= 0) {
                return Response::json(['success' => false, 'message' => 'Invalid notification ID']);
            }

            if (!$this->authentication->isLoggedIn()) {
                return Response::json(['success' => false, 'message' => 'Not authenticated'], 401);
            }

            $user = $this->authentication->getUser();
            
            // Verify notification belongs to user
            $notification = $this->notificationService->getNotificationById($notificationId);
            
            if (!$notification || $notification['for_id'] != $user->id) {
                return Response::json(['success' => false, 'message' => 'Notification not found or access denied']);
            }
            
            // Toggle sticky status
            $result = $this->notificationService->toggleStickyStatus($notificationId);
            
            return Response::json([
                'success' => $result,
                'message' => $result ? 'Sticky status updated' : 'Failed to update sticky status'
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error toggling notification sticky status', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false, 
                'message' => 'An error occurred'
            ], 500);
        }
    }

    /**
     * Get notification types and icons for JavaScript
     */
    public function getNotificationTypesData(Request $request): Response
    {
        try {
            $data = NotificationTypeRegistry::getNotificationTypesData();
            
            return Response::json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            $this->logger->error('Error getting notification types data', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json([
                'success' => false,
                'error' => 'Failed to get notification types data'
            ], 500);
        }
    }
}