<?php
/**
 * Password kontroler
 * 
 * Rukuje reset-om lozinke i promjenom lozinke
 * 
 * @package Controllers
 */
namespace Controllers;

use baseKRIZAN\Services\EmailService;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Validation\Validator;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

class Password extends BaseController
{
    /**
     * Vrijeme isteka tokena u sekundama (1 sat)
     */
    private const TOKEN_EXPIRY = 3600;
    
    /**
     * Tablica korisnika
     * 
     * @var DatabaseTable
     */
    protected DatabaseTable $korisniciTable;
    
    /**
     * Tablica s tokenima za reset lozinke
     * 
     * @var DatabaseTable
     */
    protected DatabaseTable $passwordresetTable;
    
    /**
     * Validator
     * 
     * @var Validator
     */
    protected Validator $validator;
    
    /**
     * Servis za slanje emaila
     * 
     * @var EmailService
     */
    protected EmailService $emailService;

    /**
     * Konstruktor
     *
     * @param DatabaseTable $korisniciTable Tablica korisnika
     * @param DatabaseTable $passwordresetTable Tablica za reset lozinke
     * @param Validator $validator Validator
     * @param Logger $logger Logger
     * @param EmailService $emailService Servis za slanje emaila
     * @param SessionManager|null $sessionManager Session manager
     * @param Authentication|null $authentication Authentication servis
     */
    public function __construct(
        DatabaseTable $korisniciTable,
        DatabaseTable $passwordresetTable,
        Validator $validator,
        Logger $logger,
        EmailService $emailService,
        ?SessionManager $sessionManager = null,
        ?Authentication $authentication = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->passwordresetTable = $passwordresetTable;
        $this->validator = $validator;
        $this->emailService = $emailService;
    }

    /**
     * Prikazuje obrazac za reset lozinke
     *
     * @return Response
     */
    public function passwordresetForm(): Response
    {
        return $this->response()->render(
            'Password/passwordreset.html.php',
            [],
            'Password reset'
        );
    }

    /**
     * Obrađuje zahtjev za reset lozinke
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    public function passwordreset(Request $request): Response
    {
        try {
            $rules = [
                'email' => 'required|email'
            ];
    
            // Koristi novu metodu za validaciju
            $validation = $this->validateForm(
                $request, 
                $rules, 
                'Password/passwordreset.html.php', 
                'Password reset'
            );
            
            // Ako je validacija vratila response, znači da postoje greške
            if ($validation instanceof Response) {
                return $validation;
            }
    
            $email = $request->getPost('email');
            $userInfo = $this->korisniciTable->find('user_email', $email);
    
            if (!$userInfo) {
                $this->logger->security('Password reset attempted for non-existent email', [
                    'email' => $email,
                    'ip' => $request->getIp()
                ]);
                
                return $this->response()->renderForm(
                    ['email' => $email],
                    ['email' => ['Unknown email address']],
                    'Password/passwordreset.html.php',
                    'Password reset'
                );
            }
    
            $userId = $userInfo[0]->id;
            $userEmail = $userInfo[0]->user_email;
    
            $token = bin2hex(random_bytes(32));
            
            // First invalidate existing tokens
            $this->invalidateExistingTokens($userId);
    
            // Create new reset token record
            $resetRecord = [
                'user_id' => $userId,
                'user_email' => $userEmail,
                'date_requested' => date("Y-m-d H:i:s"),
                'token' => $token,
                'token_status' => 'new_token',
                'ip_address' => $request->getIp()
            ];
            
            $this->passwordresetTable->save($resetRecord);
            
            // Generate and save the reset link
            $linkToSend = $this->generateResetLink($userId, $token);
            $this->passwordresetTable->save([
                'id' => $this->passwordresetTable->getLastInsertedId(),
                'linkzaposlat' => $linkToSend
            ]);

            try {
                $appName = $this->getConfig()->get('appname');
                $this->emailService->send(
                    $userEmail,
                    "$appName reset lozinke",
                    $this->getPasswordResetEmailContent($linkToSend)
                );
            } catch (\Throwable $e) {
                $this->logger->error('Failed to send reset email', [
                    'message' => $e->getMessage(),
                    'email' => $userEmail
                ]);
                // Nastavljamo dalje jer je token već kreiran
            }

            $this->logger->security('Password reset initiated', [
                'user_id' => $userId,
                'email' => $userEmail,
                'ip' => $request->getIp()
            ]);

            // Show an informative message instead of redirecting directly to login
            return $this->response()->renderInfo(
                'Zahtjev za resetiranje lozinke poslan',
                'Poslali smo vam e-mail s uputama za resetiranje lozinke. Molimo provjerite svoju e-mail adresu i slijedite upute za postavljanje nove lozinke.',
                'Resetiranje lozinke'
            );

        } catch (\Throwable $e) {
            $this->logger->error('Password reset error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->response()->renderForm(
                ['email' => $email ?? ''],
                ['system' => ['An error occurred. Please try again later.']],
                'Password/passwordreset.html.php',
                'Password reset'
            );
        }
    }

    /**
     * Potvrđuje reset lozinke kroz link u emailu
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    public function confirmpasswordreset(Request $request): Response
    {
        try {
            $passwordRequestId = $request->getQuery('id');
            $userId = $request->getQuery('uid');
            $token = $request->getQuery('t');
        
            if (empty($passwordRequestId) || empty($userId) || empty($token)) {
                $this->logger->security('Invalid reset parameters', [
                    'ip' => $request->getIp(),
                    'params' => [
                        'id' => $passwordRequestId,
                        'uid' => $userId,
                        'token_provided' => !empty($token)
                    ]
                ]);

                return $this->response()->renderError('Invalid or expired reset token');
            }
        
            if (!$this->isValidResetRequest($passwordRequestId, $userId, $token)) {
                return $this->response()->renderError('Invalid or expired reset token');
            }
        
            $this->sessionManager->set('user_id_reset_pass', $userId);
            $this->sessionManager->set('reset_token_id', $passwordRequestId); // Store token ID for later update
            
            return Response::redirect('password/update');

        } catch (\Throwable $e) {
            $this->logger->error('Confirm password reset error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return $this->response()->renderError('An error occurred. Please try again later');
        }
    }

    /**
     * Prikazuje obrazac za ažuriranje lozinke i obrađuje zahtjev
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    public function passwordupdateForm(Request $request): Response
    {
        if (!$this->sessionManager->has('user_id_reset_pass')) {
            return $this->response()->renderError('No password reset session found');
        }

        if ($request->getMethod() === 'POST') {
            return $this->handlePasswordUpdate($request);
        }

        return $this->response()->renderForm(
            ['password' => '', 'password2' => ''],
            $this->validator->getErrors(),
            'Password/passwordupdate.html.php',
            'Password update'
        );
    }

    /**
     * Prikazuje listu zahtjeva za reset lozinke
     *
     * @return Response
     */
    public function list(): Response
    {
        $korisnici = $this->passwordresetTable->findAll('date_requested desc');

        return $this->response()->render(
            'Password/passwordresetlist.html.php',
            ['korisnici' => $korisnici],
            'Password reset list'
        );
    }

    /**
     * Poništava postojeće tokene za korisnika
     *
     * @param int $userId ID korisnika
     * @return void
     */
    private function invalidateExistingTokens(int $userId): void
    {
        $existingTokens = $this->passwordresetTable->find('user_id', $userId);
        if ($existingTokens) {
            foreach ($existingTokens as $token) {
                if ($token->token_status === 'new_token') {
                    $this->passwordresetTable->save([
                        'id' => $token->id,
                        'token_status' => 'expired_token'
                    ]);
                }
            }
        }
    }

    /**
     * Generira link za reset lozinke
     *
     * @param string $userId ID korisnika
     * @param string $token Token za reset
     * @return string Puni URL za reset lozinke
     */
    private function generateResetLink(string $userId, string $token): string
    {
        $confirmResetUrl = $this->getConfig()->get('paths.app_url') . '/password/confirmreset';
        
        // Build query parameters
        $queryParams = http_build_query([
            'uid' => $userId, 
            'id' => $this->passwordresetTable->getLastInsertedId(), 
            't' => $token
        ]);

        // Combine URL with query parameters
        return $confirmResetUrl . '?' . $queryParams;
    }

    /**
     * Generira HTML sadržaj emaila za reset lozinke
     *
     * @param string $resetLink Link za reset lozinke
     * @return string HTML sadržaj
     */
    private function getPasswordResetEmailContent(string $resetLink): string 
    {
        $appName = $this->getConfig()->get('appname');
        $currentYear = date('Y');

        return <<<HTML
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body {
                    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                    line-height: 1.6;
                    color: #333333;
                    margin: 0;
                    padding: 0;
                }
                .email-container {
                    max-width: 600px;
                    margin: 0 auto;
                    padding: 20px;
                    background-color: #ffffff;
                }
                .header {
                    background-color: #ff6600;
                    color: white;
                    padding: 20px;
                    text-align: center;
                    border-radius: 5px 5px 0 0;
                }
                .content {
                    padding: 30px 20px;
                    background-color: #f8f9fa;
                    border: 1px solid #dee2e6;
                    border-radius: 0 0 5px 5px;
                }
                .reset-title {
                    color: #ff6600;
                    margin-top: 0;
                }
                .button {
                    display: inline-block;
                    padding: 12px 24px;
                    background-color: #ff6600;
                    color: white;
                    text-decoration: none;
                    border-radius: 5px;
                    margin: 20px 0;
                    font-weight: bold;
                }
                .button-container {
                    text-align: center;
                }
                .warning {
                    margin-top: 20px;
                    padding: 10px;
                    background-color: #fff3cd;
                    border: 1px solid #ffeeba;
                    border-radius: 4px;
                    color: #856404;
                    font-size: 14px;
                }
                .footer {
                    margin-top: 20px;
                    text-align: center;
                    font-size: 12px;
                    color: #6c757d;
                    border-top: 1px solid #dee2e6;
                    padding-top: 20px;
                }
            </style>
        </head>
        <body>
            <div class="email-container">
                <div class="header">
                    <h2>{$appName}</h2>
                </div>
                <div class="content">
                    <h1 class="reset-title">Reset lozinke</h1>
                    <p>Poštovani,</p>
                    <p>Primili smo zahtjev za reset vaše lozinke. Za nastavak procesa, molimo kliknite na gumb ispod:</p>
                    
                    <div class="button-container">
                        <a href="{$resetLink}" class="button">Reset lozinke</a>
                    </div>

                    <div class="warning">
                        <strong>Napomena:</strong> Link za reset lozinke je valjan 24 sata od trenutka slanja ovog emaila.
                    </div>

                    <p>Ako niste zatražili reset lozinke, molimo ignorirajte ovaj email ili nas kontaktirajte ako smatrate da je došlo do pogreške.</p>
                </div>
                <div class="footer">
                    <p>Ovo je automatski generirana poruka. Molimo ne odgovarajte na nju.</p>
                    <p>&copy; {$appName} {$currentYear}. Sva prava pridržana.</p>
                </div>
            </div>
        </body>
        </html>
        HTML;
    }

    /**
     * Provjerava je li zahtjev za reset lozinke valjan
     *
     * @param string $passwordRequestId ID zahtjeva
     * @param string $userId ID korisnika
     * @param string $token Token za reset
     * @return bool
     */
    private function isValidResetRequest(string $passwordRequestId, string $userId, string $token): bool
    {
        try {
            $conditions = [
                ['id', $passwordRequestId, '='],
                ['user_id', $userId, '='],
                ['token', $token, '='],
                ['token_status', 'new_token', '='],
                ['date_requested', date('Y-m-d H:i:s', time() - self::TOKEN_EXPIRY), '>']
            ];

            $result = $this->passwordresetTable->findMultiCondition($conditions);
            
            if (!$result) {
                $this->logger->security('Invalid password reset attempt', [
                    'request_id' => $passwordRequestId,
                    'user_id' => $userId,
                    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
                ]);
                return false;
            }

            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Password reset validation error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }
    
    /**
     * Obrađuje ažuriranje lozinke
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    private function handlePasswordUpdate(Request $request): Response
    {
        try {
            $rules = [
                'password' => ['required', 'strong_password', ['min', 8]],
                'password2' => ['required', ['password_match', $request->getPost('password')]]
            ];

            // Koristi novu metodu za validaciju
            $validation = $this->validateForm(
                $request, 
                $rules, 
                'Password/passwordupdate.html.php', 
                'Password update'
            );
            
            // Ako je validacija vratila response, znači da postoje greške
            if ($validation instanceof Response) {
                return $validation;
            }

            $this->saveNewPassword($request->getPost('password'));

            // Update the specific token record
            if ($this->sessionManager->has('reset_token_id')) {
                $this->passwordresetTable->save([
                    'id' => $this->sessionManager->get('reset_token_id'),
                    'token_status' => 'used_token'
                ]);
            }
            
            $this->logger->security('Password reset completed', [
                'user_id' => $this->sessionManager->get('user_id_reset_pass'),
                'ip' => $request->getIp()
            ]);
            
            $this->sessionManager->clear();
            return Response::redirect('login');

        } catch (\Throwable $e) {
            $this->logger->error('Password update error', [
                'message' => $e->getMessage(),
                'user_id' => $this->sessionManager->get('user_id_reset_pass')
            ]);

            return $this->response()->renderForm(
                ['password' => '', 'password2' => ''],
                ['system' => ['An error occurred while updating your password. Please try again.']],
                'Password/passwordupdate.html.php',
                'Password update'
            );
        }
    }

    /**
     * Sprema novu lozinku za korisnika
     *
     * @param string $password Nova lozinka
     * @return void
     */
    private function saveNewPassword(string $password): void
    {
        $this->korisniciTable->save([
            'id' => $this->sessionManager->get('user_id_reset_pass'),
            'user_password' => password_hash($password, PASSWORD_DEFAULT)
        ]);
    }
}