<?php

namespace Controllers;

use baseKRIZAN\Services\EmailService;
use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Validation\Validator;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;

/**
 * Kontroler za registraciju korisnika
 * 
 * @package Controllers
 */
class Register extends BaseController
{
    /**
     * Tablica korisnika
     * 
     * @var DatabaseTable
     */
    protected DatabaseTable $korisniciTable;
    
    /**
     * Validator
     * 
     * @var Validator
     */
    protected Validator $validator;
    
    /**
     * Servis za slanje e-mailova
     * 
     * @var EmailService
     */
    protected EmailService $emailService;

    /**
     * Konstruktor
     *
     * @param DatabaseTable $korisniciTable Tablica korisnika
     * @param Validator $validator Validator
     * @param Logger $logger Logger
     * @param EmailService $emailService Servis za slanje e-mailova
     * @param SessionManager|null $sessionManager Session manager
     * @param Authentication|null $authentication Authentication servis
     */
    public function __construct(
        DatabaseTable $korisniciTable,
        Validator $validator,
        Logger $logger,
        EmailService $emailService,
        ?SessionManager $sessionManager = null,
        ?Authentication $authentication = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->validator = $validator;
        $this->emailService = $emailService;
    }

    /**
     * Prikazuje formular za registraciju
     *
     * @return Response
     */
    public function registrationForm(): Response
    {
        return $this->response()->render(
            'Register/register.html.php',
            [],
            'Register an account'
        );
    }

    /**
     * Obrađuje registraciju korisnika
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    public function registerUser(Request $request): Response
    {
        try {
            $rules = [
                'email' => 'required|email|allowed_domain|unique_email',
                'password' => ['required', ['min', 8], 'strong_password']
            ];

            // Koristimo novu metodu za validaciju forme
            $validation = $this->validateForm(
                $request, 
                $rules, 
                'Register/register.html.php', 
                'Register an account'
            );
            
            // Ako je validacija vratila response, znači da postoje greške
            if ($validation instanceof Response) {
                $this->logger->security('Failed registration attempt - validation failed', [
                    'email' => $request->getPost('email'),
                    'ip' => $request->getIp()
                ]);
                
                return $validation;
            }

            $email = strtolower($request->getPost('email'));
            
            // Double-check email uniqueness
            if ($this->emailExists($email)) {
                $this->logger->security('Registration attempt with existing email', [
                    'email' => $email,
                    'ip' => $request->getIp()
                ]);
                
                return $this->response()->renderForm(
                    ['email' => $email],
                    ['email' => ['Email already registered']],
                    'Register/register.html.php',
                    'Register an account'
                );
            }

            // Generiranje tokena za verifikaciju
            $verificationToken = bin2hex(random_bytes(32));

            $korisnik = [
                'date_created' => (new \DateTime($this->getConfig()->get('defaulttimezone')))->format('Y-m-d H:i:s'),
                'user_email' => $email,
                'user_password' => password_hash($request->getPost('password'), PASSWORD_DEFAULT),
                'registration_ip' => $request->getIp(),
                'email_verified' => 0,
                'verification_token' => $verificationToken,
                'verification_expires' => (new \DateTime($this->getConfig()->get('defaulttimezone')))
                    ->modify('+1 hours')
                    ->format('Y-m-d H:i:s') // Token istječe za 1h
            ];

            // Spremanje korisnika
            $this->korisniciTable->save($korisnik);

            // Slanje verifikacijskog emaila korisniku
            $this->sendVerificationEmail($email, $verificationToken);

            // Slanje emaila adminu
            try {
                $this->emailService->send(
                    $this->getConfig()->get('email.admin'),
                    "Registracija novog korisnika",
                    $this->getAdminEmailContent($email)
                );
            } catch (\Throwable $e) {
                $this->logger->error('Failed to send admin notification', [
                    'message' => $e->getMessage(),
                    'new_user_email' => $email
                ]);
            }

            return $this->response()->renderInfo(
                'Info',
                'Da biste završili registraciju, molimo potvrdite svoj email. Poslali smo vam link za verifikaciju.',
                'Registration Info'
            );

        } catch (\Throwable $e) {
            $this->logger->error('Registration error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'email' => $request->getPost('email') ?? null
            ]);

            return $this->response()->renderForm(
                ['email' => $request->getPost('email')],
                ['system' => ['Registration failed. Please try again later.']],
                'Register/register.html.php',
                'Register an account'
            );
        }
    }

    /**
     * Obrađuje verifikaciju e-maila
     *
     * @param Request $request HTTP zahtjev
     * @return Response
     */
    public function verifyEmail(Request $request): Response 
    {
        $token = $request->getQuery('token');
        
        if (empty($token)) {
            $this->logger->security('Empty verification token provided', [
                'ip' => $request->getIp()
            ]);
            return $this->response()->renderError('Neispravan verifikacijski token.');
        }

        try {
            $korisnik = $this->korisniciTable->find('verification_token', $token);
            
            if (empty($korisnik)) {
                $this->logger->security('Invalid verification token provided', [
                    'token' => $token,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Neispravan verifikacijski token.');
            }

            // Check if token has expired
            $expires = new \DateTime($korisnik[0]->verification_expires);
            if ($expires < new \DateTime()) {
                $this->logger->security('Expired verification token used', [
                    'token' => $token,
                    'email' => $korisnik[0]->user_email,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Verifikacijski link je istekao.');
            }

            // Mark email as verified
            $this->korisniciTable->save([
                'id' => $korisnik[0]->id,
                'email_verified' => 1,
                'verification_token' => null,
                'verification_expires' => null
            ]);

            $this->logger->security('Email verified successfully', [
                'email' => $korisnik[0]->user_email,
                'ip' => $request->getIp()
            ]);

            // Use direct rendering instead of redirect to avoid potential redirect loops
            return $this->registrationSuccess();
        } catch (\Throwable $e) {
            $this->logger->error('Email verification failed', [
                'token' => $token,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return $this->response()->renderError('Verifikacija nije uspjela. Molimo pokušajte ponovno.');
        }
    }

    /**
     * Prikazuje stranicu uspješne registracije
     * 
     * Preimenovali smo success() u registrationSuccess() da izbjegnemo konflikt
     * s metodom BaseController::success()
     *
     * @return Response
     */
    public function registrationSuccess(): Response
    {
        return $this->response()->render(
            'Register/registersuccess.html.php',
            [],
            'Registration Successful'
        );
    }

    /**
     * Provjerava postoji li e-mail adresa
     *
     * @param string $email E-mail adresa
     * @return bool
     */
    private function emailExists(string $email): bool
    {
        $existingUser = $this->korisniciTable->find('user_email', $email);
        return !empty($existingUser);
    }

    /**
     * Šalje verifikacijski e-mail
     *
     * @param string $email E-mail adresa
     * @param string $token Token za verifikaciju
     * @return void
     */
    private function sendVerificationEmail(string $email, string $token): void 
    {
        $verificationLink = $this->getConfig()->get('paths.app_url') . '/verify-email?token=' . $token;
        
        $subject = "Potvrdite vašu email adresu - " . $this->getConfig()->get('appname');
        
        $body = $this->getVerificationEmailTemplate([
            'verificationLink' => $verificationLink,
            'appName' => $this->getConfig()->get('appname'),
            'expiresIn' => '24 sata'
        ]);

        try {
            $this->emailService->send(
                $email,
                $subject,
                $body
            );

            $this->logger->security('Verification email sent', [
                'email' => $email
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to send verification email', [
                'email' => $email,
                'error' => $e->getMessage()
            ]);
            // Ne prekidamo registraciju ako email nije poslan
        }
    }

    /**
     * Generira HTML predložak za verifikacijski e-mail
     *
     * @param array $data Podaci za predložak
     * @return string HTML sadržaj e-maila
     */
    private function getVerificationEmailTemplate(array $data): string 
    {
        $currentYear = date('Y');
        
        return <<<HTML
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body {
                    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                    line-height: 1.6;
                    color: #333333;
                    margin: 0;
                    padding: 0;
                }
                .email-container {
                    max-width: 600px;
                    margin: 0 auto;
                    padding: 20px;
                    background-color: #ffffff;
                }
                .header {
                    background-color: #ff6600;
                    color: white;
                    padding: 20px;
                    text-align: center;
                    border-radius: 5px 5px 0 0;
                }
                .content {
                    padding: 30px 20px;
                    background-color: #f8f9fa;
                    border: 1px solid #dee2e6;
                    border-radius: 0 0 5px 5px;
                }
                .welcome-title {
                    color: #ff6600;
                    margin-top: 0;
                }
                .button {
                    display: inline-block;
                    padding: 12px 24px;
                    background-color: #ff6600;
                    color: white;
                    text-decoration: none;
                    border-radius: 5px;
                    margin: 20px 0;
                    font-weight: bold;
                }
                .verification-link {
                    background-color: #e9ecef;
                    padding: 15px;
                    border-radius: 4px;
                    margin: 20px 0;
                    word-break: break-all;
                    font-size: 14px;
                }
                .expiry-notice {
                    margin-top: 20px;
                    padding: 10px;
                    background-color: #fff3cd;
                    border: 1px solid #ffeeba;
                    border-radius: 4px;
                    color: #856404;
                    font-size: 14px;
                }
                .footer {
                    margin-top: 20px;
                    text-align: center;
                    font-size: 12px;
                    color: #6c757d;
                    border-top: 1px solid #dee2e6;
                    padding-top: 20px;
                }
                .button-container {
                    text-align: center;
                }
            </style>
        </head>
        <body>
            <div class="email-container">
                <div class="header">
                    <h2>{$data['appName']}</h2>
                </div>
                <div class="content">
                    <h1 class="welcome-title">Dobrodošli!</h1>
                    
                    <p>Hvala vam na registraciji. Za aktivaciju vašeg računa, molimo potvrdite vašu email adresu:</p>
                    
                    <div class="button-container">
                        <a href="{$data['verificationLink']}" class="button">Potvrdi email adresu</a>
                    </div>

                    <p>Ili kopirajte ovaj link u vaš preglednik:</p>
                    <div class="verification-link">
                        {$data['verificationLink']}
                    </div>

                    <div class="expiry-notice">
                        <strong>Napomena:</strong> Link je valjan {$data['expiresIn']}.
                    </div>

                    <p>Ako niste vi kreirali ovaj račun, možete ignorirati ovaj email.</p>
                </div>
                <div class="footer">
                    <p>Lijep pozdrav,<br>{$data['appName']} tim</p>
                    <p>&copy; {$data['appName']} {$currentYear}. Sva prava pridržana.</p>
                </div>
            </div>
        </body>
        </html>
    HTML;
    }

    /**
     * Generira sadržaj e-maila za administratora
     *
     * @param string $userEmail E-mail adresa korisnika
     * @return string HTML sadržaj e-maila
     */
    private function getAdminEmailContent(string $userEmail): string 
    {
        $appName = $this->getConfig()->get('appname');
        $fullPath = $this->getConfig()->get('paths.app_url');
        
        return "
            <p>Registriran novi korisnik: {$userEmail}</p>
            <p>Dodijelite mu prava na: 
                <a href='{$fullPath}/users/list'>{$appName}</a>
            </p>
            <p>Vrijeme registracije: " . date('Y-m-d H:i:s') . "</p>
        ";
    }
}