<?php
// Create a new file: app/Controllers/Security.php
namespace Controllers;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use baseKRIZAN\Security\Authentication;

class Security extends BaseController
{
    public function __construct(
        Logger $logger,
        ?SessionManager $sessionManager = null,
        ?Authentication $authentication = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
    }
    
    /**
     * Handle CSP violation reports
     */
    public function handleCspReport(Request $request): Response
    {
        try {
            // Get the raw POST data
            $reportJson = file_get_contents('php://input');
            $report = json_decode($reportJson, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->logger->security('Invalid CSP report JSON', [
                    'error' => json_last_error_msg(),
                    'raw' => $reportJson
                ]);
                return Response::json(['status' => 'error', 'message' => 'Invalid JSON'], 400);
            }
            
            // Log the CSP violation
            $this->logger->security('CSP Violation', [
                'document-uri' => $report['csp-report']['document-uri'] ?? 'unknown',
                'referrer' => $report['csp-report']['referrer'] ?? 'unknown',
                'blocked-uri' => $report['csp-report']['blocked-uri'] ?? 'unknown',
                'violated-directive' => $report['csp-report']['violated-directive'] ?? 'unknown',
                'effective-directive' => $report['csp-report']['effective-directive'] ?? 'unknown',
                'source-file' => $report['csp-report']['source-file'] ?? 'unknown',
                'line-number' => $report['csp-report']['line-number'] ?? 'unknown',
                'column-number' => $report['csp-report']['column-number'] ?? 'unknown',
                'original-policy' => $report['csp-report']['original-policy'] ?? 'unknown',
                'disposition' => $report['csp-report']['disposition'] ?? 'unknown',
            ]);
            
            // Return a success response
            return Response::json(['status' => 'success']);
        } catch (\Throwable $e) {
            $this->logger->error('Error processing CSP report', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Response::json(['status' => 'error', 'message' => 'Internal Server Error'], 500);
        }
    }
}