<?php

namespace Controllers;

use baseKRIZAN\Security\Authentication;
use baseKRIZAN\Session\SessionManager;
use baseKRIZAN\Validation\Validator;
use Models\DatabaseTable;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\Response;
use Models\Entity\Korisnik;

class User extends BaseController
{
    private const MAX_EXPORT_ROWS = 100;
    private const EDIT_TOKEN_EXPIRY = 1800;
    
    protected DatabaseTable $korisniciTable;
    protected Validator $validator;
    protected ?\baseKRIZAN\Services\EmailService $emailService;

    public function __construct(
        DatabaseTable $korisniciTable,
        Authentication $authentication,
        Validator $validator,
        Logger $logger,
        ?SessionManager $sessionManager = null,
        ?\baseKRIZAN\Services\EmailService $emailService = null
    ) {
        parent::__construct($logger, $sessionManager, $authentication);
        $this->korisniciTable = $korisniciTable;
        $this->validator = $validator;
        $this->emailService = $emailService;
    }

    private function addCustomValidationRules(): void
    {
        $this->validator->addRule(
            'valid_permissions',
            function($value) {
                if (is_numeric($value)) return true;
                if (is_array($value)) {
                    return array_reduce($value, function($carry, $item) {
                        return $carry && is_numeric($item);
                    }, true);
                }
                return false;
            },
            'Invalid permissions value'
        );

        $this->validator->addRule(
            'valid_modules',
            function($value) {
                if ($value === null || $value === '') return true;
                
                if (is_array($value)) {
                    return count($value) === count(array_filter($value, 'is_string'));
                }
                
                return is_string($value) && trim($value) !== '';
            },
            'Invalid modules permissions format'
        );

        $this->validator->addRule(
            'password_strength',
            function($value) {
                if (empty($value)) return true; // Optional field
                return strlen($value) >= 8 && 
                       preg_match('/[A-Za-z]/', $value) && 
                       preg_match('/[0-9]/', $value);
            },
            'Password must be at least 8 characters and contain both letters and numbers'
        );

        $this->validator->addRule(
            'passwords_match',
            function($value, $params, $data) {
                $passwordField = $params[0] ?? 'new_password';
                return empty($data[$passwordField]) || $value === $data[$passwordField];
            },
            'Password confirmation does not match'
        );
    }

    public function list(Request $request): Response
    {
        try {
            $korisnici = $this->korisniciTable->findAll();
            $this->sessionManager->set('valid_user_ids', array_map(fn($korisnik) => $korisnik->id, $korisnici));

            if ($request->getPost('export')) {
                try {
                    $columns = [
                        'id', 'date_created', 'user_email', 'user_firstandlastname', 'user_address', 
                        'user_company', 'user_role', 'user_password', 'user_permissions', 'user_modules', 
                        'user_lastlogin'
                    ];
                    $this->exportUsers($korisnici, $columns, "registrirani korisnici");
                    exit;
                } catch (\Throwable $e) {
                    $this->logger->error('Export failed', [
                        'message' => $e->getMessage(),
                        'user_id' => $this->sessionManager->get['user_id'] ?? null,
                        'ip' => $request->getIp()
                    ]);
                    return $this->response()->renderError('Failed to export users');
                }
            }

            $reflected = new \ReflectionClass(Korisnik::class);
            $constants = $reflected->getConstants();

            return $this->response()->render(
                'User/userslist.html.php',
                [
                    'korisnici' => $korisnici,
                    'permissions' => $constants
                ],
                'Registered users list'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading users list', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return $this->response()->renderError('Failed to load users list');
        }
    }

    public function add(Request $request): Response
    {
        try {
            $reflected = new \ReflectionClass(Korisnik::class);
            $constants = $reflected->getConstants();

            return $this->response()->render(
                'User/useradd.html.php',
                [
                    'permissions' => $constants,
                    'formData' => []
                ],
                'Add New User'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading add user form', [
                'message' => $e->getMessage()
            ]);
            return $this->response()->renderError('Failed to load add user form');
        }
    }

    public function saveAdd(Request $request): Response
    {
        try {
            $this->addCustomValidationRules();

            $formData = [
                'user_email' => $request->getPost('user_email'),
                'user_firstandlastname' => $request->getPost('user_firstandlastname'),
                'user_password' => $request->getPost('user_password'),
                'user_password_confirm' => $request->getPost('user_password_confirm'),
                'user_address' => $request->getPost('user_address'),
                'user_company' => $request->getPost('user_company'),
                'user_role' => $request->getPost('user_role'),
                'user_modules' => isset($_POST['user_modules']) 
                    ? implode(", ", $_POST['user_modules']) 
                    : '',
                'permissions' => $_POST['permissions'] ?? [],
                'user_permissions' => array_sum($_POST['permissions'] ?? []),
                'email_verified' => $request->getPost('email_verified') ? 1 : 0
            ];

            // Validation rules using your existing system
            $rules = [
                'user_email' => ['required', 'email'],
                'user_firstandlastname' => ['required', 'alpha_spaces'],
                'user_password' => ['required', 'min:8'],
                'user_password_confirm' => ['required'],
                'user_address' => ['optional'],
                'user_company' => ['optional'],
                'user_role' => ['optional'],
                'permissions' => 'valid_permissions',
                'user_modules' => 'valid_modules'
            ];

            $validation = $this->validator->validate($request, $rules);
            
            if ($validation->fails()) {
                return $this->renderAddUserForm($formData, $validation->errors());
            }

            // Manual validations
            $errors = [];

            // Check email uniqueness  
            $existingUser = $this->korisniciTable->findBy('user_email', $formData['user_email']);
            if ($existingUser) {
                $errors['user_email'] = ['This email is already registered'];
            }

            // Check password confirmation
            if ($formData['user_password'] !== $formData['user_password_confirm']) {
                $errors['user_password_confirm'] = ['Password confirmation does not match'];
            }

            // Check password strength
            if (strlen($formData['user_password']) < 8 || 
                !preg_match('/[A-Za-z]/', $formData['user_password']) || 
                !preg_match('/[0-9]/', $formData['user_password'])) {
                $errors['user_password'] = ['Password must be at least 8 characters and contain both letters and numbers'];
            }

            if (!empty($errors)) {
                return $this->renderAddUserForm($formData, $errors);
            }

            // Check permissions
            $currentUser = $this->authentication->getUser();
            if (!$currentUser) {
                return $this->response()->renderError('Authentication required');
            }

            $newPermissions = $formData['user_permissions'];
            
            if ($currentUser->user_permissions < Korisnik::KORISNIK_MASTER && 
                $newPermissions > $currentUser->user_permissions) {
                return $this->response()->renderError('Cannot assign higher permissions than your own');
            }

            // Hash password
            $formData['user_password'] = password_hash($formData['user_password'], PASSWORD_DEFAULT);
            
            // Remove confirmation field and permissions array
            unset($formData['user_password_confirm'], $formData['permissions']);
            
            // Set creation date and email verification status
            $formData['date_created'] = date('Y-m-d H:i:s');
            // email_verified is already set from form data
            
            $this->korisniciTable->save($formData);
            
            $this->logger->security('New user created', [
                'email' => $formData['user_email'],
                'created_by' => $currentUser->id,
                'ip' => $request->getIp()
            ]);

            return Response::redirect('users/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error creating user', [
                'message' => $e->getMessage(),
                'email' => $formData['user_email'] ?? null
            ]);
            return $this->response()->renderError('Failed to create user');
        }
    }

    public function permissions(Request $request): Response
    {
        try {
            $userId = $request->getParam('id');
            $korisnik = $this->korisniciTable->findById($userId);
            
            if (!$korisnik) {
                $this->logger->security('Attempt to edit non-existent user', [
                    'user_id' => $userId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unknown user ID');
            }

            $editToken = $this->createEditToken($korisnik->id, 'user');
            $reflected = new \ReflectionClass(Korisnik::class);
            $constants = $reflected->getConstants();

            return $this->response()->render(
                'User/userpermissions.html.php',
                [
                    'title' => $korisnik->user_email,
                    'korisnik' => $korisnik,
                    'permissions' => $constants,
                    'edit_token' => $editToken
                ],
                'Edit Permissions'
            );
        } catch (\Throwable $e) {
            $this->logger->error('Error loading user permissions', [
                'message' => $e->getMessage(),
                'user_id' => $userId ?? null
            ]);
            return $this->response()->renderError('Failed to load user permissions');
        }
    }

    public function savePermissions(Request $request): Response
    {
        try {
            $this->addCustomValidationRules();

            // 1. DOHVAĆANJE OSNOVNIH PODATAKA
            $editToken = $request->getPost('edit_token');
            $userId = $request->getParam('id');
            $validToken = null;
            
            // 2. VALIDACIJA TOKENA (AKO POSTOJI) - ovo NE troši token
            if ($editToken) {
                $tokenResult = $this->validateEditTokenForValidation($request, 'user');
                if (isset($tokenResult['error'])) {
                    $this->logger->security('Invalid edit token', [
                        'ip' => $request->getIp(),
                        'user_id' => $userId
                    ]);
                    
                    return $this->renderUserForm(
                        ['id' => $userId, 'edit_token' => $editToken], 
                        $tokenResult
                    );
                }
                $userId = $tokenResult['id'];
                $validToken = $tokenResult['token'];
            }
            
            // 3. PRIPREMA PODATAKA FORME
            $formData = [
                'id' => $userId,
                'user_firstandlastname' => $request->getPost('user_firstandlastname'),
                'user_address' => $request->getPost('user_address'),
                'user_company' => $request->getPost('user_company'),
                'user_role' => $request->getPost('user_role'),
                'user_modules' => isset($_POST['user_modules']) 
                    ? implode(", ", $_POST['user_modules']) 
                    : '',
                'user_permissions' => array_sum($_POST['permissions'] ?? []),
                'edit_token' => $editToken,
                'new_password' => $request->getPost('new_password'),
                'confirm_password' => $request->getPost('confirm_password')
            ];

            // 4. VALIDACIJA PODATAKA FORME
            $rules = [
                'user_firstandlastname' => ['optional', 'alpha_spaces'],
                'user_address' => ['optional'],
                'user_company' => ['optional'],
                'user_role' => ['optional'],
                'permissions' => 'valid_permissions',
                'user_modules' => 'valid_modules',
                'new_password' => ['optional', 'min:8'],
                'confirm_password' => ['optional']
            ];

            $validation = $this->validator->validate($request, $rules);
            
            if ($validation->fails()) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($userId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'user');
                }
                $formData['edit_token'] = $newEditToken;
                
                return $this->renderUserForm($formData, $validation->errors());
            }

            // Manual validations
            $errors = [];

            // Manual password confirmation check if password is being changed
            if (!empty($formData['new_password'])) {
                if ($formData['new_password'] !== $formData['confirm_password']) {
                    $errors['confirm_password'] = ['Password confirmation does not match'];
                }
                
                // Check password strength
                if (strlen($formData['new_password']) < 8 || 
                    !preg_match('/[A-Za-z]/', $formData['new_password']) || 
                    !preg_match('/[0-9]/', $formData['new_password'])) {
                    $errors['new_password'] = ['Password must be at least 8 characters and contain both letters and numbers'];
                }
            }

            if (!empty($errors)) {
                // Stvori novi token za ponovni pokušaj
                $newEditToken = null;
                if ($userId && $validToken) {
                    $newEditToken = $this->refreshEditToken($validToken, 'user');
                }
                $formData['edit_token'] = $newEditToken;
                
                return $this->renderUserForm($formData, $errors);
            }

            // 5. VALIDACIJA POSLOVNIH PRAVILA
            $currentUser = $this->authentication->getUser();
            if (!$currentUser) {
                return $this->response()->renderError('Authentication required');
            }

            $newPermissions = $formData['user_permissions'];
            
            if ($currentUser->user_permissions < Korisnik::KORISNIK_MASTER && 
                $newPermissions > $currentUser->user_permissions) {
                $this->logger->security('Attempt to escalate privileges', [
                    'user_id' => $userId,
                    'current_user_id' => $currentUser->id,
                    'current_permissions' => $currentUser->user_permissions,
                    'attempted_permissions' => $newPermissions,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Invalid permission assignment');
            }

            // 6. USPJEŠNA VALIDACIJA - SADA troši token (jednokratno)
            if ($validToken) {
                $this->consumeEditToken($validToken);
            }

            // 7. SPREMANJE PODATAKA
            $userData = $formData;
            unset($userData['edit_token'], $userData['confirm_password']);
            
            // Handle password change
            if (!empty($formData['new_password'])) {
                $userData['user_password'] = password_hash($formData['new_password'], PASSWORD_DEFAULT);
                $this->logger->security('User password changed', [
                    'user_id' => $userId,
                    'changed_by' => $currentUser->id,
                    'ip' => $request->getIp()
                ]);
            }
            unset($userData['new_password']);
            
            $this->korisniciTable->save($userData);
            
            $this->logger->security('User permissions updated', [
                'user_id' => $userId,
                'updated_by' => $currentUser->id,
                'ip' => $request->getIp()
            ]);

            return Response::redirect('users/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error saving user permissions', [
                'message' => $e->getMessage(),
                'user_id' => $userId ?? null
            ]);
            return $this->response()->renderError('Failed to save user permissions');
        }
    }

    public function delete(Request $request): Response 
    {
        try {
            $userId = $request->getPost('id');
            $currentUser = $this->authentication->getUser();
            
            if (!$currentUser) {
                return $this->response()->renderError('Authentication required');
            }
        
            $validUserIds = $this->sessionManager->get('valid_user_ids', []);
            if (!in_array($userId, $validUserIds)) {
                $this->logger->security('Unauthorized user deletion attempt', [
                    'user_id' => $userId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Unauthorized action');
            }

            if ($userId == $currentUser->id) {
                $this->logger->security('Attempt to delete own account', [
                    'user_id' => $userId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Cannot delete your own account');
            }

            // Check if trying to delete user with higher permissions
            $userToDelete = $this->korisniciTable->findById($userId);
            if ($userToDelete && 
                $userToDelete->user_permissions > $currentUser->user_permissions) {
                $this->logger->security('Attempt to delete user with higher permissions', [
                    'deleter_id' => $currentUser->id,
                    'target_id' => $userId,
                    'ip' => $request->getIp()
                ]);
                return $this->response()->renderError('Cannot delete user with higher permissions');
            }

            $this->korisniciTable->delete($userId);
            $validUserIds = array_filter(
                $validUserIds, 
                fn($id) => $id != $userId
            );
            $this->sessionManager->set('valid_user_ids', $validUserIds);

            $this->logger->security('User deleted', [
                'deleted_user_id' => $userId,
                'deleted_by' => $currentUser->id,
                'ip' => $request->getIp()
            ]);

            return Response::redirect('users/list');
        } catch (\Throwable $e) {
            $this->logger->error('Error deleting user', [
                'message' => $e->getMessage(),
                'user_id' => $userId ?? null
            ]);
            return $this->response()->renderError('Failed to delete user');
        }
    }

    private function exportUsers(array $korisnici, array $columns, string $filename): void
    {
        try {
            $currentUser = $this->authentication->getUser();
            if (!$currentUser || $currentUser->user_permissions < Korisnik::KORISNIK_MASTER) {
                throw new \RuntimeException('Unauthorized to export users');
            }

            if (count($korisnici) > self::MAX_EXPORT_ROWS) {
                throw new \RuntimeException(
                    sprintf('Export limit exceeded: %d rows', self::MAX_EXPORT_ROWS)
                );
            }

            require_once APP_ROOT . '/app/Helpers/excelexport.php';
            $savefileas = sprintf(
                "%s - %s tjedan %s",
                $filename,
                date('d-m-Y'),
                date("W", strtotime(date('d-m-Y')))
            );
            
            $this->logger->security('Users export initiated', [
                'user_id' => $currentUser->id,
                'rows_count' => count($korisnici)
            ]);

            // Get the export result
            $result = excelexport('korisnici', $columns, $savefileas, $korisnici);
            
            // Set headers for file download
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment; filename="' . $result['filename'] . '"');
            header('Cache-Control: max-age=0');
            header('Pragma: public');
            
            // Output the content and exit
            echo $result['content'];
            exit;
        } catch (\Throwable $e) {
            $this->logger->error('Export failed', [
                'message' => $e->getMessage(),
                'rows_count' => count($korisnici)
            ]);
            throw $e;
        }
    }

    public function permissionsError(): Response
    {
        return $this->response()->render(
            'User/permissionserror.html.php',
            [],
            'Access Denied'
        );
    }

    /**
     * Renders user permissions form with errors
     */
    private function renderUserForm(array $korisnik, array $errors): Response 
    {
        $reflected = new \ReflectionClass(Korisnik::class);
        $constants = $reflected->getConstants();

        // Ako imamo postojeće podatke o korisniku, koristimo njih
        $userPermissions = $korisnik['user_permissions'] ?? 0;
        $korisnikObj = (object)[
            'id' => $korisnik['id'] ?? null,
            'user_permissions' => $userPermissions,
            'user_firstandlastname' => $korisnik['user_firstandlastname'] ?? '',
            'user_address' => $korisnik['user_address'] ?? '',
            'user_company' => $korisnik['user_company'] ?? '',
            'user_role' => $korisnik['user_role'] ?? '',
            'user_email' => $korisnik['user_email'] ?? '',
            'user_modules' => $korisnik['user_modules'] ?? '',
            'hasPermission' => function($permission) use ($userPermissions) {
                return ($userPermissions & $permission) === $permission;
            }
        ];

        return $this->response()->render(
            'User/userpermissions.html.php',
            [
                'errors' => $errors, 
                'korisnik' => $korisnikObj,
                'title' => $korisnik['user_firstandlastname'] ?? 'Edit Permissions',
                'permissions' => $constants,
                'edit_token' => $korisnik['edit_token'] ?? null
            ],
            'Edit Permissions'
        );
    }

    /**
     * Renders add user form with errors
     */
    private function renderAddUserForm(array $formData, array $errors): Response 
    {
        $reflected = new \ReflectionClass(Korisnik::class);
        $constants = $reflected->getConstants();

        return $this->response()->render(
            'User/useradd.html.php',
            [
                'errors' => $errors,
                'formData' => $formData,
                'permissions' => $constants
            ],
            'Add New User'
        );
    }

    /**
     * Send verification email to user
     */
    private function sendVerificationEmail(string $email, string $token): void
    {
        if (!property_exists($this, 'emailService') || !$this->emailService) {
            // If EmailService is not available, skip sending
            $this->logger->warning('EmailService not available for verification email', [
                'email' => $email
            ]);
            return;
        }

        $verificationLink = $this->getConfig()->get('paths.app_url') . '/verify-email?token=' . $token;
        
        $subject = "Please verify your email - " . $this->getConfig()->get('appname');
        
        $body = $this->getVerificationEmailTemplate([
            'verificationLink' => $verificationLink,
            'appName' => $this->getConfig()->get('appname'),
            'expiresIn' => '24 hours'
        ]);

        $this->emailService->send($email, $subject, $body);
        
        $this->logger->security('Verification email sent to admin-created user', [
            'email' => $email
        ]);
    }

    /**
     * Send admin notification about new user
     */
    private function sendAdminNotification(string $userEmail, string $creationType): void
    {
        if (!property_exists($this, 'emailService') || !$this->emailService) {
            return;
        }

        $appName = $this->getConfig()->get('appname');
        $fullPath = $this->getConfig()->get('paths.app_url');
        $createdBy = $creationType === 'admin-created' ? 'by admin' : 'through registration';
        
        $subject = "New user {$createdBy} - {$appName}";
        $body = "
            <p>New user {$createdBy}: {$userEmail}</p>
            <p>Manage user permissions: 
                <a href='{$fullPath}/users/list'>{$appName} User Management</a>
            </p>
            <p>Creation time: " . date('Y-m-d H:i:s') . "</p>
        ";

        try {
            $this->emailService->send(
                $this->getConfig()->get('email.admin'),
                $subject,
                $body
            );
        } catch (\Throwable $e) {
            $this->logger->error('Failed to send admin notification', [
                'user_email' => $userEmail,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Generate verification email template
     */
    private function getVerificationEmailTemplate(array $data): string 
    {
        $currentYear = date('Y');
        
        return <<<HTML
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6; color: #333333; margin: 0; padding: 0; }
                .email-container { max-width: 600px; margin: 0 auto; padding: 20px; background-color: #ffffff; }
                .header { background-color: #ff6600; color: white; padding: 20px; text-align: center; border-radius: 5px 5px 0 0; }
                .content { padding: 30px 20px; background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: 0 0 5px 5px; }
                .welcome-title { color: #ff6600; margin-top: 0; }
                .button { display: inline-block; padding: 12px 24px; background-color: #ff6600; color: white; text-decoration: none; border-radius: 5px; margin: 20px 0; font-weight: bold; }
                .button-container { text-align: center; }
                .footer { margin-top: 20px; text-align: center; font-size: 12px; color: #6c757d; border-top: 1px solid #dee2e6; padding-top: 20px; }
            </style>
        </head>
        <body>
            <div class="email-container">
                <div class="header"><h2>{$data['appName']}</h2></div>
                <div class="content">
                    <h1 class="welcome-title">Account Created</h1>
                    <p>Your account has been created by an administrator. Please verify your email address to complete the setup:</p>
                    <div class="button-container">
                        <a href="{$data['verificationLink']}" class="button">Verify Email Address</a>
                    </div>
                    <p>This link expires in {$data['expiresIn']}.</p>
                </div>
                <div class="footer">
                    <p>&copy; {$data['appName']} {$currentYear}. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>
        HTML;
    }
}