<?php

namespace Models;

use PDO;
use PDOException;

class DatabaseTable
{
    protected $dbConnection;
    protected $table;
    protected $primaryKey;
    protected $className;
    protected $constructorArgs;

    public function __construct(\baseKRIZAN\Database\DatabaseConnection $dbConnection, string $table, string $primaryKey, string $className = '\stdClass', array $constructorArgs = [])
    {
        $this->dbConnection = $dbConnection;
        $this->table = $table;
        $this->primaryKey = $primaryKey;
        $this->className = $className;
        $this->constructorArgs = $constructorArgs;
    }

    public function total($field = null, $value = null)
    {
        $sql = 'SELECT COUNT(*) FROM `' . $this->table . '`';
        $parameters = [];

        if (!empty($field)) {
            $sql .= ' WHERE `' . $field . '` = :value';
            $parameters = ['value' => $value];
        }

        // Koristimo novu metodu iz DatabaseConnection
        return $this->dbConnection->querySingleValue($sql, $parameters);
    }

    public function findById($value)
    {
        $query = 'SELECT * FROM `' . $this->table . '` WHERE `' . $this->primaryKey . '` = :value';

        $parameters = [
            'value' => $value
        ];
        
        // Koristimo novu metodu iz DatabaseConnection
        return $this->dbConnection->queryAndFetchObject($query, $parameters, $this->className, $this->constructorArgs);
    }

    public function findBy($column, $value)
    {
        $query = 'SELECT * FROM `' . $this->table . '` WHERE `' . $column . '` = :value LIMIT 1';
        $parameters = ['value' => $value];
        
        return $this->dbConnection->queryAndFetchObject($query, $parameters, $this->className, $this->constructorArgs);
    }

    public function find($column, $value, $orderBy = null, $limit = null, $offset = null)
    {
        $query = 'SELECT * FROM ' . $this->table . ' WHERE ' . $column . ' = :value';

        $parameters = [
            'value' => $value
        ];

        if ($orderBy != null) {
            $query .= ' ORDER BY ' . $orderBy;
        }

        if ($limit != null) {
            $query .= ' LIMIT ' . $limit;
        }

        if ($offset != null) {
            $query .= ' OFFSET ' . $offset;
        }

        // Koristimo novu metodu iz DatabaseConnection
        return $this->dbConnection->queryAndFetchAllObjects($query, $parameters, $this->className, $this->constructorArgs);
    }

    private function insert($fields)
    {
        $query = 'INSERT INTO `' . $this->table . '` (';

        foreach ($fields as $key => $value) {
            $query .= '`' . $key . '`,';
        }

        $query = rtrim($query, ',');

        $query .= ') VALUES (';


        foreach ($fields as $key => $value) {
            $query .= ':' . $key . ',';
        }

        $query = rtrim($query, ',');

        $query .= ')';

        $fields = $this->processDates($fields);

        $this->dbConnection->execute($query, $fields);

        // Koristimo DatabaseConnection metodu umjesto direktno PDO
        return $this->dbConnection->lastInsertId();
    }


    private function update($fields)
    {
        $query = ' UPDATE `' . $this->table . '` SET ';

        foreach ($fields as $key => $value) {
            $query .= '`' . $key . '` = :' . $key . ',';
        }

        $query = rtrim($query, ',');

        $query .= ' WHERE `' . $this->primaryKey . '` = :primaryKey';

        //Set the :primaryKey variable
        $fields['primaryKey'] = $fields[$this->primaryKey];

        $fields = $this->processDates($fields);

        // Koristimo DatabaseConnection metodu
        $this->dbConnection->execute($query, $fields);
    }


    public function delete($id)
    {
        $parameters = [':id' => $id];

        $this->dbConnection->execute('DELETE FROM `' . $this->table . '` WHERE `' . $this->primaryKey . '` = :id', $parameters);
    }

    public function deleteWhere($column, $value) {
        $query = 'DELETE FROM ' . $this->table . ' WHERE ' . $column . ' = :value';

        $placeholders = [
            'value' => $value
        ];

        $this->dbConnection->execute($query, $placeholders);
    }

    public function findAll($orderBy = null, $limit = null, $offset = null)
    {
        $query = 'SELECT * FROM ' . $this->table;

        if ($orderBy != null) {
            $query .= ' ORDER BY ' . $orderBy;
        }

        if ($limit != null) {
            $query .= ' LIMIT ' . $limit;
        }

        if ($offset != null) {
            $query .= ' OFFSET ' . $offset;
        }

        // Koristimo novu metodu iz DatabaseConnection
        return $this->dbConnection->queryAndFetchAllObjects($query, [], $this->className, $this->constructorArgs);
    }

    private function processDates($fields)
    {
        foreach ($fields as $key => $value) {
            if ($value instanceof \DateTime) {
                $fields[$key] = $value->format('Y-m-d H:i:s');
            }
        }

        return $fields;
    }


    public function save($record)
    {
        $entity = new $this->className(...$this->constructorArgs);

        try {
            if (empty($record[$this->primaryKey])) {
                $record[$this->primaryKey] = null;
            }
            $insertId = $this->insert($record);

            $entity->{$this->primaryKey} = $insertId;
        } catch (PDOException $e) {
            $this->update($record);
        }

        foreach ($record as $key => $value) {
            if (!empty($value)) {
                $entity->$key = $value;
            }
        }

        return $entity;
    }
    
    // MY addition
    public function getLastInsertedId()
    {
        // Koristimo novu metodu iz DatabaseConnection
        return $this->dbConnection->querySingleValue('SELECT LAST_INSERT_ID() AS last_id');
    }

    public function findMultiCondition($conditions, $orderBy = null, $limit = null, $offset = null)
    {
        // Check if conditions is an array
        if (!is_array($conditions)) {
            throw new \InvalidArgumentException('Conditions must be an array.');
        }

        // Start building the query
        $query = 'SELECT * FROM ' . $this->table . ' WHERE ';

        // Add conditions for each column-value-operator triplet
        $conditionClauses = [];
        $parameters = [];
        foreach ($conditions as $index => $condition) {
            if (!is_array($condition) || count($condition) !== 3) {
                throw new \InvalidArgumentException('Each condition must be an array with column, value, and operator.');
            }

            list($column, $value, $operator) = $condition;

            // Check for NULL conditions
            if (strtoupper($value) === 'NULL') {
                if ($operator === '=') {
                    $conditionClauses[] = $column . ' IS NULL';
                } elseif ($operator === '<>') {
                    $conditionClauses[] = $column . ' IS NOT NULL';
                } else {
                    throw new \InvalidArgumentException('Invalid operator for NULL condition: ' . $operator);
                }
                continue; // Skip the parameter binding for NULL conditions
            }

            // Handle LIKE operator for partial matches
            if (strtoupper($operator) === 'LIKE') {
                // Ensure '%' is added to value if not already present
                if (strpos($value, '%') === false) {
                    $value = '%' . $value . '%';
                }
            }

            // Sanitize operator to prevent SQL injection
            if (!in_array($operator, ['=', '<>', '>', '<', '>=', '<=', 'LIKE'])) {
                throw new \InvalidArgumentException('Invalid operator ' . $operator);
            }

            $param = ':value' . $index;
            $conditionClauses[] = $column . ' ' . $operator . ' ' . $param;
            $parameters['value' . $index] = $value;
        }

        // Join conditions with 'AND'
        $query .= implode(' AND ', $conditionClauses);

        // Add ORDER BY clause if provided
        if ($orderBy != null) {
            $query .= ' ORDER BY ' . $orderBy;
        }

        // Add LIMIT clause if provided
        if ($limit != null) {
            $query .= ' LIMIT ' . $limit;
        }

        // Add OFFSET clause if provided
        if ($offset != null) {
            $query .= ' OFFSET ' . $offset;
        }

        // Koristimo novu metodu iz DatabaseConnection
        return $this->dbConnection->queryAndFetchAllObjects($query, $parameters, $this->className, $this->constructorArgs);
    }
}