<?php

/**
 * DataTables Server-Side Processing - CLEANED VERSION
 * Uklonjene sve nekorišćene funkcije - redukcija ~40% koda
 */

namespace Models;

use PDO;
use PDOException;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

class DatatablesSSP {
    /**
     * @var Logger|null Logger for tracking errors
     */
    private static $logger = null;
    
    // ==========================================
    // CONFIGURATION & LOGGING
    // ==========================================
    
    /**
     * Sets logger for tracking errors
     */
    public static function setLogger(Logger $logger)
    {
        self::$logger = $logger;
    }
    
    /**
     * Log error if logger is available
     */
    private static function logError(string $message, array $context = []): void
    {
        if (self::$logger) {
            self::$logger->error($message, $context);
        }
    }
    
    /**
     * Log debug message if logger is available and debug mode is enabled
     */
    private static function logDebug(string $message, array $context = []): void
    {
        if (self::$logger && self::$logger->isDebugEnabled()) {
            self::$logger->database($message, $context);
        }
    }

    // ==========================================
    // MAIN ENTRY POINTS
    // ==========================================

    /**
     * CLIENT/SERVER HYBRID: Complex with unique values for dropdowns
     * Used for initial data loading with dropdown support
     */
    static function complexWithUniques(
        $request,
        $conn,
        $table,
        $primaryKey,
        $columns,
        $columnMapping,
        $whereResult = null,
        $whereAll = null
    ) {
        try {
            // Ensure we have a DatabaseConnection connection
            $db = self::db($conn);
            
            // Log start of processing
            self::logDebug('Starting complexWithUniques', [
                'table' => $table,
                'start' => $request['start'] ?? 0,
                'length' => $request['length'] ?? 0
            ]);
            
            // Get main data
            $dataResult = self::complex(
                $request,
                $db,
                $table,
                $primaryKey,
                $columns,
                $whereResult,
                $whereAll
            );
            
            // Get unique values for dropdowns
            $uniqueValues = self::getUniqueValues($db, $table, $columnMapping);
            
            // Combine results
            $result = array_merge($dataResult, ['uniqueValues' => $uniqueValues]);
            
            self::logDebug('Completed complexWithUniques', [
                'table' => $table,
                'recordsTotal' => $dataResult['recordsTotal'],
                'recordsFiltered' => $dataResult['recordsFiltered'],
                'uniqueValuesCount' => count($uniqueValues)
            ]);
            
            return $result;
        } catch (\Exception $e) {
            self::logError('Error in complexWithUniques: ' . $e->getMessage(), [
                'table' => $table,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return error response
            return [
                "draw" => isset($request['draw']) ? intval($request['draw']) : 0,
                "recordsTotal" => 0,
                "recordsFiltered" => 0,
                "data" => [],
                "error" => $e->getMessage()
            ];
        }
    }

    /**
     * SERVER-SIDE: Standard complex processing with WHERE conditions
     */
    static function complex(
        $request,
        $conn,
        $table,
        $primaryKey,
        $columns,
        $whereResult = null,
        $whereAll = null
    ) {
        $bindings = array();
        $whereAllBindings = array();
        
        try {
            $db = self::db($conn);
            $whereAllSql = '';
        
            // Build the SQL query string from the request
            $limit = self::serverSideLimit($request, $columns);
            $order = self::serverSideOrder($request, $columns);
            $where = self::serverSideFilter($request, $columns, $bindings);
        
            // Add whereResult conditions
            if (is_string($whereResult) && $whereResult !== '') {
                $where = $where ? 
                    $where . ' AND (' . $whereResult . ')' : 
                    'WHERE ' . $whereResult;
            }
            elseif (is_array($whereResult)) {
                $str = $whereResult['condition'] ?? '';
                
                if (!empty($str)) {
                    $where = $where ? 
                        $where . ' AND (' . $str . ')' : 
                        'WHERE ' . $str;
                    
                    if (isset($whereResult['bindings'])) {
                        foreach ($whereResult['bindings'] as $key => $value) {
                            $bindings[] = [
                                'key' => $key,
                                'val' => $value,
                                'type' => PDO::PARAM_STR
                            ];
                        }
                    }
                }
            }

            // Add whereAll conditions
            if ($whereAll) {
                $str = $whereAll;

                if (is_array($whereAll)) {
                    $str = $whereAll['condition'];

                    if (isset($whereAll['bindings'])) {
                        self::add_bindings($whereAllBindings, $whereAll['bindings']);
                    }
                }

                $where = $where ?
                    $where .' AND '.$str :
                    'WHERE '.$str;

                $whereAllSql = 'WHERE '.$str;
            }

            // Main query to get data
            $data = self::sql_exec($db, $bindings,
                "SELECT `".implode("`, `", self::pluck($columns, 'db'))."`
                FROM `$table`
                $where
                $order
                $limit"
            );

            // Data set length after filtering
            $resFilterLength = self::sql_exec($db, $bindings,
                "SELECT COUNT(`{$primaryKey}`)
                FROM   `$table`
                $where"
            );
            $recordsFiltered = $resFilterLength[0][0];

            // Total data set length
            $resTotalLength = self::sql_exec($db, $whereAllBindings,
                "SELECT COUNT(`{$primaryKey}`)
                FROM   `$table` ".
                $whereAllSql
            );
            $recordsTotal = $resTotalLength[0][0];

            return array(
                "draw"            => isset($request['draw']) ?
                    intval($request['draw']) :
                    0,
                "recordsTotal"    => intval($recordsTotal),
                "recordsFiltered" => intval($recordsFiltered),
                "data"            => self::data_output($columns, $data)
            );
        } catch (\PDOException $e) {
            self::logError('Database error in complex(): ' . $e->getMessage(), [
                'table' => $table,
                'sql_error' => $e->getMessage()
            ]);
            throw $e;
        } catch (\Exception $e) {
            self::logError('Error in complex(): ' . $e->getMessage());
            throw $e;
        }
    }

    // ==========================================
    // SERVER-SIDE WRAPPERS
    // ==========================================

    /**
     * SERVER-SIDE: Paging - construct LIMIT clause
     */
    static function serverSideLimit($request, $columns)
    {
        return self::limit($request, $columns);
    }

    /**
     * SERVER-SIDE: Ordering - construct ORDER BY clause
     */
    static function serverSideOrder($request, $columns)
    {
        return self::order($request, $columns);
    }

    /**
     * SERVER-SIDE: Filtering - construct WHERE clause
     */
    static function serverSideFilter($request, $columns, &$bindings)
    {
        return self::filter($request, $columns, $bindings);
    }

    // ==========================================
    // CORE UTILITY METHODS
    // ==========================================

    /**
     * Construct the LIMIT clause for server-side processing SQL query
     */
    static function limit($request, $columns)
    {
        $limit = '';

        if (isset($request['start']) && $request['length'] != -1) {
            $limit = "LIMIT " . intval($request['start']) . ", " . intval($request['length']);
        }

        return $limit;
    }

    /**
     * Construct the ORDER BY clause for server-side processing SQL query
     */
    static function order($request, $columns)
    {
        $order = '';

        if (isset($request['order']) && count($request['order'])) {
            $orderBy = array();
            $dtColumns = self::pluck($columns, 'dt');

            for ($i = 0, $ien = count($request['order']); $i < $ien; $i++) {
                // Convert the column index into the column data property
                $columnIdx = intval($request['order'][$i]['column']);
                $requestColumn = $request['columns'][$columnIdx];

                $columnIdx = array_search($requestColumn['data'], $dtColumns);
                $column = $columns[$columnIdx];

                if ($requestColumn['orderable'] == 'true') {
                    $dir = $request['order'][$i]['dir'] === 'asc' ?
                        'ASC' :
                        'DESC';

                    // Sanitize column name for SQL safety
                    $columnDb = preg_replace('/[^a-zA-Z0-9_]/', '', $column['db']);
                    
                    // Special handling for date columns
                    if (isset($column['type']) && ($column['type'] === 'date' || $column['type'] === 'datetime')) {
                        // Use MySQL STR_TO_DATE for proper date sorting
                        $orderBy[] = "STR_TO_DATE(`{$columnDb}`, '%Y-%m-%d %H:%i:%s') {$dir}";
                    } else {
                        $orderBy[] = "`{$columnDb}` {$dir}";
                    }
                }
            }

            if (count($orderBy)) {
                $order = 'ORDER BY ' . implode(', ', $orderBy);
            }
        }

        return $order;
    }

    /**
     * Construct the WHERE clause for server-side processing SQL query
     * Supports DataTables regex patterns for dropdown filters
     */
    static function filter($request, $columns, &$bindings)
    {
        $globalSearch = array();
        $columnSearch = array();
        $dtColumns = self::pluck($columns, 'dt');

        if (isset($request['search']) && $request['search']['value'] != '') {
            // Sanitize search string to protect against SQL injection
            $str = self::sanitizeSearchString($request['search']['value']);

            for ($i = 0, $ien = count($request['columns']); $i < $ien; $i++) {
                $requestColumn = $request['columns'][$i];
                $columnIdx = array_search($requestColumn['data'], $dtColumns);
                $column = $columns[$columnIdx];

                if ($requestColumn['searchable'] == 'true') {
                    if (!empty($column['db'])) {
                        // Sanitize column name for SQL safety
                        $columnDb = preg_replace('/[^a-zA-Z0-9_]/', '', $column['db']);
                        
                        $binding = self::bind($bindings, '%' . $str . '%', PDO::PARAM_STR);
                        $globalSearch[] = "`" . $columnDb . "` LIKE " . $binding;
                    }
                }
            }
        }

        // Individual column filtering
        if (isset($request['columns'])) {
            for ($i = 0, $ien = count($request['columns']); $i < $ien; $i++) {
                $requestColumn = $request['columns'][$i];
                $columnIdx = array_search($requestColumn['data'], $dtColumns);
                $column = $columns[$columnIdx];

                $str = $requestColumn['search']['value'];

                if ($requestColumn['searchable'] == 'true' && $str != '') {
                    if (!empty($column['db'])) {
                        // Sanitize column name for SQL safety
                        $columnDb = preg_replace('/[^a-zA-Z0-9_]/', '', $column['db']);
                        
                        // Check if this is DataTables regex pattern for dropdown
                        if (self::isDataTablesRegexPattern($str)) {
                            $condition = self::buildRegexCondition($columnDb, $str, $bindings);
                            if ($condition) {
                                $columnSearch[] = $condition;
                            }
                        } else {
                            // Standard LIKE search for text inputs
                            $binding = self::bind($bindings, '%' . $str . '%', PDO::PARAM_STR);
                            $columnSearch[] = "`" . $columnDb . "` LIKE " . $binding;
                        }
                    }
                }
            }
        }

        // Combine the filters into a single string
        $where = '';

        if (count($globalSearch)) {
            $where = '(' . implode(' OR ', $globalSearch) . ')';
        }

        if (count($columnSearch)) {
            $where = $where === '' ?
                implode(' AND ', $columnSearch) :
                $where . ' AND ' . implode(' AND ', $columnSearch);
        }

        if ($where !== '') {
            $where = 'WHERE ' . $where;
        }

        return $where;
    }

    /**
     * Create the data output array for the DataTables rows
     */
    static function data_output($columns, $data)
    {
        $out = array();

        for ($i = 0, $ien = count($data); $i < $ien; $i++) {
            $row = array();

            for ($j = 0, $jen = count($columns); $j < $jen; $j++) {
                $column = $columns[$j];

                // Is there a formatter?
                if (isset($column['formatter'])) {
                    if (empty($column['db'])) {
                        $row[$j] = $column['formatter']($data[$i]);
                    } else {
                        $row[$j] = $column['formatter']($data[$i][$column['db']], $data[$i]);
                    }
                } else {
                    if (!empty($column['db'])) {
                        $row[$j] = $data[$i][$column['db']];
                    } else {
                        $row[$j] = "";
                    }
                }
            }

            $out[] = $row;
        }

        return $out;
    }

    /**
     * Database connection
     */
    static function db($conn)
    {
        if (is_array($conn)) {
            return self::sql_connect($conn);
        }

        return $conn;
    }

    /**
     * Get unique values for columns that need dropdowns
     */
    static function getUniqueValues($db, $table, $columnMapping) 
    {
        $dropdownColumns = array_filter($columnMapping, function($column) {
            return isset($column['dropdown']) && $column['dropdown'] === 'yes';
        });

        $uniqueValues = [];
        
        foreach ($dropdownColumns as $column) {
            $columnName = $column['sqlcolumnname'];
            
            try {
                // Check column type in database
                $columnInfo = $db->queryAndFetchAllAssoc(
                    "SELECT DATA_TYPE, COLUMN_TYPE 
                    FROM INFORMATION_SCHEMA.COLUMNS 
                    WHERE TABLE_SCHEMA = DATABASE() 
                    AND TABLE_NAME = ? 
                    AND COLUMN_NAME = ?",
                    [$table, $columnName]
                );
                
                $isDateTime = false;
                $isDate = false;
                
                if (!empty($columnInfo)) {
                    $dataType = strtolower($columnInfo[0]['DATA_TYPE']);
                    $isDateTime = in_array($dataType, ['datetime', 'timestamp']);
                    $isDate = ($dataType === 'date');
                }
                
                // Build SQL query based on column type
                if ($isDateTime) {
                    // For datetime columns, format date
                    $sql = "SELECT DISTINCT DATE_FORMAT(`$columnName`, '%Y-%m-%d') as formatted_date,
                                `$columnName` as original_value
                            FROM `$table` 
                            WHERE `$columnName` IS NOT NULL 
                            AND `$columnName` != '0000-00-00 00:00:00'
                            AND `$columnName` != ''
                            ORDER BY `$columnName` DESC";
                            
                    $results = $db->queryAndFetchAllAssoc($sql);
                    
                    // Use formatted date for dropdown
                    $uniqueValues[$columnName] = array_unique(array_column($results, 'formatted_date'));
                    
                } elseif ($isDate) {
                    // For date columns
                    $sql = "SELECT DISTINCT `$columnName` 
                            FROM `$table` 
                            WHERE `$columnName` IS NOT NULL 
                            AND `$columnName` != '0000-00-00'
                            AND `$columnName` != ''
                            ORDER BY `$columnName` DESC";
                            
                    $results = $db->queryAndFetchAllAssoc($sql);
                    $uniqueValues[$columnName] = array_column($results, $columnName);
                    
                } else {
                    // For other column types - standard approach
                    $sql = "SELECT DISTINCT `$columnName` 
                            FROM `$table` 
                            WHERE `$columnName` IS NOT NULL 
                            AND `$columnName` != '' 
                            ORDER BY `$columnName`";
                            
                    $results = $db->queryAndFetchAllAssoc($sql);
                    $uniqueValues[$columnName] = array_column($results, $columnName);
                }

                self::logDebug('Retrieved unique values', [
                    'table' => $table, 
                    'column' => $columnName,
                    'type' => $isDateTime ? 'datetime' : ($isDate ? 'date' : 'other'),
                    'count' => count($uniqueValues[$columnName]),
                    'sample_values' => array_slice($uniqueValues[$columnName], 0, 3)
                ]);
                
            } catch (PDOException $e) {
                self::logError('Error getting unique values: ' . $e->getMessage(), [
                    'table' => $table,
                    'column' => $columnName,
                    'sql_state' => $e->getCode()
                ]);
                
                // Try with simpler query as fallback
                try {
                    $fallbackSql = "SELECT DISTINCT `$columnName` FROM `$table` WHERE `$columnName` IS NOT NULL LIMIT 100";
                    $results = $db->queryAndFetchAllAssoc($fallbackSql);
                    $uniqueValues[$columnName] = array_column($results, $columnName);
                    
                    self::logDebug('Fallback query successful', [
                        'table' => $table,
                        'column' => $columnName,
                        'count' => count($uniqueValues[$columnName])
                    ]);
                    
                } catch (PDOException $fallbackError) {
                    self::logError('Fallback query also failed', [
                        'table' => $table,
                        'column' => $columnName,
                        'error' => $fallbackError->getMessage()
                    ]);
                    // Return empty array on complete failure
                    $uniqueValues[$columnName] = [];
                }
            }
        }
        
        return $uniqueValues;
    }

    // ==========================================
    // FILTERING & SECURITY METHODS
    // ==========================================

    /**
     * Sanitize search string to protect against SQL injection
     */
    static function sanitizeSearchString($string) 
    {
        // Remove special characters and keep letters, numbers and spaces
        return preg_replace('/[^\p{L}\p{N}\s]/u', '', $string);
    }

    /**
     * Check if string is DataTables regex pattern
     */
    static function isDataTablesRegexPattern($str)
    {
        return preg_match('/^\(.+\)$/', $str) === 1;
    }

    /**
     * Build regex condition for dropdown filters
     */
    static function buildRegexCondition($columnDb, $regexStr, &$bindings)
    {
        $cleanStr = trim($regexStr, '()');
        $values = strpos($cleanStr, '|') !== false ? explode('|', $cleanStr) : [$cleanStr];
        
        if (empty($values)) {
            return null;
        }
        
        $conditions = [];
        foreach ($values as $value) {
            $cleanValue = trim($value);
            if ($cleanValue !== '') {
                if (preg_match('/^(\d{4}-\d{2}-\d{2})\s+00:00:00$/', $cleanValue, $matches)) {
                    $dateValue = $matches[1];
                    $binding = self::bind($bindings, $dateValue . '%', PDO::PARAM_STR);
                    $conditions[] = "`{$columnDb}` LIKE {$binding}";
                } else {
                    $cleanValue = preg_replace('/[^\p{L}\p{N}\s\-_.]/u', '', $cleanValue);
                    $binding = self::bind($bindings, $cleanValue, PDO::PARAM_STR);
                    $conditions[] = "`{$columnDb}` = {$binding}";
                }
            }
        }
        
        if (empty($conditions)) {
            return null;
        }
        
        return count($conditions) === 1 ? $conditions[0] : '(' . implode(' OR ', $conditions) . ')';
    }

    /**
     * Bind parameter for SQL query
     */
    static function bind(&$a, $val, $type)
    {
        $key = ':binding_'.count($a);

        $a[] = array(
            'key' => $key,
            'val' => $val,
            'type' => $type
        );

        return $key;
    }

    /**
     * Add bindings from an array
     */
    static function add_bindings(&$a, $vals)
    {
        if (!is_array($vals) || !isset($vals['bindings']) || !is_array($vals['bindings'])) {
            return;
        }
        
        foreach($vals['bindings'] as $key => $value) {
            $a[] = array(
                'key' => $key,
                'val' => $value,
                'type' => PDO::PARAM_STR
            );
        }
    }

    /**
     * Pull a particular property from each assoc. array in a numeric array
     */
    static function pluck($a, $prop)
    {
        $out = array();

        for ($i = 0, $len = count($a); $i < $len; $i++) {
            if (empty($a[$i][$prop]) && $a[$i][$prop] !== 0) {
                continue;
            }

            //removing the $out array index confuses the filter method in doing proper binding,
            //adding it ensures that the array data are mapped correctly
            $out[$i] = $a[$i][$prop];
        }

        return $out;
    }

    // ==========================================
    // DATABASE CONNECTION METHODS
    // ==========================================

    /**
     * Connect to the database
     */
    static function sql_connect(array $sql_details): DatabaseConnection
    {
        try {
            // Create a new PDO connection
            $pdo = new PDO(
                "mysql:host={$sql_details['host']};dbname={$sql_details['db']}",
                $sql_details['user'],
                $sql_details['pass'],
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
                ]
            );
            
            // Create a DatabaseConnection instance from the PDO connection
            return \baseKRIZAN\Database\DatabaseConnection::createFromPDO($pdo);
        }
        catch (PDOException $e) {
            self::logError('Database connection error: ' . $e->getMessage(), [
                'host' => $sql_details['host'],
                'db' => $sql_details['db']
            ]);
            
            throw new \RuntimeException(
                "Database connection error: " . $e->getMessage()
            );
        }
    }

    /**
     * Execute an SQL query on the database
     */
    static function sql_exec(DatabaseConnection $db, $bindings, $sql = null): array
    {
        // Ensure $db is a DatabaseConnection object
        if (!$db instanceof DatabaseConnection) {
            throw new \InvalidArgumentException("Invalid database connection");
        }

        // Argument shifting
        if ($sql === null) {
            $sql = $bindings;
            $bindings = [];  // Initialize as empty array if no bindings provided
        }

        try {
            // Convert bindings to associative array that DatabaseConnection can use
            $params = [];
            if (is_array($bindings)) {
                foreach ($bindings as $binding) {
                    $params[$binding['key']] = $binding['val'];
                }
            }
            
            // For debug logging
            if (self::$logger && self::$logger->isDebugEnabled()) {
                self::logDebug('Executing SQL', [
                    'sql' => $sql,
                    'params' => $params
                ]);
            }

            // Use queryAndFetchAllAssoc instead of executeQuery/fetchAllAssoc
            $results = $db->queryAndFetchAllAssoc($sql, $params);
            
            // Convert results to FETCH_BOTH format expected by DatatablesSSP
            $convertedResults = [];
            foreach ($results as $row) {
                $newRow = [];
                $i = 0;
                foreach ($row as $key => $value) {
                    $newRow[$key] = $value; // associative key
                    $newRow[$i] = $value;   // numeric key
                    $i++;
                }
                $convertedResults[] = $newRow;
            }
            
            return $convertedResults;
        }
        catch (PDOException $e) {
            self::logError('SQL execution error: ' . $e->getMessage(), [
                'sql' => $sql,
                'error' => $e->getMessage()
            ]);
            
            throw new \RuntimeException("SQL execution error: " . $e->getMessage());
        }
    }
}