<?php

namespace Models;

use PDO;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * FCMTokenModel - Database operations for FCM tokens
 * 
 * Handles all database operations related to FCM tokens
 */
class FCMTokenModel extends DatabaseTable
{
    private Logger $logger;
    private string $userTokensTable = 'notifications_user_tokens';

    public function __construct(DatabaseConnection $dbConnection, Logger $logger)
    {
        parent::__construct($dbConnection, $this->userTokensTable, 'id');
        $this->logger = $logger;
        
        // Ensure tokens table exists
        $this->ensureUserTokensTableExists();
    }
    
    /**
     * Ensure the tokens table exists
     */
    private function ensureUserTokensTableExists(): void
    {
        try {
            // Validate table name for safety
            $this->validateTableName($this->userTokensTable);
            
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
                ['table' => $this->userTokensTable]
            );
            
            if ($tableExists === null) {
                // Create the table if it doesn't exist
                $sql = "CREATE TABLE `{$this->userTokensTable}` (
                    `id` INT(11) AUTO_INCREMENT PRIMARY KEY,
                    `user_id` INT(11) NOT NULL,
                    `user_email` VARCHAR(250) NOT NULL,
                    `device_token` TEXT NOT NULL,
                    `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    `device_type` VARCHAR(20) NULL DEFAULT 'unknown',
                    UNIQUE KEY `uk_user_device` (`user_id`, `device_token`(191)),
                    INDEX `idx_user_id` (`user_id`),
                    INDEX `idx_device_type` (`device_type`),
                    INDEX `idx_user_email` (`user_email`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                
                $this->dbConnection->execute($sql);
                
                $this->logger->notification('Created FCM user tokens table', [
                    'table' => $this->userTokensTable
                ]);
            }
        } catch (\PDOException $e) {
            $this->logger->error('Error ensuring FCM user tokens table exists', [
                'error' => $e->getMessage(),
                'table' => $this->userTokensTable
            ]);
            throw $e;
        }
    }
    
    /**
     * Validate table name to prevent SQL injection
     */
    private function validateTableName(string $tableName): void
    {
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $tableName)) {
            throw new \InvalidArgumentException("Invalid table name: {$tableName}");
        }
    }

    /**
     * Get user device token for FCM
     * 
     * @param int $userId User ID
     * @return string|null Device token or null if not found
     */
    public function getUserToken(int $userId): ?string
    {
        try {
            return $this->dbConnection->querySingleValue(
                "SELECT device_token FROM {$this->userTokensTable} WHERE user_id = :user_id LIMIT 1",
                ['user_id' => $userId]
            );
        } catch (\Exception $e) {
            $this->logger->error('Failed to get user FCM token', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return null;
        }
    }

    /**
     * Save user token for FCM
     * 
     * @param int $userId User ID
     * @param string $email User email
     * @param string $token Device token
     * @param string $deviceType Device type (mobile, desktop, etc.)
     * @return bool Success status
     */
    public function saveUserToken(int $userId, string $email, string $token, string $deviceType = 'unknown'): bool
    {
        try {
            // Prvo izbrišite postojeći token za istog korisnika i tip uređaja
            $deleteQuery = "DELETE FROM {$this->userTokensTable} 
                        WHERE user_id = :user_id AND device_type = :device_type";
            
            $this->dbConnection->execute($deleteQuery, [
                'user_id' => $userId,
                'device_type' => $deviceType
            ]);
            
            // Zatim dodajte novi token
            $insertQuery = "INSERT INTO {$this->userTokensTable} 
                        (user_id, user_email, device_token, device_type) 
                        VALUES (:user_id, :user_email, :device_token, :device_type)";

            $this->dbConnection->execute($insertQuery, [
                'user_id' => $userId,
                'user_email' => $email,
                'device_token' => $token,
                'device_type' => $deviceType
            ]);

            return true;
        } catch (\Exception $e) {
            $this->logger->error('Failed to save user FCM token', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return false;
        }
    }

    /**
     * Get all user tokens for FCM
     * 
     * @param int $userId User ID
     * @return array List of tokens
     */
    public function getAllUserTokens(int $userId): array
    {
        try {
            $query = "SELECT device_token, device_type FROM {$this->userTokensTable} WHERE user_id = :user_id";
            $stmt = $this->dbConnection->executeQuery($query, ['user_id' => $userId]);
            return $this->dbConnection->fetchAllAssoc($stmt);
        } catch (\Exception $e) {
            $this->logger->error('Failed to get all user FCM tokens', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return [];
        }
    }
    
    /**
     * Get device type for a user
     * 
     * @param int $userId User ID
     * @return string Device type or 'unknown' if not found
     */
    public function getUserDeviceType(int $userId): string
    {
        try {
            $deviceType = $this->dbConnection->querySingleValue(
                "SELECT device_type FROM {$this->userTokensTable} WHERE user_id = :user_id LIMIT 1",
                ['user_id' => $userId]
            );
            return $deviceType !== null ? $deviceType : 'unknown';
        } catch (\Exception $e) {
            $this->logger->error('Failed to get user device type', [
                'error' => $e->getMessage(),
                'user_id' => $userId
            ]);
            return 'unknown';
        }
    }

    /**
     * Remove invalid token from database
     * 
     * @param string $token Invalid token
     * @return bool Operation success
     */
    public function removeInvalidToken(string $token): bool
    {
        try {
            $query = "DELETE FROM {$this->userTokensTable} WHERE device_token = :token";
            $this->dbConnection->execute($query, ['token' => $token]);
            
            $this->logger->notification('Removed invalid FCM token', [
                'token_prefix' => substr($token, 0, 10) . '...'
            ]);
            
            return true;
        } catch (\Exception $e) {
            $this->logger->error('Error removing invalid FCM token', [
                'error' => $e->getMessage(),
                'token_prefix' => substr($token, 0, 10) . '...'
            ]);
            return false;
        }
    }
}