<?php

namespace Models;

use PDO;
use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * NotificationModel - Database operations for standard notifications
 * 
 * Handles database operations for core notification functionality
 * without external service dependencies.
 */
class NotificationModel extends DatabaseTable
{
    private Logger $logger;
    private string $notificationsTable = 'notifications';

    public function __construct(DatabaseConnection $dbConnection, Logger $logger)
    {
        parent::__construct($dbConnection, $this->notificationsTable, 'id');
        $this->logger = $logger;
        
        // Ensure notifications table exists
        $this->ensureNotificationsTableExists();
    }
    
    /**
     * Ensure the notifications table exists
     */
    private function ensureNotificationsTableExists(): void
    {
        try {
            // Validate table name for safety
            $this->validateTableName($this->notificationsTable);
            
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
                ['table' => $this->notificationsTable]
            );
            
            if ($tableExists === null) {
                // Create the table if it doesn't exist
                $sql = "CREATE TABLE `{$this->notificationsTable}` (
                    `id` INT(11) AUTO_INCREMENT PRIMARY KEY,
                    `parent_id` INT(11) NULL DEFAULT NULL,
                    `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                    `from_id` INT(11) NOT NULL,
                    `for_id` INT(11) NOT NULL,
                    `title` VARCHAR(255) NOT NULL,
                    `message` TEXT NOT NULL,
                    `tags` LONGTEXT NULL DEFAULT NULL,
                    `is_solved` TINYINT(1) NULL DEFAULT 0,
                    `type` VARCHAR(50) NOT NULL DEFAULT 'notification',
                    `route` VARCHAR(255) NULL DEFAULT NULL,
                    `related_id` INT(11) NULL DEFAULT NULL,
                    `is_sticky` TINYINT(1) NULL DEFAULT 0,
                    `is_read` TINYINT(1) NULL DEFAULT 0,
                    `send_push` TINYINT(1) NULL DEFAULT 0,
                    INDEX `idx_parent_id` (`parent_id`),
                    INDEX `idx_from_id` (`from_id`),
                    INDEX `idx_for_id` (`for_id`),
                    INDEX `idx_type` (`type`),
                    INDEX `idx_route` (`route`),
                    INDEX `idx_is_read` (`is_read`),
                    INDEX `idx_is_sticky` (`is_sticky`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                
                $this->dbConnection->execute($sql);
                
                $this->logger->notification('Created notifications table', [
                    'table' => $this->notificationsTable
                ]);
            }
        } catch (\PDOException $e) {
            $this->logger->error('Error ensuring notifications table exists', [
                'error' => $e->getMessage(),
                'table' => $this->notificationsTable
            ]);
            throw $e;
        }
    }
    
    /**
     * Validate table name to prevent SQL injection
     */
    private function validateTableName(string $tableName): void
    {
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $tableName)) {
            throw new \InvalidArgumentException("Invalid table name: {$tableName}");
        }
    }

    /**
     * Get user notifications with filter options
     * 
     * @param int $userId User ID
     * @param int $limit Maximum number of notifications to return
     * @param string $filter Filter type: 'default', 'unread', 'sticky', 'read', 'all'
     * @return array List of notifications
     */
    public function getUserNotifications(int $userId, int $limit = 10, string $filter = 'default'): array
    {
        try {
            // Cast limit to integer for safety
            $limit = (int)$limit;
            
            $query = "SELECT n.*, u.user_email as sender_email 
                    FROM {$this->notificationsTable} n
                    LEFT JOIN korisnici u ON n.from_id = u.id
                    WHERE n.for_id = :for_id";

            // Add filter based on parameter
            switch ($filter) {
                case 'unread':
                    $query .= " AND n.is_read = FALSE";
                    break;
                case 'sticky':
                    $query .= " AND n.is_sticky = TRUE";
                    break;
                case 'read':
                    $query .= " AND n.is_read = TRUE";
                    break;
                case 'default':
                    $query .= " AND (n.is_read = FALSE OR n.is_sticky = TRUE)";
                    break;
                // no filter for 'all' case
            }

            // Insert the limit directly into the query
            $query .= " ORDER BY n.is_sticky DESC, n.created_at DESC LIMIT {$limit}";

            $params = [
                'for_id' => $userId
                // Removed limit parameter
            ];
            
            return $this->dbConnection->queryAndFetchAllAssoc($query, $params);
        } catch (\Exception $e) {
            $this->logger->error('Failed to fetch notifications', [
                'error' => $e->getMessage(),
                'userId' => $userId
            ]);
            return [];
        }
    }

    /**
     * Get notification by ID
     * 
     * @param int $notificationId Notification ID
     * @return array|null Notification data or null if not found
     */
    public function getNotificationById(int $notificationId): ?array
    {
        try {
            $query = "SELECT n.*, u.user_email as sender_email 
                    FROM {$this->notificationsTable} n
                    LEFT JOIN korisnici u ON n.from_id = u.id
                    WHERE n.id = :id
                    LIMIT 1";

            $result = $this->dbConnection->queryAndFetchAssoc($query, ['id' => $notificationId]);
            return $result ?: null;
        } catch (\Exception $e) {
            $this->logger->error('Failed to get notification by ID', [
                'error' => $e->getMessage(),
                'id' => $notificationId
            ]);
            return null;
        }
    }

    /**
     * Update read status for a specific notification
     * 
     * @param int $notificationId Notification ID
     * @param bool $isRead Read status to set
     * @return bool Success status
     */
    public function updateReadStatus(int $notificationId, bool $isRead): bool
    {
        try {
            $query = "UPDATE {$this->notificationsTable} 
                    SET is_read = :is_read 
                    WHERE id = :id";

            $params = [
                'id' => $notificationId,
                'is_read' => $isRead ? 1 : 0
            ];
            
            $this->dbConnection->execute($query, $params);
            return true;
        } catch (\Exception $e) {
            $this->logger->error('Failed to update read status', [
                'error' => $e->getMessage(),
                'id' => $notificationId,
                'is_read' => $isRead
            ]);
            return false;
        }
    }

    /**
     * Mark all notifications as read for a user
     * 
     * @param int $userId User ID
     * @return bool Success status
     */
    public function markNotificationsAsRead(int $userId): bool
    {
        try {
            $query = "UPDATE {$this->notificationsTable} 
                    SET is_read = TRUE 
                    WHERE for_id = :for_id
                    AND is_sticky = FALSE";  // Don't mark sticky notifications as read

            $this->dbConnection->execute($query, ['for_id' => $userId]);
            return true;
        } catch (\Exception $e) {
            $this->logger->error('Failed to mark notifications as read', [
                'error' => $e->getMessage(),
                'userId' => $userId
            ]);
            return false;
        }
    }

    /**
     * Update sticky status for notification
     * 
     * @param int $notificationId Notification ID
     * @param bool $isSticky Sticky status
     * @return bool Success status
     */
    public function updateStickyStatus(int $notificationId, bool $isSticky): bool
    {
        try {
            $query = "UPDATE {$this->notificationsTable} 
                    SET is_sticky = :is_sticky 
                    WHERE id = :id";

            $params = [
                'id' => $notificationId,
                'is_sticky' => $isSticky ? 1 : 0
            ];
            
            $this->dbConnection->execute($query, $params);
            return true;
        } catch (\Exception $e) {
            $this->logger->error('Failed to update sticky status', [
                'error' => $e->getMessage(),
                'id' => $notificationId,
                'is_sticky' => $isSticky
            ]);
            return false;
        }
    }
}