<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Asset Bundler
 * 
 * Handles bundling of assets for production:
 * - Combines multiple CSS and JS files into single bundles
 * - Minifies bundle content
 * - Updates manifest with bundle information
 */
class AssetBundler
{
    private string $rootDir;
    private string $sourcePath;
    private string $outputPath;
    private bool $bundlingEnabled;
    private array $bundleConfig = [];
    
    // Logger property
    private ?Logger $logger = null;
    
    /**
     * Constructor
     */
    public function __construct(string $rootDir, ?Logger $logger = null)
    {
        // Use centralized path management
        $this->rootDir = AssetPathResolver::getRootPath();
        $this->logger = $logger;
        
        // Initialize paths using the centralized path resolver
        $pathResolver = AssetPathResolver::getInstance($logger);
        
        // Set the source path for assets
        $this->sourcePath = AssetPathResolver::getCoreAssetsSourcePath();
        
        // Set the output path for bundled assets
        $this->outputPath = AssetPathResolver::getCoreAssetsCachePath();
        
        // Get bundling setting
        $this->bundlingEnabled = \baseKRIZAN\Config\Config::get('asset.bundling_enabled');
        
        // Load bundle configuration
        $this->loadBundleConfig();
        
        if ($logger) {
            $logger->assets("AssetBundler initialized - bundlingEnabled=" . ($this->bundlingEnabled ? 'true' : 'false'));
        }
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }
    
    /**
     * Load bundle configuration
     */
    private function loadBundleConfig(): void
    {
        $configPath = AssetPathResolver::getRootPath() . '/app/config/app_asset_bundle.php';
        
        if (file_exists($configPath)) {
            $config = include $configPath;
            if (is_array($config)) {
                $this->bundleConfig = $config;
                $this->log('info', "Loaded bundle config from: $configPath");
                return;
            }
        }
        
        // If config doesn't exist, create empty configuration and default file
        $this->bundleConfig = [];
        $this->generateDefaultConfigFile($configPath);
        $this->log('warning', "No valid bundle configuration found. Created empty config at: $configPath");
    }
    
    /**
     * Generate default configuration file template
     */
    private function generateDefaultConfigFile(string $configPath): void
    {
        // Ensure directory exists
        if (!is_dir(dirname($configPath))) {
            mkdir(dirname($configPath), 0755, true);
        }
        
        $content = "<?php\n\n";
        $content .= "/**\n";
        $content .= " * Asset Bundle Configuration\n";
        $content .= " * Define bundles of CSS and JS files to be combined in production\n";
        $content .= " */\n\n";
        $content .= "return [\n";
        $content .= "    'css' => [\n";
        $content .= "        'core' => [\n";
        $content .= "            'output' => 'css/core.bundle.css',\n";
        $content .= "            'files' => [\n";
        $content .= "                'default/css/bootstrap.min.css',\n";
        $content .= "                'default/css/app_theme.css',\n";
        $content .= "                'default/css/app_content.css',\n";
        $content .= "                'default/css/app_layout.css',\n";
        $content .= "                'default/css/app_notifications.css'\n";
        $content .= "            ]\n";
        $content .= "        ],\n";
        $content .= "        'plugins' => [\n";
        $content .= "            'output' => 'css/plugins.bundle.css',\n";
        $content .= "            'files' => [\n";
        $content .= "                'select2/css/select2.min.css',\n";
        $content .= "                'select2/css/select2fix.css',\n";
        $content .= "                'flatpickr/css/flatpickr.min.css',\n";
        $content .= "                'flatpickr/css/dark.css',\n";
        $content .= "                'flatpickr/css/flatpickrfix.css',\n";
        $content .= "                'datatables/css/datatables.min.css',\n";
        $content .= "                'datetime/css/tempusdominus-bootstrap-4.min.css',\n";
        $content .= "                'multidatetime/css/multi_tempusdominus-bootstrap-4.min.css'\n";
        $content .= "            ]\n";
        $content .= "        ]\n";
        $content .= "    ],\n";
        $content .= "    'js' => [\n";
        $content .= "        'core' => [\n";
        $content .= "            'output' => 'js/core.bundle.js',\n";
        $content .= "            'files' => [\n";
        $content .= "                'default/js/jquery-3.7.1.min.js',\n";
        $content .= "                'default/js/app_csrf-jquery.js',\n";
        $content .= "                'default/js/app_csrf-fetch.js',\n";
        $content .= "                'default/js/popper.min.js',\n";
        $content .= "                'default/js/bootstrap.min.js',\n";
        $content .= "                'default/js/app_theme.js',\n";
        $content .= "                'default/js/app_menu-toggle.js'\n";
        $content .= "            ]\n";
        $content .= "        ],\n";
        $content .= "        'plugins' => [\n";
        $content .= "            'output' => 'js/plugins.bundle.js',\n";
        $content .= "            'files' => [\n";
        $content .= "                'select2/js/select2.min.js',\n";
        $content .= "                'flatpickr/js/flatpickr.js',\n";
        $content .= "                'flatpickr/js/hr.js',\n";
        $content .= "                'datatables/js/datatables.min.js',\n";
        $content .= "                'datatables/js/datetime-moment.js',\n";
        $content .= "                'datetime/js/moment.min.js',\n";
        $content .= "                'datetime/js/tempusdominus-bootstrap-4.min.js',\n";
        $content .= "                'multidatetime/js/multi_moment.min.js',\n";
        $content .= "                'multidatetime/js/multi_tempusdominus-bootstrap-4.min.js'\n";
        $content .= "            ]\n";
        $content .= "        ],\n";
        $content .= "        'firebase' => [\n";
        $content .= "            'output' => 'js/firebase.bundle.js',\n";
        $content .= "            'files' => [\n";
        $content .= "                'firebase-app-compat961.js',\n";
        $content .= "                'firebase-messaging-compat961.js',\n";
        $content .= "                'firebase-init.js'\n";
        $content .= "            ]\n";
        $content .= "        ]\n";
        $content .= "    ]\n";
        $content .= "    ]\n";
        $content .= "];\n";
        
        // Write the file
        file_put_contents($configPath, $content);
    }
    
    /**
     * Generate bundled assets for production
     */
    public function generateBundles(array &$manifest, bool $forceMinify = true): array
    {
        if (!$this->bundlingEnabled && !$forceMinify) {
            $this->log('info', 'Bundling is disabled, skipping bundle generation');
            return ['bundles' => 0, 'size_before' => 0, 'size_after' => 0];
        }
        
        $stats = [
            'bundles' => 0,
            'size_before' => 0,
            'size_after' => 0
        ];
        
        // Initialize bundles in manifest if not exists
        if (!isset($manifest['bundles'])) {
            $manifest['bundles'] = [
                'css' => [],
                'js' => []
            ];
        }
        
        // Process CSS bundles
        if (isset($this->bundleConfig['css'])) {
            foreach ($this->bundleConfig['css'] as $name => $bundle) {
                $this->log('info', "Generating CSS bundle: $name");
                $bundleStats = $this->generateBundle('css', $name, $bundle, $forceMinify, $manifest);
                
                $stats['bundles']++;
                $stats['size_before'] += $bundleStats['size_before'];
                $stats['size_after'] += $bundleStats['size_after'];
            }
        }
        
        // Process JS bundles
        if (isset($this->bundleConfig['js'])) {
            foreach ($this->bundleConfig['js'] as $name => $bundle) {
                $this->log('info', "Generating JS bundle: $name");
                $bundleStats = $this->generateBundle('js', $name, $bundle, $forceMinify, $manifest);
                
                $stats['bundles']++;
                $stats['size_before'] += $bundleStats['size_before'];
                $stats['size_after'] += $bundleStats['size_after'];
            }
        }
        
        return $stats;
    }
    
    /**
     * Generate a single bundle
     */
    private function generateBundle(
        string $type,
        string $name,
        array $bundle,
        bool $forceMinify,
        array &$manifest
    ): array {
        $stats = [
            'size_before' => 0,
            'size_after' => 0
        ];
        
        // Skip if no files defined
        if (empty($bundle['files']) || empty($bundle['output'])) {
            $this->log('warning', "Bundle $name has no files or output defined");
            return $stats;
        }
        
        // Initialize path resolver for consistent path handling
        $pathResolver = AssetPathResolver::getInstance($this->logger);
        
        // Ensure output directory exists
        $outputDir = dirname($this->outputPath . '/' . $bundle['output']);
        $pathResolver->ensureDirectoryExists($outputDir);
        
        // Combine files
        $combined = '';
        foreach ($bundle['files'] as $file) {
            $filePath = $this->sourcePath . '/' . $file;
            
            if (file_exists($filePath)) {
                $content = file_get_contents($filePath);
                $stats['size_before'] += strlen($content);
                
                // Add comment for file path
                $combined .= "/* $file */\n";
                $combined .= $content . "\n";
            } else {
                $this->log('warning', "Bundle file not found: $filePath");
            }
        }
        
        // Skip if no content
        if (empty($combined)) {
            $this->log('warning', "Bundle $name has no content");
            return $stats;
        }
        
        // Process content using AssetCompiler for minification
        $compiler = new AssetCompiler(AssetPathResolver::getRootPath(), false, $this->logger); // Production mode
        $compiler->setMinifyEnabled($forceMinify);
        
        if ($forceMinify) {
            $combined = $compiler->minifyContent($combined, $type);
        }
        
        // Generate hash
        $hash = substr(md5($combined), 0, 8);
        $outputFileName = $pathResolver->getHashedFileName($bundle['output'], $hash);
        $outputPath = $this->outputPath . '/' . $outputFileName;
        
        // Write bundle
        file_put_contents($outputPath, $combined);
        $stats['size_after'] = strlen($combined);
        
        // Get the public path appropriate for the bundle type
        $publicPath = AssetPathResolver::getCoreAssetsPublicPath();
        
        // Add to manifest as special bundle entry
        $manifest['bundles'][$type][$name] = $publicPath . '/' . $outputFileName;
        
        $reduction = $stats['size_before'] > 0 ? 
            round(($stats['size_before'] - $stats['size_after']) / $stats['size_before'] * 100, 2) : 0;
        $this->log('info', "Bundle $name: {$stats['size_before']} bytes → {$stats['size_after']} bytes ($reduction% reduction)");
        
        return $stats;
    }
    
    /**
     * Get bundle configuration
     */
    public function getBundleConfig(): array
    {
        return $this->bundleConfig;
    }
    
    /**
     * Check if bundling is enabled
     */
    public function isBundlingEnabled(): bool
    {
        return $this->bundlingEnabled;
    }
    
    /**
     * Set bundling enabled/disabled
     */
    public function setBundlingEnabled(bool $enabled): void
    {
        $this->bundlingEnabled = $enabled;
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}