<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Asset Compiler
 * 
 * Handles compilation, minification, and optimization of assets:
 * - Compiles single assets on demand
 * - Processes CSS URL replacements
 * - Minifies code for production
 * - Generates source maps
 */
class AssetCompiler
{
    private string $rootDir;
    private bool $developmentMode;
    private bool $minifyEnabled;
    private bool $sourceMapsEnabled;
    
    // Logger property
    private ?Logger $logger = null;
    
    /**
     * Constructor
     */
    public function __construct(string $rootDir, bool $developmentMode, ?Logger $logger = null)
    {
        // Use AssetPathResolver's centralized path
        $this->rootDir = AssetPathResolver::getRootPath();
        $this->developmentMode = $developmentMode;
        $this->logger = $logger;
        
        // Initialize settings from config
        $this->minifyEnabled = \baseKRIZAN\Config\Config::get('asset.bundling_enabled');
        $this->sourceMapsEnabled = \baseKRIZAN\Config\Config::get('asset.sourcemaps_enabled');
        
        if ($logger) {
            $logger->assets("AssetCompiler initialized - minify=" . ($this->minifyEnabled ? 'true' : 'false') . 
                            ", sourceMaps=" . ($this->sourceMapsEnabled ? 'true' : 'false'));
        }
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }
    
    /**
     * Check if an asset needs to be compiled
     */
    public function shouldCompileAsset(string $sourcePath, string $cachePath): bool
    {
        // If cache doesn't exist, we need to compile
        if (!file_exists($cachePath)) {
            return true;
        }
        
        // If source is newer than cache, we need to recompile
        return filemtime($sourcePath) > filemtime($cachePath);
    }
    
    /**
     * Compile a single asset
     */
    public function compileAsset(string $sourcePath, string $cachePath, string $ext): void
    {
        try {
            // Read source content
            $content = file_get_contents($sourcePath);
            if ($content === false) {
                $this->log('error', "Failed to read source file: $sourcePath");
                return;
            }
            
            $type = $this->getAssetType($ext);
            
            // Minify if enabled
            if ($this->minifyEnabled && in_array($type, ['css', 'js'])) {
                $content = $this->minifyContent($content, $type);
            }
            
            // Add source maps if enabled
            if ($this->sourceMapsEnabled && in_array($type, ['css', 'js'])) {
                $content = $this->addSourceMapUrl($content, $type, $sourcePath, $cachePath);
            }
            
            // Write to cache
            file_put_contents($cachePath, $content);
            
            $this->log('debug', 'Compiled asset: ' . basename($sourcePath) . ' -> ' . basename($cachePath));
        } catch (\Exception $e) {
            $this->log('error', "Failed to compile asset $sourcePath: " . $e->getMessage());
            // Create a minimal fallback file to prevent repeated compilation attempts
            $this->createFallbackAsset($cachePath, $ext);
        }
    }
    
    /**
     * Compile asset with pre-processed content
     */
    public function compileAssetWithContent(string $sourcePath, string $cachePath, string $ext, string $content): void
    {
        try {
            $type = $this->getAssetType($ext);
            
            // Minify if enabled
            if ($this->minifyEnabled && in_array($type, ['css', 'js'])) {
                $content = $this->minifyContent($content, $type);
            }
            
            // Add source maps if enabled
            if ($this->sourceMapsEnabled && in_array($type, ['css', 'js'])) {
                $content = $this->addSourceMapUrl($content, $type, $sourcePath, $cachePath);
            }
            
            // Write to cache
            file_put_contents($cachePath, $content);
            
            $this->log('debug', 'Compiled asset with pre-processed content: ' . basename($sourcePath) . ' -> ' . basename($cachePath));
        } catch (\Exception $e) {
            $this->log('error', "Failed to compile asset with content $sourcePath: " . $e->getMessage());
            // Create a minimal fallback file to prevent repeated compilation attempts
            $this->createFallbackAsset($cachePath, $ext);
        }
    }
    
    /**
     * Minify content based on type
     */
    public function minifyContent(string $content, string $type): string
    {
        switch ($type) {
            case 'css':
                // Basic CSS minification
                $content = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $content);
                return str_replace(["\r\n", "\r", "\n", "\t", '  ', '    '], '', $content);
            
            case 'js':
                // Basic JS minification
                $content = preg_replace('/(?:(?:\/\*(?:[^*]|(?:\*+[^*\/]))*\*+\/)|(?:(?<!\:|\\\)\/\/[^"\'].*))/', '', $content);
                return str_replace(["\r\n", "\r", "\n", "\t", '  ', '    '], '', $content);
            
            default:
                return $content;
        }
    }
    
    /**
     * Add source map URL to content
     */
    public function addSourceMapUrl(string $content, string $type, string $sourcePath, string $cachePath): string
    {
        $sourceMapPath = $cachePath . '.map';
        $sourceMapUrl = basename($cachePath) . '.map';
        
        // Create basic source map
        $sourceMap = [
            'version' => 3,
            'file' => basename($cachePath),
            'sources' => [basename($sourcePath)],
            'sourcesContent' => [$content],
            'mappings' => 'AAAA' // Very basic mapping
        ];
        
        // Write source map file
        file_put_contents($sourceMapPath, json_encode($sourceMap));
        
        // Add reference at the end of file
        if ($type === 'css') {
            return $content . "\n/*# sourceMappingURL=" . $sourceMapUrl . " */";
        } elseif ($type === 'js') {
            return $content . "\n//# sourceMappingURL=" . $sourceMapUrl;
        }
        
        return $content;
    }
    
    /**
     * Create fallback asset in case of compilation error
     */
    private function createFallbackAsset(string $cachePath, string $ext): void
    {
        // Create a minimal valid file based on extension
        $content = '';
        
        if ($ext === 'css') {
            $content = '/* Compilation error occurred */';
        } elseif ($ext === 'js') {
            $content = '/* Compilation error occurred */';
        }
        
        file_put_contents($cachePath, $content);
        $this->log('debug', "Created fallback asset: " . basename($cachePath));
    }
    
    /**
     * Get the asset type based on file extension
     */
    private function getAssetType(string $ext): string
    {
        $fileTypes = AssetPathResolver::getFileTypesConfig();
        
        foreach ($fileTypes as $type => $extensions) {
            if (in_array(strtolower($ext), $extensions)) {
                return $type;
            }
        }
        
        return 'other';
    }
    
    /**
     * Get/set minify setting
     */
    public function isMinifyEnabled(): bool
    {
        return $this->minifyEnabled;
    }
    
    public function setMinifyEnabled(bool $enabled): void
    {
        $this->minifyEnabled = $enabled;
    }
    
    /**
     * Get/set source maps setting
     */
    public function isSourceMapsEnabled(): bool
    {
        return $this->sourceMapsEnabled;
    }
    
    public function setSourceMapsEnabled(bool $enabled): void
    {
        $this->sourceMapsEnabled = $enabled;
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}