<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Assets\Strategy\AssetStrategyInterface;
use baseKRIZAN\Assets\Strategy\CoreAssetStrategy;
use baseKRIZAN\Assets\Strategy\SrcAssetStrategy;
use baseKRIZAN\Assets\Strategy\ModuleAssetStrategy;
use baseKRIZAN\Cache\CacheManager;
use baseKRIZAN\Cache\CacheInterface;

/**
 * Enhanced Asset Management System - Uses main CacheManager directly
 */
class AssetManager 
{
    private static ?self $instance = null;
    
    // Base components
    private AssetPathResolver $pathResolver;
    private AssetCompiler $compiler;
    private AssetManifestManager $manifestManager;
    private ?AssetBundler $bundler = null;
    private ?AssetPipeline $pipeline = null;
    
    // Strategy components
    private array $strategies = [];
    
    // Base configuration
    private string $rootDir;
    private bool $developmentMode;
    private bool $bundlingEnabled;
    private bool $cdnEnabled;
    private string $cdnUrl;
    private bool $pipelineEnabled;
    
    // Cache system - direktno koristi glavni CacheInterface
    private CacheInterface $cache;
    
    // Prefiks za ključeve za asset cache
    private string $cachePrefix = 'asset:';
    
    // Performance monitoring
    private array $performanceStats = [
        'requests' => 0,
        'cache_hits' => 0,
        'processing_time' => 0
    ];
    
    // Cache for already processed files
    private array $processedFiles = [];
    
    // Logger property
    private ?Logger $logger = null;
    
    /**
     * Get singleton instance
     */
    public static function getInstance(?Logger $logger = null): self
    {
        if (self::$instance === null) {
            self::$instance = new self($logger);
        } else if ($logger !== null && self::$instance->logger === null) {
            // Update logger if instance exists but doesn't have logger
            self::$instance->setLogger($logger);
        }
        return self::$instance;
    }
    
    /**
     * Private constructor for singleton pattern
     */
    private function __construct(?Logger $logger = null)
    {
        // Use centralized paths from AssetPathResolver
        $this->rootDir = AssetPathResolver::getRootPath();
        $this->logger = $logger;
        
        // Initialize configuration
        $this->developmentMode = \baseKRIZAN\Config\Config::get('environment') === 'development';
        $this->bundlingEnabled = \baseKRIZAN\Config\Config::get('asset.bundling_enabled');
        $this->cdnEnabled = \baseKRIZAN\Config\Config::get('asset.cdn_enabled');
        $this->cdnUrl = \baseKRIZAN\Config\Config::get('asset.cdn_url');
        $this->pipelineEnabled = \baseKRIZAN\Config\Config::get('asset.pipeline_enabled', false);
        
        // Initialize base components
        $this->pathResolver = AssetPathResolver::getInstance($logger);
        $this->compiler = new AssetCompiler($this->rootDir, $this->developmentMode, $logger);
        
        // Use getInstance instead of direct instantiation for AssetManifestManager
        $this->manifestManager = AssetManifestManager::getInstance($logger);
        
        // Initialize bundler only if bundling is enabled
        if ($this->bundlingEnabled) {
            $this->bundler = new AssetBundler($this->rootDir, $logger);
        }
        
        // Initialize pipeline if enabled
        if ($this->pipelineEnabled) {
            $this->pipeline = new AssetPipeline($this->rootDir, $logger);
        }
        
        // Initialize cache system - direktno koristimo CacheManager
        $this->initializeCache();
        
        // Initialize strategies
        $this->initializeStrategies();
        
        // Load manifest in production mode
        if (!$this->developmentMode) {
            $this->manifestManager->loadManifest();
        }
        
        // Set logger for ResourceLoader
        if (class_exists('\\baseKRIZAN\\Assets\\ResourceLoader')) {
            \baseKRIZAN\Assets\ResourceLoader::setLogger($logger);
        }
        
        if ($logger) {
            $logger->assets('AssetManager initialized - mode: ' . ($this->developmentMode ? 'development' : 'production'));
        }
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
        
        // Pass logger to components
        if ($logger) {
            $this->pathResolver->setLogger($logger);
            $this->compiler->setLogger($logger);
            $this->manifestManager->setLogger($logger);
            
            if ($this->bundler) {
                $this->bundler->setLogger($logger);
            }
            
            if ($this->pipeline) {
                $this->pipeline->setLogger($logger);
            }
            
            // Pass logger to all strategies
            foreach ($this->strategies as $strategy) {
                $strategy->setLogger($logger);
            }
            
            // Set logger for ResourceLoader
            if (class_exists('\\baseKRIZAN\\Assets\\ResourceLoader')) {
                \baseKRIZAN\Assets\ResourceLoader::setLogger($logger);
            }
        }
    }
    
    /**
     * Initialize the cache system
     */
    private function initializeCache(): void
    {
        // Dohvati instance CacheManager-a
        $this->cache = CacheManager::getInstance($this->logger)->driver();
        
        if ($this->logger) {
            $this->logger->assets('AssetManager using main cache system');
        }
    }
    
    /**
     * Set a different cache system
     */
    public function setCacheSystem(CacheInterface $cache): void
    {
        $this->cache = $cache;
    }
    
    /**
     * Initialize asset strategies
     */
    private function initializeStrategies(): void
    {
        // Create strategies
        $this->strategies[] = new CoreAssetStrategy(
            $this->pathResolver,
            $this->compiler,
            $this->logger
        );
        
        $this->strategies[] = new SrcAssetStrategy(
            $this->pathResolver,
            $this->compiler,
            $this->logger
        );
        
        $this->strategies[] = new ModuleAssetStrategy(
            $this->pathResolver,
            $this->compiler,
            $this->logger
        );
        
        $this->log('info', 'Initialized ' . count($this->strategies) . ' asset strategies');
    }
    
    /**
     * Get appropriate strategy for an asset using improved detection
     */
    private function getStrategy(string $file, ?string $module = null): ?AssetStrategyInterface
    {
        // Use a cache key for strategy lookups
        $cacheKey = $this->cachePrefix . 'strategy_' . md5($file . ($module ?? 'none'));
        
        // Check if we have cached the strategy
        $cachedStrategy = $this->cache->get($cacheKey);
        if ($cachedStrategy !== null) {
            foreach ($this->strategies as $strategy) {
                if (get_class($strategy) === $cachedStrategy) {
                    return $strategy;
                }
            }
        }
        
        // Find the right strategy
        foreach ($this->strategies as $strategy) {
            if ($strategy->canHandle($file, $module)) {
                // Cache the strategy class name for future lookups
                $this->cache->set($cacheKey, get_class($strategy), 3600); // 1 hour cache
                return $strategy;
            }
        }
        
        $this->log('warning', "No strategy found for file: $file, module: " . ($module ?? 'none'));
        return null;
    }
    
    /**
     * Get asset path - main public method
     * 
     * @param string $file The asset file path
     * @param string|null $module Optional module name
     * @param array $options Additional options for asset processing
     * @return string The public URL for the asset
     */
    public static function getAssetPath(string $file, ?string $module = null, array $options = []): string
    {
        $instance = self::getInstance();
        $startTime = microtime(true);
        
        // Increment request counter
        $instance->performanceStats['requests']++;
        
        $result = $instance->processAssetPath($file, $module, $options);
        
        // Calculate and record processing time
        $processingTime = microtime(true) - $startTime;
        $instance->performanceStats['processing_time'] += $processingTime;
        
        if ($instance->logger) {
            $instance->logger->assets("Asset path resolved: $file with module " . ($module ?? 'none') . " => $result (in " . round($processingTime * 1000, 2) . "ms)");
        }
        
        return $result;
    }
    
    /**
     * Process an asset path with enhanced caching
     */
    public function processAssetPath(string $file, ?string $module = null, array $options = []): string
    {
        // Debug log to trace the calls
        $this->log('debug', "Processing asset path: $file, module: " . ($module ?? 'none'));
        
        // Skip processing if already an absolute path or URL
        if (strpos($file, 'http://') === 0 || strpos($file, 'https://') === 0) {
            return $file;
        }
        
        // For absolute paths or fully specified resource paths, return as is
        if ($file[0] === '/') {
            return $this->pathResolver->resolveAbsolutePath($file);
        }

        // Handle special paths (appassets/, moduliassets/, srcassets/)
        $specialPath = $this->pathResolver->handleSpecialPaths($file, $module);
        if ($specialPath) {
            return $specialPath;
        }
        
        // Cache key for this file including options hash
        $optionsHash = !empty($options) ? md5(json_encode($options)) : '';
        $cacheKey = $this->cachePrefix . 'path_' . md5(($module ? "module-$module:" : '') . $file . ($optionsHash ? ":$optionsHash" : ''));
        
        // Check if we've already processed this file in this request
        if (isset($this->processedFiles[$cacheKey])) {
            $this->performanceStats['cache_hits']++;
            return $this->processedFiles[$cacheKey];
        }
        
        // Check if it's in the persistent cache
        $cachedResult = $this->cache->get($cacheKey);
        if ($cachedResult !== null && !$this->developmentMode) {
            $this->processedFiles[$cacheKey] = $cachedResult;
            $this->performanceStats['cache_hits']++;
            return $cachedResult;
        }
        
        // Get the appropriate strategy for this asset
        $strategy = $this->getStrategy($file, $module);
        
        if (!$strategy) {
            // If no strategy found, return a default path
            $defaultPath = $this->pathResolver->getDefaultAssetPath($file, $module);
            $this->processedFiles[$cacheKey] = $defaultPath;
            return $defaultPath;
        }
        
        // Process the asset using the selected strategy
        $result = $strategy->processAssetPath($file, $module);
        
        // If pipeline is enabled, pass through the pipeline
        if ($this->pipelineEnabled && $this->pipeline) {
            $ext = pathinfo($file, PATHINFO_EXTENSION);
            if (in_array($ext, ['css', 'js'])) {
                $result = $this->pipeline->process($result, $ext, $options);
            }
        }
        
        // Cache the result for this request
        $this->processedFiles[$cacheKey] = $result;
        
        // Cache the result persistently in non-development mode
        if (!$this->developmentMode) {
            $this->cache->set($cacheKey, $result, 3600); // 1 hour cache
        }
        
        return $result;
    }

    /**
     * Compile all assets for production with enhanced versioning
     */
    public function compileAll(bool $forceMinify = true, array $options = []): array
    {
        $this->log('info', 'Starting asset compilation for production...');
        
        $startTime = microtime(true);
        
        // Set compiler settings for production
        $tempMinify = $this->compiler->isMinifyEnabled();
        $this->compiler->setMinifyEnabled($forceMinify);
        
        // Create manifest structure with version information
        $manifest = $this->manifestManager->initializeManifest();
        
        // Add version information to the manifest
        $version = $options['version'] ?? date('YmdHis');
        $manifest['meta']['version'] = $version;
        $manifest['meta']['options'] = $options;
        
        // Process core assets
        $this->log('info', 'Compiling core assets...');
        $coreStrategy = $this->getStrategy('core.css'); // Any core file to identify strategy
        $coreStats = $coreStrategy->compileForProduction($manifest);
        
        // Process src assets
        $this->log('info', 'Compiling src assets...');
        $srcStrategy = $this->getStrategy('srcassets/dummy.css');
        $srcStats = $srcStrategy->compileForProduction($manifest);
        
        // Process module assets
        $this->log('info', 'Compiling module assets...');
        $moduleStrategy = $this->getStrategy('dummy.css', 'dummy-module');
        $moduleStats = $moduleStrategy->compileForProduction($manifest);
        
        // Generate bundles if enabled
        $bundleStats = [];
        if ($this->bundlingEnabled && $this->bundler) {
            $this->log('info', 'Generating asset bundles...');
            $bundleStats = $this->bundler->generateBundles($manifest, $forceMinify);
        }
        
        // Run the asset pipeline if enabled
        if ($this->pipelineEnabled && $this->pipeline) {
            $this->log('info', 'Running assets through pipeline...');
            $pipelineStats = $this->pipeline->processAll($manifest, $options);
            
            // Merge pipeline stats with overall stats
            $bundleStats['pipeline_processed'] = $pipelineStats['processed'] ?? 0;
            $bundleStats['pipeline_size_before'] = $pipelineStats['size_before'] ?? 0;
            $bundleStats['pipeline_size_after'] = $pipelineStats['size_after'] ?? 0;
        }
        
        // Save manifest
        $this->manifestManager->saveManifest($manifest);
        
        // Reset minify setting
        $this->compiler->setMinifyEnabled($tempMinify);
        
        $duration = round(microtime(true) - $startTime, 2);
        $this->log('info', "Asset compilation complete in {$duration}s");
        
        return [
            'success' => true,
            'duration' => $duration,
            'version' => $version,
            'core' => $coreStats,
            'src' => $srcStats,
            'modules' => $moduleStats,
            'bundles' => $bundleStats
        ];
    }
    
    /**
     * Get performance statistics
     */
    public function getPerformanceStats(): array
    {
        $stats = $this->performanceStats;
        
        // Add cache efficiency percentage
        if ($stats['requests'] > 0) {
            $stats['cache_efficiency'] = round(($stats['cache_hits'] / $stats['requests']) * 100, 2);
            $stats['avg_processing_time'] = $stats['requests'] > 0 ? 
                round(($stats['processing_time'] / $stats['requests']) * 1000, 2) : 0; // in ms
        } else {
            $stats['cache_efficiency'] = 0;
            $stats['avg_processing_time'] = 0;
        }
        
        return $stats;
    }
    
    /**
     * Clear the asset cache
     */
    public function clearCache(): void
    {
        // Clear filesystem caches for backward compatibility
        $cachePath = AssetPathResolver::getCoreAssetsCachePath();
        $this->removeDirectory($cachePath);
        $this->pathResolver->ensureDirectoryExists($cachePath);
        
        $srcCachePath = AssetPathResolver::getSrcAssetsCachePath();
        $this->removeDirectory($srcCachePath);
        $this->pathResolver->ensureDirectoryExists($srcCachePath);
        
        $moduleCachePath = AssetPathResolver::getModuleAssetsCachePath();
        $this->removeDirectory($moduleCachePath);
        $this->pathResolver->ensureDirectoryExists($moduleCachePath);
        
        // Clear the memory cache
        $this->processedFiles = [];
        
        // Očisti samo asset: prefixed ključeve u glavnom cache-u
        // Međutim, ovo nije potpuno moguće sa standardnim CacheInterface
        // Zato čistimo cijeli cache (manje optimalno, ali jednostavnije)
        $this->cache->clear();
        
        $this->log('info', 'Asset cache cleared');
    }
    
    /**
     * Remove a directory and all its contents
     */
    private function removeDirectory(string $dir): void
    {
        if (!is_dir($dir)) {
            return;
        }
        
        $objects = scandir($dir);
        foreach ($objects as $object) {
            if ($object === '.' || $object === '..') continue;
            
            $path = $dir . '/' . $object;
            
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }
        
        rmdir($dir);
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}