<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Enhanced Asset Manifest Manager
 * 
 * Handles the asset manifest for production with improved versioning:
 * - Supports multiple manifest versions
 * - Provides rollback capabilities
 * - Manages manifests with detailed metadata
 * - Tracks asset sizes and optimizations
 */
class AssetManifestManager
{
    private static ?self $instance = null;
    
    private string $rootDir;
    private string $manifestPath;
    private string $manifestsDir;
    private array $manifestCache = [];
    private ?string $currentVersion = null;
    
    // Logger property
    private ?Logger $logger = null;
    
    /**
     * Get singleton instance
     */
    public static function getInstance(?Logger $logger = null): self
    {
        if (self::$instance === null) {
            self::$instance = new self(AssetPathResolver::getRootPath(), $logger);
        } else if ($logger !== null && self::$instance->logger === null) {
            // Update logger if instance exists but doesn't have logger
            self::$instance->setLogger($logger);
        }
        return self::$instance;
    }
    
    /**
     * Constructor - Now private for singleton pattern
     */
    private function __construct(string $rootDir, ?Logger $logger = null)
    {
        // Use centralized path management
        $this->rootDir = $rootDir;
        $this->logger = $logger;
        
        // Define manifest file paths
        $this->manifestPath = AssetPathResolver::getCoreAssetsCachePath() . '/manifest.json';
        $this->manifestsDir = AssetPathResolver::getCoreAssetsCachePath() . '/manifests';
        
        // Ensure the manifests directory exists
        if (!is_dir($this->manifestsDir)) {
            mkdir($this->manifestsDir, 0755, true);
        }
        
        if ($logger) {
            $logger->assets("Enhanced AssetManifestManager initialized with path: {$this->manifestPath}");
        }
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }
    
    /**
     * Load manifest from file
     * 
     * @param string|null $version Specific version to load (null for latest)
     */
    public function loadManifest(?string $version = null): void
    {
        // If specific version is requested
        if ($version !== null) {
            $versionPath = $this->getVersionPath($version);
            
            if (file_exists($versionPath)) {
                $this->manifestCache = json_decode(file_get_contents($versionPath), true) ?: [];
                $this->currentVersion = $version;
                $this->log('info', "Loaded asset manifest version $version with " . count($this->manifestCache) . " entries");
                return;
            }
            
            $this->log('warning', "Asset manifest version $version not found, falling back to latest");
        }
        
        // Load the main manifest (latest)
        if (file_exists($this->manifestPath)) {
            $this->manifestCache = json_decode(file_get_contents($this->manifestPath), true) ?: [];
            $this->currentVersion = $this->manifestCache['meta']['version'] ?? date('YmdHis');
            $this->log('info', "Loaded asset manifest with " . count($this->manifestCache) . " entries, version " . $this->currentVersion);
        } else {
            $this->log('warning', "Asset manifest file not found: " . $this->manifestPath);
            $this->manifestCache = [];
            $this->currentVersion = null;
        }
    }
    
    /**
     * Get asset path from manifest
     */
    public function getAssetPathFromManifest(string $file, ?string $module = null): ?string
    {
        // If manifest is empty, load it
        if (empty($this->manifestCache)) {
            $this->loadManifest();
        }
        
        // Get asset type based on extension
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $type = AssetPathResolver::getInstance()->getAssetType($ext);
        
        // Special handling for srcassets
        if (strpos($file, 'srcassets/') === 0) {
            $relativePath = substr($file, 10); // Remove 'srcassets/' prefix
            if (isset($this->manifestCache['src'][$type][$relativePath])) {
                return $this->manifestCache['src'][$type][$relativePath];
            }
        }
        
        // Check if file exists in manifest
        if ($module && isset($this->manifestCache[$module][$type][$file])) {
            return $this->manifestCache[$module][$type][$file];
        } elseif (isset($this->manifestCache['core'][$type][$file])) {
            return $this->manifestCache['core'][$type][$file];
        }
        
        // Special handling for bundles
        if (isset($this->manifestCache['bundles'][$type])) {
            foreach ($this->manifestCache['bundles'][$type] as $bundleName => $bundlePath) {
                // Check if the file is part of a bundle
                if (isset($this->manifestCache['meta']['bundleFiles'][$bundleName][$file])) {
                    return $bundlePath;
                }
            }
        }
        
        return null;
    }
    
    /**
     * Initialize a new manifest structure with enhanced metadata
     */
    public function initializeManifest(): array
    {
        // Generate a version identifier
        $version = date('YmdHis');
        
        // Create basic manifest structure
        $manifest = [
            'core' => [
                'css' => [],
                'js' => [],
                'images' => [],
                'other' => []
            ],
            'src' => [
                'css' => [],
                'js' => [],
                'images' => [],
                'other' => []
            ],
            'meta' => [
                'version' => $version,
                'generated' => date('Y-m-d H:i:s'),
                'environment' => \baseKRIZAN\Config\Config::get('environment'),
                'bundlingEnabled' => \baseKRIZAN\Config\Config::get('asset.bundling_enabled'),
                'bundleFiles' => [],
                'stats' => [
                    'totalAssets' => 0,
                    'totalSize' => 0,
                    'sizeByType' => []
                ]
            ]
        ];
        
        $this->currentVersion = $version;
        
        return $manifest;
    }
    
    /**
     * Save manifest to file with versioning
     */
    public function saveManifest(array $manifest): void
    {
        // Make sure directory exists
        $dirPath = dirname($this->manifestPath);
        if (!is_dir($dirPath)) {
            mkdir($dirPath, 0755, true);
        }
        
        // Add metadata if not already present
        if (!isset($manifest['meta'])) {
            $manifest['meta'] = [
                'version' => date('YmdHis'),
                'generated' => date('Y-m-d H:i:s'),
                'environment' => \baseKRIZAN\Config\Config::get('environment')
            ];
        }
        
        // Calculate statistics
        $this->calculateManifestStats($manifest);
        
        // Get the version from metadata
        $version = $manifest['meta']['version'];
        
        // Save as the current version
        file_put_contents($this->manifestPath, json_encode($manifest, JSON_PRETTY_PRINT));
        
        // Also save as a specific version
        $versionPath = $this->getVersionPath($version);
        file_put_contents($versionPath, json_encode($manifest, JSON_PRETTY_PRINT));
        
        // Update cache
        $this->manifestCache = $manifest;
        $this->currentVersion = $version;
        
        $this->log('info', "Saved asset manifest version $version to: " . $this->manifestPath);
    }
    
    /**
     * Calculate manifest statistics
     */
    private function calculateManifestStats(array &$manifest): void
    {
        $totalAssets = 0;
        $totalSize = 0;
        $sizeByType = [];
        
        // Function to process a section of the manifest
        $processSection = function(array $section, string $type) use (&$totalAssets, &$totalSize, &$sizeByType) {
            foreach ($section as $file => $path) {
                $totalAssets++;
                
                // Try to get the actual file size
                $fullPath = $this->rootDir . '/' . ltrim($path, '/');
                $size = file_exists($fullPath) ? filesize($fullPath) : 0;
                
                $totalSize += $size;
                
                if (!isset($sizeByType[$type])) {
                    $sizeByType[$type] = 0;
                }
                
                $sizeByType[$type] += $size;
            }
        };
        
        // Process core assets
        foreach (['css', 'js', 'images', 'other'] as $type) {
            if (isset($manifest['core'][$type])) {
                $processSection($manifest['core'][$type], $type);
            }
        }
        
        // Process src assets
        foreach (['css', 'js', 'images', 'other'] as $type) {
            if (isset($manifest['src'][$type])) {
                $processSection($manifest['src'][$type], $type);
            }
        }
        
        // Process modules
        foreach ($manifest as $moduleName => $moduleData) {
            if ($moduleName === 'core' || $moduleName === 'src' || $moduleName === 'meta' || $moduleName === 'bundles') {
                continue;
            }
            
            foreach (['css', 'js', 'images', 'other'] as $type) {
                if (isset($moduleData[$type])) {
                    $processSection($moduleData[$type], $type);
                }
            }
        }
        
        // Update manifest metadata
        $manifest['meta']['stats'] = [
            'totalAssets' => $totalAssets,
            'totalSize' => $totalSize,
            'sizeByType' => $sizeByType
        ];
    }
    
    /**
     * Roll back to a previous manifest version
     * 
     * @param string|null $version Specific version to roll back to (null for previous)
     * @return bool Success status
     */
    public function rollback(?string $version = null): bool
    {
        $versions = $this->getAvailableVersions();
        
        if (empty($versions)) {
            $this->log('warning', "No manifest versions available for rollback");
            return false;
        }
        
        // If no specific version, use the previous one
        if ($version === null) {
            // Find the index of the current version
            $currentIndex = array_search($this->currentVersion, $versions);
            
            if ($currentIndex === false || $currentIndex === count($versions) - 1) {
                $this->log('warning', "No previous version available for rollback");
                return false;
            }
            
            $version = $versions[$currentIndex + 1];
        }
        
        // Check if the version exists
        $versionPath = $this->getVersionPath($version);
        
        if (!file_exists($versionPath)) {
            $this->log('warning', "Manifest version $version not found");
            return false;
        }
        
        // Load the version
        $manifest = json_decode(file_get_contents($versionPath), true);
        
        if (!$manifest) {
            $this->log('error', "Failed to decode manifest version $version");
            return false;
        }
        
        // Save it as the current manifest
        file_put_contents($this->manifestPath, json_encode($manifest, JSON_PRETTY_PRINT));
        
        // Update cache
        $this->manifestCache = $manifest;
        $this->currentVersion = $version;
        
        $this->log('info', "Rolled back to manifest version $version");
        return true;
    }
    
    /**
     * Get all available manifest versions
     * 
     * @return array Array of versions, newest first
     */
    public function getAvailableVersions(): array
    {
        $versions = [];
        
        // Get all manifest files
        $files = glob($this->manifestsDir . '/manifest_*.json');
        
        // Extract versions from filenames
        foreach ($files as $file) {
            if (preg_match('/manifest_(.+)\.json$/', $file, $matches)) {
                $versions[] = $matches[1];
            }
        }
        
        // Sort by version (newest first)
        rsort($versions);
        
        return $versions;
    }
    
    /**
     * Get the path for a specific manifest version
     */
    private function getVersionPath(string $version): string
    {
        return $this->manifestsDir . '/manifest_' . $version . '.json';
    }
    
    /**
     * Get the current manifest version
     */
    public function getCurrentVersion(): ?string
    {
        return $this->currentVersion;
    }
    
    /**
     * Get the current manifest cache
     */
    public function getManifestCache(): array
    {
        return $this->manifestCache;
    }
    
    /**
     * Delete a specific manifest version
     */
    public function deleteVersion(string $version): bool
    {
        $versionPath = $this->getVersionPath($version);
        
        if (file_exists($versionPath)) {
            $result = unlink($versionPath);
            
            if ($result) {
                $this->log('info', "Deleted manifest version $version");
            } else {
                $this->log('error', "Failed to delete manifest version $version");
            }
            
            return $result;
        }
        
        $this->log('warning', "Manifest version $version not found");
        return false;
    }
    
    /**
     * Cleanup old manifest versions
     * 
     * @param int $keep Number of versions to keep (besides the current one)
     */
    public function cleanupVersions(int $keep = 5): int
    {
        $versions = $this->getAvailableVersions();
        
        if (count($versions) <= $keep + 1) {
            // Not enough versions to clean up
            return 0;
        }
        
        // Get versions to delete (keep the current version and the specified number of older ones)
        $versionsToDelete = array_slice($versions, $keep + 1);
        
        $deleted = 0;
        foreach ($versionsToDelete as $version) {
            if ($this->deleteVersion($version)) {
                $deleted++;
            }
        }
        
        $this->log('info', "Cleaned up $deleted old manifest versions, kept current + $keep previous versions");
        return $deleted;
    }
    
    /**
     * Get detailed manifest information
     */
    public function getManifestInfo(?string $version = null): array
    {
        // Load the specified version or use current
        $manifest = $this->manifestCache;
        
        if ($version !== null && $version !== $this->currentVersion) {
            $versionPath = $this->getVersionPath($version);
            
            if (file_exists($versionPath)) {
                $manifest = json_decode(file_get_contents($versionPath), true) ?: [];
            } else {
                return [
                    'error' => "Manifest version $version not found"
                ];
            }
        }
        
        if (empty($manifest)) {
            return [
                'error' => 'No manifest loaded'
            ];
        }
        
        // Extract metadata
        $info = $manifest['meta'] ?? [];
        
        // Count assets
        $info['assetCounts'] = [
            'core' => [
                'css' => count($manifest['core']['css'] ?? []),
                'js' => count($manifest['core']['js'] ?? []),
                'images' => count($manifest['core']['images'] ?? []),
                'other' => count($manifest['core']['other'] ?? [])
            ],
            'src' => [
                'css' => count($manifest['src']['css'] ?? []),
                'js' => count($manifest['src']['js'] ?? []),
                'images' => count($manifest['src']['images'] ?? []),
                'other' => count($manifest['src']['other'] ?? [])
            ],
            'bundles' => [
                'css' => count($manifest['bundles']['css'] ?? []),
                'js' => count($manifest['bundles']['js'] ?? [])
            ]
        ];
        
        // Count module assets
        $moduleCounts = [];
        
        foreach ($manifest as $moduleName => $moduleData) {
            if ($moduleName === 'core' || $moduleName === 'src' || $moduleName === 'meta' || $moduleName === 'bundles') {
                continue;
            }
            
            $moduleCounts[$moduleName] = [
                'css' => count($moduleData['css'] ?? []),
                'js' => count($moduleData['js'] ?? []),
                'images' => count($moduleData['images'] ?? []),
                'other' => count($moduleData['other'] ?? [])
            ];
        }
        
        $info['modules'] = $moduleCounts;
        
        return $info;
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}