<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Asset Path Resolver
 * 
 * Centralizirani sustav za upravljanje svim putanjama vezanim za assete:
 * - Definira i inicijalizira sve relevantne putanje
 * - Pruža statičke metode za dohvaćanje putanja
 * - Osigurava konzistentnost putanja kroz cijeli asset sustav
 */
class AssetPathResolver
{
    // Statička instanca za singleton
    private static ?self $instance = null;
    
    // Base paths
    private string $rootDir;
    private string $publicPath;
    private string $basePath;
    
    // Source paths
    private string $coreSourcePath;      // Core assets source path
    private string $srcSourcePath;       // Src assets source path
    private string $moduleSourceBasePath;  // Base path for all module sources
    
    // Cache paths
    private string $coreCachePath;       // Core assets cache path
    private string $srcCachePath;        // Src assets cache path
    private string $moduleCachePath;     // Module assets cache path
    
    // Public paths (relative to web root)
    private string $corePublicPath;      // Core assets public path
    private string $srcPublicPath;       // Src assets public path
    private string $modulePublicPath;    // Module assets public path
    
    // File types configuration
    private array $fileTypes = [];
    
    // Logger property
    private ?Logger $logger = null;
    
    /**
     * Get singleton instance
     */
    public static function getInstance(?Logger $logger = null): self
    {
        if (self::$instance === null) {
            self::$instance = new self(\baseKRIZAN\Config\Config::get('paths.root'), $logger);
        } else if ($logger !== null && self::$instance->logger === null) {
            // Update logger if instance exists but doesn't have logger
            self::$instance->setLogger($logger);
        }
        return self::$instance;
    }
    
    /**
     * Constructor - centralized path initialization
     */
    public function __construct(string $rootDir, ?Logger $logger = null)
    {
        $this->rootDir = $rootDir;
        $this->logger = $logger;
        
        // Initialize paths
        $this->initializeFilesystemPaths();
        $this->initializeWebPaths();
        $this->initializeFileTypes();
        
        // Ensure required directories exist
        $this->ensureRequiredDirectories();
        
        if ($logger) {
            $logger->assets("AssetPathResolver initialized with centralized path management");
            $this->logPathsOverview();
        }
    }
    
    /**
     * Initialize all filesystem paths used by the asset system
     * - Now directly pulling full paths from configuration
     */
    private function initializeFilesystemPaths(): void
    {
        // Base paths
        $this->rootDir = \baseKRIZAN\Config\Config::get('paths.root');
        $this->publicPath = \baseKRIZAN\Config\Config::get('paths.public');
        
        // Source paths - pull complete paths from config
        $this->coreSourcePath = \baseKRIZAN\Config\Config::get('paths.resourcesassets');
        $this->srcSourcePath = \baseKRIZAN\Config\Config::get('paths.srcassets');
        $this->moduleSourceBasePath = \baseKRIZAN\Config\Config::get('paths.moduli');
        
        // Cache paths - pull complete paths from config
        $this->coreCachePath = \baseKRIZAN\Config\Config::get('paths.publicappassets');
        $this->srcCachePath = \baseKRIZAN\Config\Config::get('paths.publicsrcassets');
        $this->moduleCachePath = \baseKRIZAN\Config\Config::get('paths.publicmoduliassets');
    }
    
    /**
     * Initialize web-accessible paths (URLs)
     */
    private function initializeWebPaths(): void
    {
        // Base web path
        $this->basePath = \baseKRIZAN\Config\Config::get('paths.app_url');
        
        // Public relative paths from config
        $this->corePublicPath = \baseKRIZAN\Config\Config::get('paths.publicappassets_relative');
        $this->srcPublicPath = \baseKRIZAN\Config\Config::get('paths.publicsrcassets_relative');
        $this->modulePublicPath = \baseKRIZAN\Config\Config::get('paths.publicmoduliassets_relative');
    }
    
    /**
     * Initialize file types from configuration
     */
    private function initializeFileTypes(): void
    {
        // Get from configuration if available
        if (\baseKRIZAN\Config\Config::has('asset.types')) {
            $this->fileTypes = \baseKRIZAN\Config\Config::get('asset.types');
        } else {
            // Fallback to hardcoded defaults
            $this->fileTypes = [
                'css' => ['css'],
                'js' => ['js'],
                'images' => ['jpg', 'jpeg', 'png', 'gif', 'svg'],
                'other' => ['ico']
            ];
        }
    }
    
    /**
     * Ensure all required directories exist
     */
    private function ensureRequiredDirectories(): void
    {
        // Only create the base directories, not the type subdirectories
        // The subdirectories will be created on demand when assets are processed
        $this->ensureDirectoryExists($this->coreCachePath);
        $this->ensureDirectoryExists($this->srcCachePath);
        $this->ensureDirectoryExists($this->moduleCachePath);
    }
    
    /**
     * Log an overview of all configured paths
     */
    private function logPathsOverview(): void
    {
        $this->log('info', "=== ASSET PATH CONFIGURATION ===");
        $this->log('info', "Root directory: {$this->rootDir}");
        $this->log('info', "Public directory: {$this->publicPath}");
        $this->log('info', "Base web path: {$this->basePath}");
        
        $this->log('info', "--- SOURCE PATHS ---");
        $this->log('info', "Core assets source: {$this->coreSourcePath}");
        $this->log('info', "Src assets source: {$this->srcSourcePath}");
        $this->log('info', "Module assets base: {$this->moduleSourceBasePath}");
        
        $this->log('info', "--- CACHE PATHS ---");
        $this->log('info', "Core assets cache: {$this->coreCachePath}");
        $this->log('info', "Src assets cache: {$this->srcCachePath}");
        $this->log('info', "Module assets cache: {$this->moduleCachePath}");
        
        $this->log('info', "--- PUBLIC PATHS ---");
        $this->log('info', "Core assets public: {$this->corePublicPath}");
        $this->log('info', "Src assets public: {$this->srcPublicPath}");
        $this->log('info', "Module assets public: {$this->modulePublicPath}");
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }
    
    /**
     * Determine asset paths (source, cache, public) with enhanced organization
     */
    public function determineAssetPaths(string $file, ?string $module = null): array
    {
        $this->log('debug', "determineAssetPaths called with file: $file, module: " . ($module ?? 'none'));
        
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $type = $this->getAssetType($ext);
        $fileName = basename($file);
        
        // Check if this is a path-based asset (contains directory structure)
        $hasPath = strpos($file, '/') !== false;
        $relativePath = $hasPath ? dirname($file) : '';
        
        // Generate hash based on file and module
        $hashSource = $module ? "$module/$file" : $file;
        $hash = substr(md5($hashSource), 0, 8);
        $hashedName = $this->getHashedFileName($fileName, $hash);
        
        if ($module) {
            // MODULE ASSET - assets from a specific module
            $sourcePath = "{$this->moduleSourceBasePath}/$module/resources/assets/$type/$file";
            $cachePath = "{$this->moduleCachePath}/$module/$type" . ($relativePath ? "/$relativePath" : "") . "/$hashedName";
            $publicPath = "{$this->modulePublicPath}/$module/$type" . ($relativePath ? "/$relativePath" : "") . "/$hashedName";
        } else if (strpos($file, 'srcassets/') === 0) {
            // SRC ASSET - assets from app/src
            $relativeSrcPath = substr($file, 10); // Remove 'srcassets/' prefix
            
            // Important fix: Keep the full relative path structure
            // For example, srcassets/datatables/css/file.css should maintain the datatables/css path
            $sourcePath = "{$this->srcSourcePath}/$relativeSrcPath";
            
            // Extract component name (first directory after srcassets/)
            $parts = explode('/', $relativeSrcPath);
            $componentName = $parts[0];
            
            // For files with subdirectories like datatables/css/file.css
            if (count($parts) > 1) {
                $subPath = implode('/', array_slice($parts, 0, -1)); // path without filename
                $cachePath = "{$this->srcCachePath}/$subPath/$hashedName";
                $publicPath = "{$this->srcPublicPath}/$subPath/$hashedName";
            } else {
                // For simple files like datatables/file.css
                $cachePath = "{$this->srcCachePath}/$componentName/$hashedName";
                $publicPath = "{$this->srcPublicPath}/$componentName/$hashedName";
            }
        } else if ($hasPath) {
            // PATH-BASED CORE ASSET (e.g., datatables/css/style.css)
            $sourcePath = "{$this->coreSourcePath}/$file";
            $cachePath = "{$this->coreCachePath}/$relativePath/$hashedName";
            $publicPath = "{$this->corePublicPath}/$relativePath/$hashedName";
        } else {
            // STANDARD CORE ASSET
            $sourcePath = "{$this->coreSourcePath}/$type/$file";
            $cachePath = "{$this->coreCachePath}/$type/$hashedName";
            $publicPath = "{$this->corePublicPath}/$type/$hashedName";
        }
        
        // Ensure cache directory exists
        $this->ensureDirectoryExists(dirname($cachePath));
        
        // Log path resolution
        $this->log('debug', "Paths resolved - source: $sourcePath, cache: $cachePath, public: $publicPath");
        
        return [
            'source' => $sourcePath,
            'cache' => $cachePath,
            'public' => $publicPath
        ];
    }
    
    /**
     * Get default asset path if source doesn't exist
     */
    public function getDefaultAssetPath(string $file, ?string $module = null): string
    {
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $type = $this->getAssetType($ext);
        $hasPath = strpos($file, '/') !== false;
        
        if ($module) {
            // Module asset
            return $this->basePath . $this->modulePublicPath . "/$module/$type/$file";
        } else if (strpos($file, 'srcassets/') === 0) {
            // Source asset
            $relativePath = substr($file, 10); // Remove 'srcassets/' prefix
            return $this->basePath . $this->srcPublicPath . "/$relativePath";
        } else if ($hasPath) {
            // Path-based asset
            return $this->basePath . $this->corePublicPath . "/$file";
        } else {
            // Standard asset
            return $this->basePath . $this->corePublicPath . "/$type/$file";
        }
    }
    
    /**
     * Handle special path formats like appassets/, srcassets/ and moduliassets/
     */
    public function handleSpecialPaths(string $file, ?string &$module): ?string
    {
        // Handle appassets/ paths (core assets)
        if (strpos($file, 'appassets/') === 0) {
            // Strip 'appassets/' prefix and process normally
            $file = substr($file, 10); // Length of 'appassets/' is 10
            return null; // Allow normal processing with the modified file
        } 
        // Handle moduliassets/ paths
        else if (strpos($file, 'moduliassets/') === 0) {
            // Extract module name and path
            $parts = explode('/', substr($file, 13)); // Length of 'moduliassets/' is 13
            if (count($parts) >= 2) {
                $module = $parts[0];
                $file = implode('/', array_slice($parts, 1));
                return null; // Allow normal processing with the modified file and module
            }
        } 
        // Handle srcassets/ paths (explicitly handle for clarity)
        else if (strpos($file, 'srcassets/') === 0) {
            // We will process this in determineAssetPaths
            return null;
        }
        
        return null; // No special path found
    }
    
    /**
     * Resolve an absolute path (starting with /)
     */
    public function resolveAbsolutePath(string $file): string
    {
        return $this->basePath . '/' . ltrim($file, '/');
    }
    
    /**
     * Apply base path to a URL
     */
    public function applyBasePath(string $path): string
    {
        return $this->basePath . $path;
    }
    
    /**
     * Apply CDN to a URL if enabled
     */
    public function applyCdn(string $url, string $cdnUrl): string
    {
        // Skip if already an absolute URL
        if (preg_match('/^https?:\/\//', $url)) {
            return $url;
        }
        
        // Get base path and determine if URL is relative to it
        if (!empty($this->basePath) && strpos($url, $this->basePath) === 0) {
            // Remove base path for CDN URL
            $url = substr($url, strlen($this->basePath));
        }
        
        // Convert URL to CDN format
        $url = ltrim($url, '/');
        
        // Return CDN URL
        return rtrim($cdnUrl, '/') . '/' . $url;
    }
    
    /**
     * Generate a hashed filename for cache busting
     */
    public function getHashedFileName(string $fileName, string $hash): string
    {
        $ext = pathinfo($fileName, PATHINFO_EXTENSION);
        $baseName = pathinfo($fileName, PATHINFO_FILENAME);
        $dir = dirname($fileName);
        
        if ($dir === '.') {
            return "$baseName.$hash.$ext";
        } else {
            return "$dir/$baseName.$hash.$ext";
        }
    }
    
    /**
     * Ensure directory exists
     */
    public function ensureDirectoryExists(string $dir): void
    {
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
    }
    
    /**
     * Helper to resolve relative path
     */
    public function resolveRelativePath(string $basePath, string $relativePath): string
    {
        // Clean paths
        $relativePath = trim($relativePath, '\'"');
        
        // If already an absolute path, return it
        if ($relativePath[0] === '/') {
            return $this->rootDir . $relativePath;
        }
        
        // If it's an empty path, return the base path
        if (empty($relativePath)) {
            return $basePath;
        }
        
        // Handle "./" prefix explicitly
        if (strpos($relativePath, './') === 0) {
            $relativePath = substr($relativePath, 2);
        }
        
        // For "../" type references, use proper path resolution
        if (strpos($relativePath, '../') !== false) {
            // Split path into parts
            $parts = explode('/', $relativePath);
            $basePathParts = explode('/', $basePath);
            
            // Start from base path
            $resultParts = $basePathParts;
            
            foreach ($parts as $part) {
                if ($part === '.') {
                    // Current directory, do nothing
                    continue;
                } elseif ($part === '..') {
                    // Go one level up
                    array_pop($resultParts);
                } else {
                    // Add part to path
                    $resultParts[] = $part;
                }
            }
            
            // Construct path back
            return implode('/', $resultParts);
        }
        
        // Simple case - just append to base path
        return $basePath . '/' . $relativePath;
    }
    
    /**
     * Get the asset type based on file extension
     */
    public function getAssetType(string $ext): string
    {
        foreach ($this->fileTypes as $type => $extensions) {
            if (in_array(strtolower($ext), $extensions)) {
                return $type;
            }
        }
        
        return 'other';
    }
    
    // Static helper methods to easily access paths throughout the application
    
    /**
     * Get root directory path
     */
    public static function getRootPath(): string
    {
        return self::getInstance()->rootDir;
    }
    
    /**
     * Get public root directory path
     */
    public static function getPublicRootPath(): string
    {
        return self::getInstance()->publicPath;
    }
    
    /**
     * Get base web path (URL prefix)
     */
    public static function getBaseWebPath(): string
    {
        return self::getInstance()->basePath;
    }
    
    /**
     * Get core assets source path (resources/assets)
     */
    public static function getCoreAssetsSourcePath(): string
    {
        return self::getInstance()->coreSourcePath;
    }
    
    /**
     * Get src assets source path (app/src/assets)
     */
    public static function getSrcAssetsSourcePath(): string
    {
        return self::getInstance()->srcSourcePath;
    }
    
    /**
     * Get module source base path
     */
    public static function getModuleSourceBasePath(): string
    {
        return self::getInstance()->moduleSourceBasePath;
    }
    
    /**
     * Get core assets cache path (public/appassets)
     */
    public static function getCoreAssetsCachePath(): string
    {
        return self::getInstance()->coreCachePath;
    }
    
    /**
     * Get src assets cache path (public/srcassets)
     */
    public static function getSrcAssetsCachePath(): string
    {
        return self::getInstance()->srcCachePath;
    }
    
    /**
     * Get module assets cache path (public/moduliassets)
     */
    public static function getModuleAssetsCachePath(): string
    {
        return self::getInstance()->moduleCachePath;
    }
    
    /**
     * Get core assets public path (relative URL path)
     */
    public static function getCoreAssetsPublicPath(): string
    {
        return self::getInstance()->corePublicPath;
    }
    
    /**
     * Get src assets public path (relative URL path)
     */
    public static function getSrcAssetsPublicPath(): string
    {
        return self::getInstance()->srcPublicPath;
    }
    
    /**
     * Get module assets public path (relative URL path)
     */
    public static function getModuleAssetsPublicPath(): string
    {
        return self::getInstance()->modulePublicPath;
    }
    
    /**
     * Get file types configuration
     */
    public static function getFileTypesConfig(): array
    {
        return self::getInstance()->fileTypes;
    }

    /**
     * Reset singleton instance
     * Used when configuration changes
     */
    public static function reset(): void
    {
        self::$instance = null;
    }

    /**
     * Reload paths from configuration
     * Call this when configuration changes
     */
    public function reloadPaths(): void
    {
        $this->initializeFilesystemPaths();
        $this->initializeWebPaths();
        
        // Log the reload
        if ($this->logger) {
            $this->logger->assets("AssetPathResolver paths reloaded from configuration");
            $this->logPathsOverview();
        }
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}