<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Asset Pipeline
 * 
 * Provides a sequential processing pipeline for assets:
 * - Applies processors in a defined order
 * - Supports different processors for different asset types
 * - Integrates with the asset management system
 */
class AssetPipeline
{
    private string $rootDir;
    private array $processors = [];
    private ?Logger $logger = null;
    
    /**
     * Constructor
     */
    public function __construct(string $rootDir, ?Logger $logger = null)
    {
        $this->rootDir = $rootDir;
        $this->logger = $logger;
        
        // Initialize default processors
        $this->initializeProcessors();
        
        if ($logger) {
            $logger->assets("AssetPipeline initialized");
        }
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
        
        // Pass logger to processors
        foreach ($this->processors as $type => $typeProcessors) {
            foreach ($typeProcessors as $processor) {
                if (method_exists($processor, 'setLogger')) {
                    $processor->setLogger($logger);
                }
            }
        }
    }
    
    /**
     * Initialize default processors
     */
    private function initializeProcessors(): void
    {
        // CSS processors
        $this->processors['css'] = [
            new Processor\CssAutoprefixer(),
            new Processor\CssMinifier(),
            new Processor\CssImageOptimizer(),
        ];
        
        // JS processors
        $this->processors['js'] = [
            new Processor\JsModuleBundler(),
            new Processor\JsMinifier(),
        ];
        
        $this->log('info', 'Initialized default processors');
    }
    
    /**
     * Add a processor to the pipeline
     * 
     * @param string $type Asset type (css, js, etc.)
     * @param object $processor Processor instance
     * @param int $priority Lower numbers run first
     */
    public function addProcessor(string $type, object $processor, int $priority = 100): void
    {
        if (!isset($this->processors[$type])) {
            $this->processors[$type] = [];
        }
        
        $this->processors[$type][$priority] = $processor;
        
        // Sort processors by priority
        ksort($this->processors[$type]);
        
        $this->log('info', 'Added processor ' . get_class($processor) . ' for type ' . $type . ' with priority ' . $priority);
    }
    
    /**
     * Remove a processor from the pipeline
     */
    public function removeProcessor(string $type, object $processor): bool
    {
        if (!isset($this->processors[$type])) {
            return false;
        }
        
        foreach ($this->processors[$type] as $priority => $existingProcessor) {
            if ($existingProcessor === $processor) {
                unset($this->processors[$type][$priority]);
                $this->log('info', 'Removed processor ' . get_class($processor) . ' for type ' . $type);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Process an asset through the pipeline
     * 
     * @param string $assetPath Path to the asset
     * @param string $type Asset type (css, js, etc.)
     * @param array $options Pipeline options
     * @return string Processed asset path
     */
    public function process(string $assetPath, string $type, array $options = []): string
    {
        if (!isset($this->processors[$type]) || empty($this->processors[$type])) {
            // No processors for this type, return the original path
            return $assetPath;
        }
        
        $this->log('debug', "Processing asset through pipeline: $assetPath (type: $type)");
        
        $currentPath = $assetPath;
        
        // Apply each processor in order
        foreach ($this->processors[$type] as $processor) {
            $processorName = get_class($processor);
            $this->log('debug', "Applying processor $processorName to $currentPath");
            
            try {
                $result = $processor->process($currentPath, $options);
                
                if ($result !== null) {
                    $currentPath = $result;
                }
            } catch (\Exception $e) {
                $this->log('error', "Error processing asset with $processorName: " . $e->getMessage());
                // Continue with next processor
            }
        }
        
        $this->log('debug', "Pipeline processing complete: $assetPath -> $currentPath");
        return $currentPath;
    }
    
    /**
     * Process all assets in the manifest
     * 
     * @param array &$manifest Asset manifest
     * @param array $options Pipeline options
     * @return array Processing statistics
     */
    public function processAll(array &$manifest, array $options = []): array
    {
        $stats = [
            'processed' => 0,
            'size_before' => 0,
            'size_after' => 0
        ];
        
        // Process core assets
        if (isset($manifest['core'])) {
            foreach ($manifest['core'] as $type => $assets) {
                if (isset($this->processors[$type])) {
                    $this->processManifestSection($manifest['core'][$type], $type, $options, $stats);
                }
            }
        }
        
        // Process src assets
        if (isset($manifest['src'])) {
            foreach ($manifest['src'] as $type => $assets) {
                if (isset($this->processors[$type])) {
                    $this->processManifestSection($manifest['src'][$type], $type, $options, $stats);
                }
            }
        }
        
        // Process module assets
        foreach ($manifest as $moduleName => $moduleTypes) {
            if ($moduleName === 'core' || $moduleName === 'src' || $moduleName === 'meta' || $moduleName === 'bundles') {
                continue;
            }
            
            foreach ($moduleTypes as $type => $assets) {
                if (isset($this->processors[$type])) {
                    $this->processManifestSection($manifest[$moduleName][$type], $type, $options, $stats);
                }
            }
        }
        
        $this->log('info', "Pipeline processed {$stats['processed']} assets, reduced size from {$stats['size_before']} to {$stats['size_after']} bytes");
        return $stats;
    }
    
    /**
     * Process a section of the manifest
     */
    private function processManifestSection(array &$section, string $type, array $options, array &$stats): void
    {
        foreach ($section as $file => $path) {
            // Skip empty paths
            if (empty($path)) {
                continue;
            }
            
            // Convert to full file path
            $fullPath = $this->rootDir . '/' . ltrim($path, '/');
            
            // Get original file size
            $originalSize = file_exists($fullPath) ? filesize($fullPath) : 0;
            $stats['size_before'] += $originalSize;
            
            // Apply processors
            try {
                // Special handling for bundles which are already at their final paths
                $tempPath = $path;
                
                // Apply processors
                foreach ($this->processors[$type] as $processor) {
                    $processorName = get_class($processor);
                    
                    try {
                        $result = $processor->process($fullPath, $options);
                        
                        if ($result !== null && file_exists($result)) {
                            $fullPath = $result;
                        }
                    } catch (\Exception $e) {
                        $this->log('error', "Error processing file $file with $processorName: " . $e->getMessage());
                    }
                }
                
                // Update stats
                $newSize = file_exists($fullPath) ? filesize($fullPath) : 0;
                $stats['size_after'] += $newSize;
                $stats['processed']++;
                
                $this->log('debug', "Processed $file: $originalSize bytes -> $newSize bytes");
                
            } catch (\Exception $e) {
                $this->log('error', "Error processing file $file: " . $e->getMessage());
            }
        }
    }
    
    /**
     * Get all registered processors
     */
    public function getProcessors(): array
    {
        return $this->processors;
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}