<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Asset Watcher
 * 
 * Watches for changes in asset files and recompiles them:
 * - Uses inotify extension to watch directories
 * - Processes file changes in real-time
 * - Only available in development mode
 */
class AssetWatcher
{
    private string $rootDir;
    private array $watchDirs = [];
    private AssetCompiler $compiler;
    private AssetPathResolver $pathResolver;
    private bool $running = false;
    
    // Logger property
    private ?Logger $logger = null;
    
    /**
     * Constructor
     */
    public function __construct(string $rootDir, ?Logger $logger = null)
    {
        // Use centralized path management
        $this->rootDir = AssetPathResolver::getRootPath();
        $this->logger = $logger;
        
        // Initialize components
        $this->compiler = new AssetCompiler($this->rootDir, true, $logger); // Development mode
        $this->pathResolver = AssetPathResolver::getInstance($logger);
        
        // Define source directories to watch
        $this->initializeWatchDirectories();
        
        if ($logger) {
            $logger->assets("AssetWatcher initialized with " . count($this->watchDirs) . " watch directories");
        }
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
        
        // Propagate logger to components
        $this->compiler->setLogger($logger);
        $this->pathResolver->setLogger($logger);
    }
    
    /**
     * Initialize directories to watch
     */
    private function initializeWatchDirectories(): void
    {
        // Get correct paths for resource directories to watch
        $resourcesPath = AssetPathResolver::getCoreAssetsSourcePath();
        $publicAppassetsPath = AssetPathResolver::getCoreAssetsCachePath();
        
        // Core directories
        $this->watchDirs = [
            $resourcesPath,
            $publicAppassetsPath
        ];
        
        // Check for modules directory
        $modulesPath = AssetPathResolver::getModuleSourceBasePath();
        if (is_dir($modulesPath)) {
            $modules = glob($modulesPath . "/*", GLOB_ONLYDIR);
            foreach ($modules as $module) {
                $assetsDir = "$module/resources/assets";
                if (is_dir($assetsDir)) {
                    $this->watchDirs[] = $assetsDir;
                }
            }
        }
    }
    
    /**
     * Start watching for changes
     */
    public function watch(): void
    {
        // Check if inotify extension is loaded
        if (!extension_loaded('inotify')) {
            $this->log('error', "Inotify extension is required for watching assets.");
            echo "ERROR: Inotify extension is required. Install using: pecl install inotify\n";
            return;
        }
        
        // Initialize inotify
        $inotify = inotify_init();
        
        // Set up non-blocking mode
        stream_set_blocking($inotify, 0);
        
        // Add watch for each directory
        $watches = [];
        foreach ($this->watchDirs as $dir) {
            if (is_dir($dir)) {
                $this->log('info', "Watching directory: $dir");
                echo "Watching directory: $dir\n";
                $watches[] = inotify_add_watch($inotify, $dir, IN_MODIFY | IN_CREATE | IN_MOVED_TO);
                
                // Also watch subdirectories
                $this->addSubdirectoryWatches($inotify, $dir, $watches);
            }
        }
        
        $this->log('info', "Watching for changes. Press Ctrl+C to stop.");
        echo "Watching for changes. Press Ctrl+C to stop.\n";
        
        // Set running flag
        $this->running = true;
        
        // Watch for changes
        while ($this->running) {
            $events = inotify_read($inotify);
            
            if ($events !== false) {
                foreach ($events as $event) {
                    if (isset($event['name'])) {
                        $filename = $event['name'];
                        
                        // Skip if not a compilable asset
                        $ext = pathinfo($filename, PATHINFO_EXTENSION);
                        if (!in_array($ext, ['css', 'js', 'svg', 'png', 'jpg', 'jpeg', 'gif'])) {
                            continue;
                        }
                        
                        $this->log('info', "File changed: $filename");
                        echo "File changed: $filename\n";
                        $this->compileFile($filename);
                    }
                }
            }
            
            // Sleep to prevent high CPU usage
            usleep(200000); // 0.2 seconds
        }
        
        // Clean up inotify watches
        foreach ($watches as $watch) {
            inotify_rm_watch($inotify, $watch);
        }
        
        // Close inotify instance
        fclose($inotify);
    }
    
    /**
     * Add subdirectories to watch list
     */
    private function addSubdirectoryWatches($inotify, string $dir, array &$watches): void
    {
        $subdirs = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($subdirs as $subdir) {
            if ($subdir->isDir()) {
                $this->log('debug', "Watching subdirectory: {$subdir->getPathname()}");
                $watches[] = inotify_add_watch($inotify, $subdir->getPathname(), IN_MODIFY | IN_CREATE | IN_MOVED_TO);
            }
        }
    }
    
    /**
     * Compile a specific file
     */
    private function compileFile(string $file): void
    {
        $this->log('info', "Compiling file: $file");
        
        // Determine if this is a module file
        $module = null;
        if (preg_match('/^modules\/([^\/]+)\/(.+)$/', $file, $matches)) {
            $module = $matches[1];
            $file = $matches[2];
        }
        
        // Process the file using AssetManager
        $assetManager = AssetManager::getInstance();
        $result = $assetManager->processAssetPath($file, $module);
        
        $this->log('info', "Compiled to: $result");
        echo "Compiled to: $result\n";
    }
    
    /**
     * Stop watching
     */
    public function stop(): void
    {
        $this->running = false;
        $this->log('info', "Stopping asset watcher");
    }
    
    /**
     * Get watched directories
     */
    public function getWatchDirectories(): array
    {
        return $this->watchDirs;
    }
    
    /**
     * Log a message if logger is available
     */
    private function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}