<?php

namespace baseKRIZAN\Assets\Processor;

/**
 * CSS Autoprefixer
 * 
 * Adds vendor prefixes to CSS properties for better browser compatibility.
 */
class CssAutoprefixer extends AbstractProcessor
{
    private array $prefixes = [
        'transform' => ['-webkit-transform', '-moz-transform', '-ms-transform', '-o-transform'],
        'animation' => ['-webkit-animation', '-moz-animation', '-o-animation'],
        'transition' => ['-webkit-transition', '-moz-transition', '-o-transition'],
        'box-shadow' => ['-webkit-box-shadow', '-moz-box-shadow'],
        'border-radius' => ['-webkit-border-radius', '-moz-border-radius'],
        'flex' => ['-webkit-flex', '-ms-flex'],
        'display: flex' => ['display: -webkit-flex', 'display: -ms-flexbox'],
        'flex-direction' => ['-webkit-flex-direction', '-ms-flex-direction'],
        'justify-content' => ['-webkit-justify-content', '-ms-flex-pack'],
        'align-items' => ['-webkit-align-items', '-ms-flex-align'],
        'user-select' => ['-webkit-user-select', '-moz-user-select', '-ms-user-select'],
        'appearance' => ['-webkit-appearance', '-moz-appearance', '-ms-appearance'],
        'column-count' => ['-webkit-column-count', '-moz-column-count'],
        'column-gap' => ['-webkit-column-gap', '-moz-column-gap'],
        'backdrop-filter' => ['-webkit-backdrop-filter']
    ];
    
    /**
     * Process a CSS file
     */
    public function process(string $assetPath, array $options = []): ?string
    {
        // Only process CSS files
        $ext = pathinfo($assetPath, PATHINFO_EXTENSION);
        if ($ext !== 'css') {
            return null;
        }
        
        $this->log('info', "Applying CSS autoprefixer to: $assetPath");
        
        // Read the file content
        $content = file_get_contents($assetPath);
        if ($content === false) {
            $this->log('error', "Failed to read CSS file: $assetPath");
            return null;
        }
        
        // Apply prefixes
        $modified = false;
        
        foreach ($this->prefixes as $property => $vendorPrefixes) {
            // Pattern to match property
            $pattern = '/(?<!-)(' . preg_quote($property, '/') . '\s*:)/';
            
            if (preg_match($pattern, $content)) {
                $modified = true;
                
                // Find all rule blocks containing this property
                preg_match_all('/([^{]*{[^}]*' . preg_quote($property, '/') . '\s*:[^}]*})/is', $content, $matches);
                
                if (!empty($matches[1])) {
                    foreach ($matches[1] as $match) {
                        $rule = $match;
                        
                        // Extract the property value
                        preg_match('/' . preg_quote($property, '/') . '\s*:\s*([^;]*)(;|$)/i', $rule, $valueMatch);
                        
                        if (!empty($valueMatch[1])) {
                            $value = trim($valueMatch[1]);
                            $replacement = $match;
                            
                            // Add vendor prefixes
                            foreach ($vendorPrefixes as $prefix) {
                                // Check if prefix already exists
                                if (!preg_match('/' . preg_quote($prefix, '/') . '\s*:/i', $rule)) {
                                    $property = trim($property, ': ');
                                    $prefixProperty = trim($prefix, ': ');
                                    
                                    // Insert before the original property
                                    $replacement = str_replace(
                                        "$property: $value",
                                        "$prefixProperty: $value;\n    $property: $value",
                                        $replacement
                                    );
                                }
                            }
                            
                            $content = str_replace($match, $replacement, $content);
                        }
                    }
                }
            }
        }
        
        if (!$modified) {
            $this->log('debug', "No changes made by autoprefixer");
            return null;
        }
        
        // Write the modified file
        $result = file_put_contents($assetPath, $content);
        
        if ($result === false) {
            $this->log('error', "Failed to write modified CSS file: $assetPath");
            return null;
        }
        
        $this->log('debug', "Applied autoprefixer successfully");
        return $assetPath;
    }
}