<?php

namespace baseKRIZAN\Assets\Processor;

use baseKRIZAN\Assets\AssetPathResolver;

/**
 * CSS Image Optimizer
 * 
 * Optimizes images referenced in CSS files:
 * - Finds image URLs and optimizes the referenced images
 * - Updates URL references in CSS
 * - Inlines small images as data URIs
 */
class CssImageOptimizer extends AbstractProcessor
{
    private int $inlineThreshold = 4096; // 4KB threshold for inlining
    
    /**
     * Process a CSS file
     */
    public function process(string $assetPath, array $options = []): ?string
    {
        // Only process CSS files
        $ext = pathinfo($assetPath, PATHINFO_EXTENSION);
        if ($ext !== 'css') {
            return null;
        }
        
        // Override the inlining threshold if specified in options
        if (isset($options['inline_threshold'])) {
            $this->inlineThreshold = (int) $options['inline_threshold'];
        }
        
        // Skip if image optimization is disabled in options
        if (isset($options['optimize_images']) && $options['optimize_images'] === false) {
            $this->log('debug', "Image optimization disabled for: $assetPath");
            return null;
        }
        
        $this->log('info', "Optimizing images in CSS file: $assetPath");
        
        // Read the file content
        $content = file_get_contents($assetPath);
        if ($content === false) {
            $this->log('error', "Failed to read CSS file: $assetPath");
            return null;
        }
        
        // Find all image URLs in the CSS
        preg_match_all('/url\s*\(\s*[\'"]?([^\'")]+)[\'"]?\s*\)/i', $content, $matches);
        
        if (empty($matches[1])) {
            $this->log('debug', "No image URLs found in CSS");
            return null;
        }
        
        $modified = false;
        $basePath = dirname($assetPath);
        
        foreach ($matches[1] as $index => $imageUrl) {
            // Skip external URLs, data URIs
            if (strpos($imageUrl, 'http://') === 0 ||
                strpos($imageUrl, 'https://') === 0 ||
                strpos($imageUrl, 'data:') === 0) {
                continue;
            }
            
            // Resolve the image path
            $imagePath = $this->resolvePath($imageUrl, $basePath);
            
            if (!file_exists($imagePath)) {
                $this->log('warning', "Image not found: $imagePath (referenced in CSS)");
                continue;
            }
            
            // Check if we should inline this image
            $imageSize = filesize($imagePath);
            
            if ($imageSize <= $this->inlineThreshold) {
                // Inline small images as data URIs
                $dataUri = $this->getDataUri($imagePath);
                
                if ($dataUri) {
                    $originalUrl = $matches[0][$index];
                    $newUrl = "url($dataUri)";
                    
                    $content = str_replace($originalUrl, $newUrl, $content);
                    $modified = true;
                    
                    $this->log('debug', "Inlined image: $imageUrl ($imageSize bytes)");
                }
            } else {
                // Apply image optimizations
                $optimized = $this->optimizeImage($imagePath);
                
                if ($optimized) {
                    $modified = true;
                    
                    // We don't need to update the CSS reference since the file is optimized in place
                    $this->log('debug', "Optimized image: $imageUrl");
                }
            }
        }
        
        if (!$modified) {
            $this->log('debug', "No changes made to CSS");
            return null;
        }
        
        // Write the modified CSS file
        $result = file_put_contents($assetPath, $content);
        
        if ($result === false) {
            $this->log('error', "Failed to write modified CSS file: $assetPath");
            return null;
        }
        
        return $assetPath;
    }
    
    /**
     * Resolve a relative image path to an absolute path
     */
    private function resolvePath(string $url, string $basePath): string
    {
        // Use AssetPathResolver for path resolution
        return AssetPathResolver::getInstance()->resolveRelativePath($basePath, $url);
    }
    
    /**
     * Get a data URI for an image
     */
    private function getDataUri(string $imagePath): ?string
    {
        $ext = pathinfo($imagePath, PATHINFO_EXTENSION);
        $mimeType = $this->getMimeType($ext);
        
        if (!$mimeType) {
            return null;
        }
        
        $content = file_get_contents($imagePath);
        
        if ($content === false) {
            return null;
        }
        
        return 'data:' . $mimeType . ';base64,' . base64_encode($content);
    }
    
    /**
     * Get the MIME type for an image extension
     */
    private function getMimeType(string $ext): ?string
    {
        $ext = strtolower($ext);
        
        $types = [
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'svg' => 'image/svg+xml',
            'webp' => 'image/webp'
        ];
        
        return $types[$ext] ?? null;
    }
    
    /**
     * Optimize an image
     */
    private function optimizeImage(string $imagePath): bool
    {
        $ext = strtolower(pathinfo($imagePath, PATHINFO_EXTENSION));
        
        // Different optimization strategies based on image type
        switch ($ext) {
            case 'jpg':
            case 'jpeg':
                return $this->optimizeJpeg($imagePath);
                
            case 'png':
                return $this->optimizePng($imagePath);
                
            case 'gif':
                return $this->optimizeGif($imagePath);
                
            case 'svg':
                return $this->optimizeSvg($imagePath);
                
            default:
                return false;
        }
    }
    
    /**
     * Optimize a JPEG image
     */
    private function optimizeJpeg(string $imagePath): bool
    {
        // Try to use the imagick extension if available
        if (extension_loaded('imagick')) {
            try {
                $image = new \Imagick($imagePath);
                $image->setImageCompression(\Imagick::COMPRESSION_JPEG);
                $image->setImageCompressionQuality(85);
                $image->stripImage();
                $result = $image->writeImage($imagePath);
                $image->destroy();
                return $result;
            } catch (\Exception $e) {
                $this->log('warning', "Failed to optimize JPEG with Imagick: " . $e->getMessage());
            }
        }
        
        // Fallback to GD
        if (function_exists('imagecreatefromjpeg')) {
            $image = @imagecreatefromjpeg($imagePath);
            
            if ($image) {
                $result = imagejpeg($image, $imagePath, 85);
                imagedestroy($image);
                return $result;
            }
        }
        
        return false;
    }
    
    /**
     * Optimize a PNG image
     */
    private function optimizePng(string $imagePath): bool
    {
        // Try to use the imagick extension if available
        if (extension_loaded('imagick')) {
            try {
                $image = new \Imagick($imagePath);
                $image->setImageCompression(\Imagick::COMPRESSION_ZIP);
                $image->setImageCompressionQuality(95);
                $image->stripImage();
                $result = $image->writeImage($imagePath);
                $image->destroy();
                return $result;
            } catch (\Exception $e) {
                $this->log('warning', "Failed to optimize PNG with Imagick: " . $e->getMessage());
            }
        }
        
        // Fallback to GD
        if (function_exists('imagecreatefrompng')) {
            $image = @imagecreatefrompng($imagePath);
            
            if ($image) {
                // Turn off alpha blending and set alpha flag
                imagealphablending($image, false);
                imagesavealpha($image, true);
                
                $result = imagepng($image, $imagePath, 9); // Max compression
                imagedestroy($image);
                return $result;
            }
        }
        
        return false;
    }
    
    /**
     * Optimize a GIF image
     */
    private function optimizeGif(string $imagePath): bool
    {
        // GIF optimization is minimal without specialized tools
        // For now, just ensure it's valid
        if (function_exists('imagecreatefromgif')) {
            $image = @imagecreatefromgif($imagePath);
            
            if ($image) {
                $result = imagegif($image, $imagePath);
                imagedestroy($image);
                return $result;
            }
        }
        
        return false;
    }
    
    /**
     * Optimize an SVG image
     */
    private function optimizeSvg(string $imagePath): bool
    {
        $content = file_get_contents($imagePath);
        
        if ($content === false) {
            return false;
        }
        
        // Basic SVG minification
        // Remove comments
        $content = preg_replace('/<!--.*?-->/s', '', $content);
        
        // Remove whitespace between tags
        $content = preg_replace('/>\s+</', '><', $content);
        
        // Remove whitespace at the beginning and end of the file
        $content = trim($content);
        
        // Remove XML declaration if present
        $content = preg_replace('/<\?xml.*?\?>/i', '', $content);
        
        return file_put_contents($imagePath, $content) !== false;
    }
}