<?php

namespace baseKRIZAN\Assets\Processor;

/**
 * CSS Minifier
 * 
 * Minifies CSS files to reduce their size.
 */
class CssMinifier extends AbstractProcessor
{
    /**
     * Process a CSS file
     */
    public function process(string $assetPath, array $options = []): ?string
    {
        // Only process CSS files
        $ext = pathinfo($assetPath, PATHINFO_EXTENSION);
        if ($ext !== 'css') {
            return null;
        }
        
        // Skip if minification is disabled in options
        if (isset($options['minify']) && $options['minify'] === false) {
            $this->log('debug', "Minification disabled for: $assetPath");
            return null;
        }
        
        $this->log('info', "Minifying CSS file: $assetPath");
        
        // Read the file content
        $content = file_get_contents($assetPath);
        if ($content === false) {
            $this->log('error', "Failed to read CSS file: $assetPath");
            return null;
        }
        
        // Store original size for comparison
        $originalSize = strlen($content);
        
        // Apply minification
        $minified = $this->minifyCss($content);
        
        // Calculate size reduction
        $minifiedSize = strlen($minified);
        $reduction = $originalSize > 0 ? round(($originalSize - $minifiedSize) / $originalSize * 100, 2) : 0;
        
        $this->log('debug', "CSS size reduced from $originalSize to $minifiedSize bytes ($reduction% reduction)");
        
        // Write the minified file
        $result = file_put_contents($assetPath, $minified);
        
        if ($result === false) {
            $this->log('error', "Failed to write minified CSS file: $assetPath");
            return null;
        }
        
        return $assetPath;
    }
    
    /**
     * Minify CSS content
     */
    private function minifyCss(string $css): string
    {
        // Remove comments
        $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
        
        // Remove whitespace around special characters
        $css = preg_replace('/\s*([{}:;,])\s*/', '$1', $css);
        
        // Remove unnecessary semicolons
        $css = preg_replace('/;}/', '}', $css);
        
        // Remove newlines, tabs and multiple spaces
        $css = preg_replace('/\s+/', ' ', $css);
        
        // Remove spaces that can safely be removed
        $css = str_replace(': ', ':', $css);
        $css = str_replace([' {', '{ '], '{', $css);
        $css = str_replace(['} ', ' }'], '}', $css);
        $css = str_replace(['; ', ' ;'], ';', $css);
        $css = str_replace(['( ', ' ('], '(', $css);
        $css = str_replace([') ', ' )'], ')', $css);
        $css = str_replace([', ', ' ,'], ',', $css);
        
        // Trim
        $css = trim($css);
        
        return $css;
    }
}