<?php

namespace baseKRIZAN\Assets\Processor;

/**
 * JavaScript Minifier
 * 
 * Minifies JavaScript files to reduce their size.
 */
class JsMinifier extends AbstractProcessor
{
    /**
     * Process a JavaScript file
     */
    public function process(string $assetPath, array $options = []): ?string
    {
        // Only process JS files
        $ext = pathinfo($assetPath, PATHINFO_EXTENSION);
        if ($ext !== 'js') {
            return null;
        }
        
        // Skip if minification is disabled in options
        if (isset($options['minify']) && $options['minify'] === false) {
            $this->log('debug', "Minification disabled for: $assetPath");
            return null;
        }
        
        $this->log('info', "Minifying JavaScript file: $assetPath");
        
        // Read the file content
        $content = file_get_contents($assetPath);
        if ($content === false) {
            $this->log('error', "Failed to read JavaScript file: $assetPath");
            return null;
        }
        
        // Store original size for comparison
        $originalSize = strlen($content);
        
        // Apply minification
        $minified = $this->minifyJs($content);
        
        // Calculate size reduction
        $minifiedSize = strlen($minified);
        $reduction = $originalSize > 0 ? round(($originalSize - $minifiedSize) / $originalSize * 100, 2) : 0;
        
        $this->log('debug', "JavaScript size reduced from $originalSize to $minifiedSize bytes ($reduction% reduction)");
        
        // Write the minified file
        $result = file_put_contents($assetPath, $minified);
        
        if ($result === false) {
            $this->log('error', "Failed to write minified JavaScript file: $assetPath");
            return null;
        }
        
        return $assetPath;
    }
    
    /**
     * Minify JavaScript content
     */
    private function minifyJs(string $js): string
    {
        // Save possible strings
        $strings = [];
        $stringPlaceholders = [];
        
        // Extract and replace strings to protect them during minification
        $js = preg_replace_callback(
            '/(\'[^\'\\\\]*(?:\\\\.[^\'\\\\]*)*\'|"[^"\\\\]*(?:\\\\.[^"\\\\]*)*")/',
            function ($matches) use (&$strings, &$stringPlaceholders) {
                $placeholder = '___STRING_' . count($strings) . '___';
                $strings[] = $matches[0];
                $stringPlaceholders[] = $placeholder;
                return $placeholder;
            },
            $js
        );
        
        // Remove single-line comments
        $js = preg_replace('/\/\/.*$/m', '', $js);
        
        // Remove multi-line comments
        $js = preg_replace('/\/\*.*?\*\//s', '', $js);
        
        // Remove whitespace
        $js = preg_replace('/\s+/', ' ', $js);
        
        // Remove whitespace around operators
        $js = preg_replace('/\s*([=\+\-\*\/<>\{\}\(\)\[\],;:])\s*/', '$1', $js);
        
        // Special handling for spaces after keywords
        $keywords = ['if', 'else', 'for', 'while', 'do', 'switch', 'try', 'catch', 'function', 'return', 'var', 'let', 'const'];
        foreach ($keywords as $keyword) {
            $js = preg_replace('/\b' . $keyword . '\s+/', $keyword . ' ', $js);
        }
        
        // Restore strings
        foreach ($stringPlaceholders as $index => $placeholder) {
            $js = str_replace($placeholder, $strings[$index], $js);
        }
        
        // Trim
        $js = trim($js);
        
        return $js;
    }
}