<?php

namespace baseKRIZAN\Assets\Processor;

use baseKRIZAN\Assets\AssetPathResolver;

/**
 * JavaScript Module Bundler
 * 
 * Bundles JavaScript modules, resolving imports and dependencies.
 */
class JsModuleBundler extends AbstractProcessor
{
    private array $processedModules = [];
    private array $moduleCache = [];
    
    /**
     * Process a JavaScript file
     */
    public function process(string $assetPath, array $options = []): ?string
    {
        // Only process JS files
        $ext = pathinfo($assetPath, PATHINFO_EXTENSION);
        if ($ext !== 'js') {
            return null;
        }
        
        // Skip if bundling is disabled in options
        if (isset($options['bundle']) && $options['bundle'] === false) {
            $this->log('debug', "Bundling disabled for: $assetPath");
            return null;
        }
        
        // Skip non-module files (quick check for ES module syntax)
        $content = file_get_contents($assetPath);
        if (strpos($content, 'import ') === false && strpos($content, 'export ') === false) {
            $this->log('debug', "Not a module (no import/export found): $assetPath");
            return null;
        }
        
        $this->log('info', "Processing JavaScript module: $assetPath");
        
        // Reset processed modules for this run
        $this->processedModules = [];
        
        // Process the entry module
        $bundledContent = $this->bundleModule($assetPath, dirname($assetPath));
        
        if ($bundledContent === null) {
            $this->log('warning', "Failed to bundle module: $assetPath");
            return null;
        }
        
        // Write the bundled file
        $result = file_put_contents($assetPath, $bundledContent);
        
        if ($result === false) {
            $this->log('error', "Failed to write bundled JavaScript file: $assetPath");
            return null;
        }
        
        $this->log('debug', "Successfully bundled module: $assetPath");
        return $assetPath;
    }
    
    /**
     * Bundle a JavaScript module with its dependencies
     */
    private function bundleModule(string $modulePath, string $basePath): ?string
    {
        // Check if we've already processed this module
        if (isset($this->processedModules[$modulePath])) {
            return '';
        }
        
        // Mark module as processed
        $this->processedModules[$modulePath] = true;
        
        // Read module content
        $content = file_get_contents($modulePath);
        if ($content === false) {
            $this->log('error', "Failed to read module file: $modulePath");
            return null;
        }
        
        // Save original content before modifications
        $originalContent = $content;
        
        // Track imports and exports
        $imports = [];
        $exports = [];
        
        // Parse imports using a regex pattern
        preg_match_all('/import\s+(?:{([^}]+)}|([^\s;]+)|\*\s+as\s+([^\s;]+))\s+from\s+[\'"]([^\'"]+)[\'"];?/i', $content, $importMatches, PREG_SET_ORDER);
        
        foreach ($importMatches as $match) {
            $importPath = $match[count($match) - 1]; // Last group contains the path
            $imports[] = $importPath;
            
            // Resolve the imported module path
            $resolvedPath = $this->resolvePath($importPath, $basePath);
            
            if ($resolvedPath) {
                // Get the imported module's content
                $importedContent = $this->bundleModule($resolvedPath, dirname($resolvedPath));
                
                if ($importedContent !== null) {
                    // Remove the import statement from the current module
                    $content = str_replace($match[0], '', $content);
                }
            }
        }
        
        // Parse dynamic imports (only basic support)
        preg_match_all('/import\s*\(\s*[\'"]([^\'"]+)[\'"]\s*\)/i', $content, $dynamicImportMatches, PREG_SET_ORDER);
        
        foreach ($dynamicImportMatches as $match) {
            $importPath = $match[1];
            $imports[] = $importPath;
            
            // We don't actually replace dynamic imports, just log them
            $this->log('debug', "Dynamic import found and preserved: $importPath");
        }
        
        // Parse exports
        preg_match_all('/export\s+(?:default\s+)?(?:const|let|var|function|class|interface|enum|type)\s+([a-zA-Z0-9_$]+)/i', $content, $exportMatches, PREG_SET_ORDER);
        
        foreach ($exportMatches as $match) {
            $exportName = $match[1];
            $exports[] = $exportName;
        }
        
        // If this is the entry module, return the final bundled content
        if (count($this->processedModules) === 1) {
            // Create a self-executing function wrapper to prevent scope pollution
            $bundledContent = "(function() {\n";
            
            // Add global module cache
            $bundledContent .= "// Module cache\n";
            $bundledContent .= "var __moduleCache = {};\n\n";
            
            // Include all processed modules
            foreach ($this->moduleCache as $path => $moduleContent) {
                $bundledContent .= "// Module: " . basename($path) . "\n";
                $bundledContent .= $moduleContent . "\n\n";
            }
            
            // Add entry module content
            $bundledContent .= "// Entry module: " . basename($modulePath) . "\n";
            $bundledContent .= $content . "\n";
            
            // Close wrapper
            $bundledContent .= "})();";
            
            return $bundledContent;
        }
        
        // For imported modules, store their content in the cache
        $this->moduleCache[$modulePath] = $content;
        
        return $content;
    }
    
    /**
     * Resolve a module path
     */
    private function resolvePath(string $importPath, string $basePath): ?string
    {
        // Handle relative paths
        if (strpos($importPath, './') === 0 || strpos($importPath, '../') === 0) {
            $resolvedPath = AssetPathResolver::getInstance()->resolveRelativePath($basePath, $importPath);
            
            // Add .js extension if not present
            if (!pathinfo($resolvedPath, PATHINFO_EXTENSION)) {
                $resolvedPath .= '.js';
            }
            
            if (file_exists($resolvedPath)) {
                return $resolvedPath;
            }
        }
        
        // Handle absolute paths
        if ($importPath[0] === '/') {
            $rootPath = AssetPathResolver::getRootPath();
            $resolvedPath = $rootPath . $importPath;
            
            if (file_exists($resolvedPath)) {
                return $resolvedPath;
            }
        }
        
        // Handle bare module specifiers (like 'jquery')
        // This would require a more complex Node.js-like resolution algorithm
        
        $this->log('warning', "Could not resolve module path: $importPath from $basePath");
        return null;
    }
}