<?php

namespace baseKRIZAN\Assets;

use baseKRIZAN\Error\Logger;

/**
 * Enhanced ResourceLoader for templates
 * 
 * Handles:
 * - Loading CSS and JS resources
 * - Managing dependencies
 * - Output in templates
 * - Support for nested resource structure
 * - CSP nonce support for inline styles and scripts
 */
class ResourceLoader
{
    // Static arrays to store resources
    private static array $cssResources = [];
    private static array $jsHeadResources = [];
    private static array $jsBodyResources = [];
    private static array $resourceModules = [];
    private static array $modulePriorities = [];
    private static array $jsBodyPriorities = [];
    private static bool $initialized = false;
    
    // Nonce za CSP
    private static ?string $nonce = null;
    private static bool $nonceLocked = false;
    private static bool $nonceScriptInjected = false;
    
    // Default module priorities
    private static array $defaultModulePriorities = [
        'default' => 10,
        'select2' => 20,
        'flatpickr' => 30,
        'datatable' => 40,
        'datetime' => 50,
        'multidatetime' => 60,
        'borna' => 70
    ];
    
    // Logger property
    private static ?Logger $logger = null;
    
    /**
     * Set logger statically
     */
    public static function setLogger(?Logger $logger): void
    {
        self::$logger = $logger;
        
        if ($logger) {
            $logger->assets("ResourceLoader logger set");
        }
    }
    
    /**
     * Initialize the resource loader
     */
    public static function init(): void
    {
        self::log('info', "Initializing ResourceLoader");
        
        if (self::$initialized) {
            self::log('debug', "ResourceLoader already initialized");
            return;
        }
        
        // Generate nonce for this request if not already set
        self::generateNonce();
        
        // Load default resources
        self::loadResource('default');
        self::$initialized = true;
        self::log('info', "ResourceLoader initialization complete");
    }
    
    /**
     * Check if ResourceLoader is initialized
     */
    public static function isInitialized(): bool
    {
        return self::$initialized;
    }
    
    /**
     * Generate a unique nonce for this request
     */
    private static function generateNonce(): void
    {
        if (self::$nonce === null) {
            // Prvo pokušaj dohvatiti iz sesije
            if (session_status() === PHP_SESSION_ACTIVE && isset($_SESSION['csp_nonce'])) {
                self::$nonce = $_SESSION['csp_nonce'];
                self::log('debug', "Using nonce from session: " . self::$nonce);
            } else {
                // Generiraj novi nonce
                self::$nonce = bin2hex(random_bytes(16));
                self::log('debug', "Generated new nonce: " . self::$nonce);
                
                // Spremi u sesiju
                if (session_status() === PHP_SESSION_ACTIVE) {
                    $_SESSION['csp_nonce'] = self::$nonce;
                }
            }
            
            // Zaključaj nonce da se ne može mijenjati
            self::$nonceLocked = true;
        } elseif (self::$nonceLocked) {
            // Ne dozvoli promjenu nonce-a ako je zaključan
            self::log('debug', "Nonce already locked, keeping: " . self::$nonce);
            return;
        }
    }
    
    /**
     * Get the current nonce
     */
    public static function getNonce(): string
    {
        if (self::$nonce === null) {
            self::generateNonce();
        }
        
        return self::$nonce;
    }
    
    /**
     * Get resource map from configuration, including module resources
     */
    private static function getResourceMap(): array
    {
        // Use Config to get the paths
        $resourcesPath = \baseKRIZAN\Config\Config::get('paths.config') . '/app_asset_resources.php';
        $modulesBasePath = \baseKRIZAN\Config\Config::get('paths.moduli');
        
        self::log('debug', "Looking for app_asset_resources.php at: $resourcesPath");
        
        // Start with an empty resources array
        $resources = [];
        
        // First, load the base application resources
        if (file_exists($resourcesPath)) {
            self::log('debug', "Found app_asset_resources.php");
            $baseResources = include $resourcesPath;
            
            if (is_array($baseResources)) {
                $resources = $baseResources;
                self::log('debug', "Base resource modules found: " . json_encode(array_keys($resources)));
            } else {
                self::log('warning', "app_asset_resources.php did not return an array");
            }
        } else {
            self::log('warning', "app_asset_resources.php not found at: $resourcesPath");
        }
        
        // Then, scan for and merge module-specific resource files
        if (is_dir($modulesBasePath)) {
            $modules = scandir($modulesBasePath);
            
            foreach ($modules as $module) {
                // Skip . and .. directories and non-directories
                if ($module === '.' || $module === '..' || !is_dir($modulesBasePath . '/' . $module)) {
                    continue;
                }
                
                // Look for {module}_asset_resources.php file
                $moduleResourcePath = $modulesBasePath . '/' . $module . '/' . $module . '_asset_resources.php';
                
                if (file_exists($moduleResourcePath)) {
                    self::log('debug', "Found module resource file: $moduleResourcePath");
                    
                    try {
                        $moduleResources = include $moduleResourcePath;
                        
                        if (is_array($moduleResources)) {
                            // Merge module resources with base resources
                            $resources = self::mergeResourceArrays($resources, $moduleResources);
                            self::log('debug', "Merged resources from module: $module");
                        } else {
                            self::log('warning', "Module resource file for $module did not return an array");
                        }
                    } catch (\Throwable $e) {
                        self::log('error', "Error including module resource file for $module: " . $e->getMessage());
                    }
                } else {
                    self::log('debug', "No resource file found for module: $module at $moduleResourcePath");
                }
            }
        } else {
            self::log('debug', "Modules directory not found or not accessible: $modulesBasePath");
        }
        
        return $resources;
    }

    /**
     * Recursively merge resource arrays, preserving nested structures
     */
    private static function mergeResourceArrays(array $baseResources, array $moduleResources): array
    {
        $result = $baseResources;
        
        foreach ($moduleResources as $key => $value) {
            if (isset($result[$key]) && is_array($result[$key]) && is_array($value)) {
                // If both are arrays, merge recursively
                $result[$key] = self::mergeResourceArrays($result[$key], $value);
            } else {
                // Otherwise, just override or add
                $result[$key] = $value;
            }
        }
        
        return $result;
    }
    
    /**
     * Load a resource module with support for nested structure
     */
    public static function loadResource(string $modulePath, int $priority = null): void
    {
        if (isset(self::$resourceModules[$modulePath])) {
            self::log('debug', "Module $modulePath already loaded, skipping");
            return; // Already loaded
        }
        
        $resourceMap = self::getResourceMap();
        $module = $resourceMap;
        $originalPath = $modulePath;
        
        // Handle nested paths (e.g. 'borna.base')
        $parts = explode('.', $modulePath);
        
        // Navigate through the nested structure
        foreach ($parts as $part) {
            if (isset($module[$part])) {
                $module = $module[$part];
            } else {
                self::log('warning', "Resource module not found: $originalPath (failed at '$part')");
                return;
            }
        }
        
        // Check if the module has the expected structure
        if (isset($module['head']) || isset($module['body'])) {
            self::log('info', "Loading resource module: $originalPath");
            self::$resourceModules[$originalPath] = $module;
            
            // Set priority
            if ($priority !== null) {
                self::$modulePriorities[$originalPath] = $priority;
            } else {
                // Set priority based on original module name or first part of path for nested modules
                $rootModule = $parts[0];
                self::$modulePriorities[$originalPath] = self::$defaultModulePriorities[$rootModule] ?? 999;
            }
        } else {
            self::log('warning', "Resource module has invalid structure: $originalPath");
        }
    }
    
    /**
     * Generate resources for HTML head with enhanced nested module support
     */
    public static function generateHeadResources(): void
    {
        self::log('debug', "Generating head resources");
        self::log('debug', "Module priorities: " . json_encode(self::$modulePriorities));
        self::log('debug', "Resource modules: " . json_encode(array_keys(self::$resourceModules)));

        // Make sure we're initialized
        if (!self::$initialized) {
            self::init();
        }
        
        // Prvo injektirajmo nonce vrijednost u globalnu JavaScript varijablu
        echo "<script nonce=\"" . self::getNonce() . "\">window.CSP_NONCE = \"" . self::getNonce() . "\";</script>\n";
        
        // Track processed resources to avoid duplicates
        $processedFiles = [];
        
        // First check if bundling is enabled and we're in production
        $developmentMode = \baseKRIZAN\Config\Config::get('environment') === 'development';
        $bundlingEnabled = \baseKRIZAN\Config\Config::get('asset.bundling_enabled');
        
        if (!$developmentMode && $bundlingEnabled) {
            // Try to load the manifest to check for bundles
            $manifestFile = AssetPathResolver::getCoreAssetsCachePath() . '/manifest.json';
            
            if (file_exists($manifestFile)) {
                $manifest = json_decode(file_get_contents($manifestFile), true);
                
                if (isset($manifest['bundles']['css']['core'])) {
                    echo '<link rel="stylesheet" type="text/css" href="' . $manifest['bundles']['css']['core'] . '">' . PHP_EOL;
                    $processedFiles[] = 'core-css-bundle';
                }
                
                if (isset($manifest['bundles']['css']['plugins'])) {
                    echo '<link rel="stylesheet" type="text/css" href="' . $manifest['bundles']['css']['plugins'] . '">' . PHP_EOL;
                    $processedFiles[] = 'plugins-css-bundle';
                }
                
                if (isset($manifest['bundles']['js']['core'])) {
                    echo '<script type="text/javascript" src="' . $manifest['bundles']['js']['core'] . '" nonce="' . self::getNonce() . '"></script>' . PHP_EOL;
                    $processedFiles[] = 'core-js-bundle';
                }
            }
        }
        
        // Sort modules by priority
        asort(self::$modulePriorities);
        
        // Process CSS resources from modules
        foreach (array_keys(self::$modulePriorities) as $modulePath) {
            self::log('debug', "Processing module: $modulePath");
            
            if (!isset(self::$resourceModules[$modulePath]['head'])) {
                self::log('debug', "No head resources for module: $modulePath");
                continue;
            }
            
            self::log('debug', "Head resources for $modulePath: " . json_encode(self::$resourceModules[$modulePath]['head']));
            
            foreach (self::$resourceModules[$modulePath]['head'] as $resource) {
                // Skip if not CSS or already processed
                if (!self::isCSS($resource) || in_array($resource, $processedFiles)) {
                    continue;
                }
                
                echo '<link rel="stylesheet" type="text/css" href="' . $resource . '">' . PHP_EOL;
                $processedFiles[] = $resource;
            }
        }
        
        // Output custom CSS with nonce
        foreach (self::$cssResources as $css) {
            // Adding nonce to any inline <style> tags in the CSS
            $css = preg_replace('/<style([^>]*)>/', '<style$1 nonce="' . self::getNonce() . '">', $css);
            echo $css . PHP_EOL;
        }
        
        // Process JS resources for head
        foreach (array_keys(self::$modulePriorities) as $modulePath) {
            if (!isset(self::$resourceModules[$modulePath]['head'])) {
                continue;
            }
            
            foreach (self::$resourceModules[$modulePath]['head'] as $resource) {
                // Skip if not JS or already processed
                if (!self::isJS($resource) || in_array($resource, $processedFiles)) {
                    continue;
                }
                
                echo '<script type="text/javascript" src="' . $resource . '" nonce="' . self::getNonce() . '"></script>' . PHP_EOL;
                $processedFiles[] = $resource;
            }
        }
        
        // Output custom JS for head with nonce
        foreach (self::$jsHeadResources as $js) {
            // Adding nonce to any inline <script> tags
            $js = preg_replace('/<script([^>]*)>/', '<script$1 nonce="' . self::getNonce() . '">', $js);
            echo $js . PHP_EOL;
        }
    }
    
    /**
     * Check if a resource is CSS
     */
    private static function isCSS($resource): bool
    {
        if (is_array($resource) && isset($resource['src'])) {
            return strpos($resource['src'], '.css') !== false;
        }
        return is_string($resource) && strpos($resource, '.css') !== false;
    }
    
    /**
     * Check if a resource is JavaScript
     */
    private static function isJS($resource): bool
    {
        if (is_array($resource) && isset($resource['src'])) {
            return strpos($resource['src'], '.js') !== false;
        }
        return is_string($resource) && strpos($resource, '.js') !== false;
    }
    
    /**
     * Generate resources for HTML body (at the end) with enhanced nested module support
     */
    public static function generateBodyResources(): void
    {
        // Make sure we're initialized
        if (!self::$initialized) {
            self::init();
        }
        
        // Track processed resources to avoid duplicates
        $processedFiles = [];
        
        // First check if bundling is enabled and we're in production
        $developmentMode = \baseKRIZAN\Config\Config::get('environment') === 'development';
        $bundlingEnabled = \baseKRIZAN\Config\Config::get('asset.bundling_enabled');
        
        if (!$developmentMode && $bundlingEnabled) {
            // Try to load the manifest to check for bundles
            $manifestFile = AssetPathResolver::getCoreAssetsCachePath() . '/manifest.json';
            
            if (file_exists($manifestFile)) {
                $manifest = json_decode(file_get_contents($manifestFile), true);
                
                if (isset($manifest['bundles']['js']['plugins'])) {
                    echo '<script type="text/javascript" src="' . $manifest['bundles']['js']['plugins'] . '" nonce="' . self::getNonce() . '"></script>' . PHP_EOL;
                    $processedFiles[] = 'plugins-js-bundle';
                }
            }
        }
        
        // Sort modules by priority
        asort(self::$modulePriorities);
        
        // Process JS resources for body
        foreach (array_keys(self::$modulePriorities) as $modulePath) {
            if (!isset(self::$resourceModules[$modulePath]['body'])) {
                continue;
            }
            
            foreach (self::$resourceModules[$modulePath]['body'] as $resource) {
                // Skip if already processed
                if (in_array($resource, $processedFiles)) {
                    continue;
                }
                
                // Provjeri je li resurs objekt s atributima ili jednostavan string
                if (is_array($resource) && isset($resource['src'])) {
                    $attributes = $resource; // Koristi postojeće atribute
                    $resourcePath = $resource['src']; // Za provjeru duplikata
                    
                    // Izgradi HTML tag sa svim atributima
                    $attrStr = '';
                    foreach ($attributes as $key => $value) {
                        if ($key !== 'src') { // 'src' već dodajemo odvojeno
                            $attrStr .= ' ' . $key . '="' . htmlspecialchars($value, ENT_QUOTES, 'UTF-8') . '"';
                        }
                    }
                    
                    echo '<script src="' . $resource['src'] . '"' . $attrStr . ' nonce="' . self::getNonce() . '"></script>' . PHP_EOL;
                } else {
                    // Originalna logika za string resurse
                    echo '<script type="text/javascript" src="' . $resource . '" nonce="' . self::getNonce() . '"></script>' . PHP_EOL;
                    $resourcePath = $resource;
                }
                
                $processedFiles[] = $resourcePath;
            }
        }
        
        // Sort custom JS by priority
        asort(self::$jsBodyPriorities);
        
        // Output custom JS for body with nonce
        foreach (array_keys(self::$jsBodyPriorities) as $index) {
            $js = self::$jsBodyResources[$index];
            // Adding nonce to any inline <script> tags
            $js = preg_replace('/<script([^>]*)>/', '<script$1 nonce="' . self::getNonce() . '">', $js);
            echo $js . PHP_EOL;
        }
    }
    
    /**
     * Clear all loaded resources
     */
    public static function clear(): void
    {
        self::$cssResources = [];
        self::$jsHeadResources = [];
        self::$jsBodyResources = [];
        self::$jsBodyPriorities = [];
        self::$resourceModules = [];
        self::$modulePriorities = [];
        self::$initialized = false;
        self::$nonce = null;
        self::$nonceLocked = false;
        self::$nonceScriptInjected = false;
    }

    /**
     * Log a message if logger is available
     */
    private static function log(string $level, string $message): void
    {
        if (self::$logger) {
            switch ($level) {
                case 'debug':
                    self::$logger->assets($message);
                    break;
                case 'info':
                    self::$logger->assets($message);
                    break;
                case 'warning':
                    self::$logger->assets($message);
                    break;
                case 'error':
                    self::$logger->error($message);
                    break;
                default:
                    self::$logger->assets($message);
            }
        }
    }
}