<?php

namespace baseKRIZAN\Assets\Strategy;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Assets\AssetPathResolver;
use baseKRIZAN\Assets\AssetCompiler;

/**
 * Abstract Asset Strategy
 * 
 * Base implementation with common functionality for all asset strategies.
 */
abstract class AbstractAssetStrategy implements AssetStrategyInterface
{
    protected AssetPathResolver $pathResolver;
    protected AssetCompiler $compiler;
    protected ?Logger $logger = null;
    protected bool $developmentMode;
    protected bool $cdnEnabled;
    protected string $cdnUrl;
    
    /**
     * Constructor
     */
    public function __construct(
        AssetPathResolver $pathResolver,
        AssetCompiler $compiler,
        ?Logger $logger = null
    ) {
        $this->pathResolver = $pathResolver;
        $this->compiler = $compiler;
        $this->logger = $logger;
        
        // Initialize configuration
        $this->developmentMode = \baseKRIZAN\Config\Config::get('environment') === 'development';
        $this->cdnEnabled = \baseKRIZAN\Config\Config::get('asset.cdn_enabled');
        $this->cdnUrl = \baseKRIZAN\Config\Config::get('asset.cdn_url');
    }
    
    /**
     * Set logger
     */
    public function setLogger(?Logger $logger): void
    {
        $this->logger = $logger;
    }
    
    /**
     * Common asset processing logic
     */
    protected function processAssetCommon(string $file, ?string $module, array $paths): string
    {
        // If source file doesn't exist, return a default path
        if (!file_exists($paths['source'])) {
            $this->log('warning', "Source file not found: {$paths['source']}");
            $defaultPath = $this->pathResolver->getDefaultAssetPath($file, $module);
            return $defaultPath;
        }
        
        // In development mode, compile on-demand if needed
        if ($this->developmentMode && $this->compiler->shouldCompileAsset($paths['source'], $paths['cache'])) {
            
            $ext = pathinfo($file, PATHINFO_EXTENSION);
            if ($ext === 'css') {
                // Get the content
                $content = file_get_contents($paths['source']);
                
                // ALSO REMOVE this line causing the error:
                // $content = $this->compiler->processCSSImageUrls($content, dirname($paths['source']));
                
                // Just compile with the original content
                $this->compiler->compileAssetWithContent($paths['source'], $paths['cache'], $ext, $content);
            } else {
                // Just compile the asset normally
                $this->compiler->compileAsset($paths['source'], $paths['cache'], $ext);
            }
        }
        
        // Build final path
        $finalPath = $this->pathResolver->applyBasePath($paths['public']);
        
        // Apply CDN if enabled
        if ($this->cdnEnabled && !empty($this->cdnUrl)) {
            $finalPath = $this->pathResolver->applyCdn($finalPath, $this->cdnUrl);
        }
        
        return $finalPath;
    }
    
    /**
     * Process a directory for production compilation
     * Common implementation that works for all asset types
     */
    protected function processDirectoryForProduction(
        string $sourceDir,
        string $outputDir,
        string $type,
        ?string $module,
        string $relativePath,
        array &$manifestSection
    ): array {
        $stats = [
            'processed' => 0,
            'size_before' => 0,
            'size_after' => 0
        ];
        
        // Ensure directory exists
        $this->pathResolver->ensureDirectoryExists($outputDir);
        
        // Get all matching files in this directory
        $fileTypes = AssetPathResolver::getFileTypesConfig();
        $pattern = "/$sourceDir/*";
        if (!empty($fileTypes[$type])) {
            $extensions = implode(',', $fileTypes[$type]);
            $pattern = "$sourceDir/*.{$extensions}";
        }
        
        $files = glob($pattern, GLOB_BRACE);
        
        // Collect all image references from CSS files
        $imageReferences = [];
        
        // First scan CSS files and collect image references
        if ($type === 'css') {
            foreach ($files as $file) {
                if (is_file($file) && pathinfo($file, PATHINFO_EXTENSION) === 'css') {
                    $content = file_get_contents($file);
                    
                    // Find URL references in CSS
                    preg_match_all('/url\(["\']?(.*?)["\']?\)/i', $content, $matches);
                    
                    if (!empty($matches[1])) {
                        foreach ($matches[1] as $imgPath) {
                            // Ignore external URLs
                            if (strpos($imgPath, 'http') === 0 || 
                                strpos($imgPath, 'data:') === 0) {
                                continue;
                            }
                            
                            // Save the relative path and CSS file that references it
                            $imageReferences[] = [
                                'cssFile' => $file,
                                'imgPath' => $imgPath
                            ];
                            
                            $this->log('debug', "Found image in CSS: $imgPath in file " . basename($file));
                        }
                    }
                }
            }
        }
        
        // Process files
        foreach ($files as $file) {
            if (is_file($file)) {
                $fileName = basename($file);
                $fileContent = file_get_contents($file);
                $fileSize = strlen($fileContent);
                
                // Generate hash for cache busting
                $hash = substr(md5($fileContent), 0, 8);
                $hashedName = $this->pathResolver->getHashedFileName($fileName, $hash);
                
                // Output path
                $outputPath = "$outputDir/$hashedName";
                
                // If it's a CSS file, replace URL references with hashed names
                if (pathinfo($file, PATHINFO_EXTENSION) === 'css') {
                    // Just get the file content without any special processing
                    $fileContent = file_get_contents($file);
                }
                
                // Process content for production
                if ($this->compiler->isMinifyEnabled() && in_array($type, ['css', 'js'])) {
                    $fileContent = $this->compiler->minifyContent($fileContent, $type);
                }
                
                // Write to output
                file_put_contents($outputPath, $fileContent);
                $processedSize = strlen($fileContent);
                
                // Update stats
                $stats['processed']++;
                $stats['size_before'] += $fileSize;
                $stats['size_after'] += $processedSize;
                
                // Add to manifest
                $sourcePath = $relativePath ? "$relativePath/$fileName" : $fileName;
                
                $publicModuliAssetsPath = AssetPathResolver::getModuleAssetsPublicPath();
                $publicAppassetsPath = AssetPathResolver::getCoreAssetsPublicPath();
                $publicSrcAssetsPath = AssetPathResolver::getSrcAssetsPublicPath();
                
                if ($module) {
                    $manifestPath = $publicModuliAssetsPath . "/$module/$type/" . ($relativePath ? "$relativePath/" : '') . $hashedName;
                } else if (strpos($sourcePath, 'srcassets/') === 0) {
                    $relativeSrcPath = substr($sourcePath, 10); // Remove 'srcassets/' prefix
                    $manifestPath = $publicSrcAssetsPath . "/" . $relativeSrcPath . "/" . $hashedName;
                } else {
                    $manifestPath = $publicAppassetsPath . "/" . ($relativePath ? "$relativePath/" : "$type/") . $hashedName;
                }
                
                $manifestSection[$sourcePath] = $manifestPath;
            }
        }
        
        // Process subdirectories
        $subdirs = glob("$sourceDir/*", GLOB_ONLYDIR);
        
        foreach ($subdirs as $subdir) {
            $subdirName = basename($subdir);
            $newRelativePath = $relativePath ? "$relativePath/$subdirName" : $subdirName;
            $newOutputDir = "$outputDir/$subdirName";
            
            $subdirStats = $this->processDirectoryForProduction(
                $subdir,
                $newOutputDir,
                $type,
                $module,
                $newRelativePath,
                $manifestSection
            );
            
            $stats['processed'] += $subdirStats['processed'];
            $stats['size_before'] += $subdirStats['size_before'];
            $stats['size_after'] += $subdirStats['size_after'];
        }
        
        return $stats;
    }
    
    /**
     * Log a message if logger is available
     */
    protected function log(string $level, string $message): void
    {
        if ($this->logger) {
            switch ($level) {
                case 'debug':
                    $this->logger->assets($message);
                    break;
                case 'info':
                    $this->logger->assets($message);
                    break;
                case 'warning':
                    $this->logger->assets($message);
                    break;
                case 'error':
                    $this->logger->error($message);
                    break;
                default:
                    $this->logger->assets($message);
            }
        }
    }
}