<?php

namespace baseKRIZAN\Assets\Strategy;

use baseKRIZAN\Assets\AssetPathResolver;

/**
 * Core Asset Strategy
 * 
 * Handles core assets from resources/assets
 */
class CoreAssetStrategy extends AbstractAssetStrategy
{
    /**
     * Check if this strategy can handle the given asset
     */
    public function canHandle(string $file, ?string $module = null): bool
    {
        // Core assets don't have a module
        if ($module !== null) {
            return false;
        }
        
        // Skip srcassets files
        if (strpos($file, 'srcassets/') === 0) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Process a core asset path
     */
    public function processAssetPath(string $file, ?string $module = null): string
    {
        $this->log('debug', "CoreAssetStrategy processing: $file");
        
        // Skip if already an absolute path or URL
        if (strpos($file, 'http://') === 0 || strpos($file, 'https://') === 0) {
            return $file;
        }
        
        // For absolute paths, return as is
        if ($file[0] === '/') {
            return $this->pathResolver->resolveAbsolutePath($file);
        }
        
        // Handle special paths
        $specialPath = $this->pathResolver->handleSpecialPaths($file, $module);
        if ($specialPath) {
            return $specialPath;
        }
        
        // In production mode, check manifest
        if (!$this->developmentMode) {
            $manifestPath = \baseKRIZAN\Assets\AssetManifestManager::getInstance($this->logger)
                ->getAssetPathFromManifest($file, null);
            
            if ($manifestPath) {
                $fullPath = $this->pathResolver->applyBasePath($manifestPath);
                
                // Apply CDN if enabled
                if ($this->cdnEnabled && !empty($this->cdnUrl)) {
                    $fullPath = $this->pathResolver->applyCdn($fullPath, $this->cdnUrl);
                }
                
                return $fullPath;
            }
        }
        
        // Determine paths for this asset
        $paths = $this->determineAssetPaths($file);
        return $this->processAssetCommon($file, null, $paths);
    }
    
    /**
     * Determine asset paths specific to core assets
     */
    protected function determineAssetPaths(string $file): array
    {
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $type = $this->pathResolver->getAssetType($ext);
        $fileName = basename($file);
        
        // Check if this is a path-based asset (contains directory structure)
        $hasPath = strpos($file, '/') !== false;
        $relativePath = $hasPath ? dirname($file) : '';
        
        // Generate hash
        $hash = substr(md5($file), 0, 8);
        $hashedName = $this->pathResolver->getHashedFileName($fileName, $hash);
        
        if ($hasPath) {
            // PATH-BASED CORE ASSET (e.g., datatables/css/style.css)
            $sourcePath = $this->getSourcePath() . "/$file";
            $cachePath = $this->getCachePath() . "/$relativePath/$hashedName";
            $publicPath = $this->getPublicPath() . "/$relativePath/$hashedName";
        } else {
            // STANDARD CORE ASSET
            $sourcePath = $this->getSourcePath() . "/$type/$file";
            $cachePath = $this->getCachePath() . "/$type/$hashedName";
            $publicPath = $this->getPublicPath() . "/$type/$hashedName";
        }
        
        // Ensure cache directory exists
        $this->pathResolver->ensureDirectoryExists(dirname($cachePath));
        
        return [
            'source' => $sourcePath,
            'cache' => $cachePath,
            'public' => $publicPath
        ];
    }
    
    /**
     * Get the source path for core assets
     */
    public function getSourcePath(?string $module = null): string
    {
        return AssetPathResolver::getCoreAssetsSourcePath();
    }
    
    /**
     * Get the cache path for core assets
     */
    public function getCachePath(?string $module = null): string
    {
        return AssetPathResolver::getCoreAssetsCachePath();
    }
    
    /**
     * Get the public path for core assets
     */
    public function getPublicPath(?string $module = null): string
    {
        return AssetPathResolver::getCoreAssetsPublicPath();
    }
    
    /**
     * Compile core assets for production
     */
    public function compileForProduction(array &$manifest): array
    {
        $stats = [
            'processed' => 0,
            'size_before' => 0,
            'size_after' => 0
        ];
        
        $sourcePath = $this->getSourcePath();
        
        // Check if source directory exists
        if (!is_dir($sourcePath)) {
            $this->log('warning', "Source directory not found: {$sourcePath}");
            return $stats;
        }
        
        // Create output directory
        $outputPath = $this->getCachePath();
        $this->pathResolver->ensureDirectoryExists($outputPath);
        
        // Process common asset directories
        $fileTypes = AssetPathResolver::getFileTypesConfig();
        foreach ($fileTypes as $type => $extensions) {
            $typeDir = "{$sourcePath}/$type";
            
            if (!is_dir($typeDir)) {
                continue;
            }
            
            $outputTypeDir = "$outputPath/$type";
            $this->pathResolver->ensureDirectoryExists($outputTypeDir);
            
            // Process files in the directory
            $processedStats = $this->processDirectoryForProduction(
                $typeDir,
                $outputTypeDir,
                $type,
                null,
                '',
                $manifest['core'][$type]
            );
            
            $stats['processed'] += $processedStats['processed'];
            $stats['size_before'] += $processedStats['size_before'];
            $stats['size_after'] += $processedStats['size_after'];
        }
        
        // Process component directories (like datatables, select2, etc.)
        $components = glob("{$sourcePath}/*", GLOB_ONLYDIR);
        
        foreach ($components as $component) {
            $componentName = basename($component);
            
            // Skip if it's a standard asset type directory
            if (in_array($componentName, array_keys($fileTypes))) {
                continue;
            }
            
            $outputComponentDir = "$outputPath/$componentName";
            $this->pathResolver->ensureDirectoryExists($outputComponentDir);
            
            // Process files in the component directory
            foreach ($fileTypes as $type => $extensions) {
                $componentTypeDir = "$component/$type";
                
                if (!is_dir($componentTypeDir)) {
                    continue;
                }
                
                $outputComponentTypeDir = "$outputComponentDir/$type";
                $this->pathResolver->ensureDirectoryExists($outputComponentTypeDir);
                
                // Process files
                $processedStats = $this->processDirectoryForProduction(
                    $componentTypeDir,
                    $outputComponentTypeDir,
                    $type,
                    null,
                    "$componentName/$type",
                    $manifest['core'][$type]
                );
                
                $stats['processed'] += $processedStats['processed'];
                $stats['size_before'] += $processedStats['size_before'];
                $stats['size_after'] += $processedStats['size_after'];
            }
        }
        
        return $stats;
    }
}