<?php

namespace baseKRIZAN\Assets\Strategy;

use baseKRIZAN\Assets\AssetPathResolver;

/**
 * Module Asset Strategy
 * 
 * Handles module assets from moduli/{module_name}/resources/assets
 */
class ModuleAssetStrategy extends AbstractAssetStrategy
{
    /**
     * Check if this strategy can handle the given asset
     */
    public function canHandle(string $file, ?string $module = null): bool
    {
        // Module assets require a module name
        return $module !== null;
    }
    
    /**
     * Process a module asset path
     */
    public function processAssetPath(string $file, ?string $module = null): string
    {
        $this->log('debug', "ModuleAssetStrategy processing: $file, module: $module");
        
        // Skip if already an absolute path or URL
        if (strpos($file, 'http://') === 0 || strpos($file, 'https://') === 0) {
            return $file;
        }
        
        // Handle moduliassets prefix if present
        if (strpos($file, 'moduliassets/') === 0) {
            // Extract module name and path from moduliassets/module/path format
            $parts = explode('/', substr($file, 13)); // Length of 'moduliassets/' is 13
            if (count($parts) >= 2) {
                // Override module parameter with the one from the path
                $moduleFromPath = $parts[0];
                $fileFromPath = implode('/', array_slice($parts, 1));
                
                $this->log('debug', "Extracted from moduliassets path: module=$moduleFromPath, file=$fileFromPath");
                
                // Use the extracted values
                $module = $moduleFromPath;
                $file = $fileFromPath;
            }
        }
        
        // In production mode, check manifest
        if (!$this->developmentMode) {
            $manifestPath = \baseKRIZAN\Assets\AssetManifestManager::getInstance($this->logger)
                ->getAssetPathFromManifest($file, $module);
            
            if ($manifestPath) {
                $fullPath = $this->pathResolver->applyBasePath($manifestPath);
                
                // Apply CDN if enabled
                if ($this->cdnEnabled && !empty($this->cdnUrl)) {
                    $fullPath = $this->pathResolver->applyCdn($fullPath, $this->cdnUrl);
                }
                
                return $fullPath;
            }
        }
        
        // Determine paths for this asset
        $paths = $this->determineAssetPaths($file, $module);
        return $this->processAssetCommon($file, $module, $paths);
    }
    
    /**
     * Determine asset paths specific to module assets
     */
    protected function determineAssetPaths(string $file, ?string $module): array
    {
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $type = $this->pathResolver->getAssetType($ext);
        $fileName = basename($file);
        
        // Check if this is a path-based asset (contains directory structure)
        $hasPath = strpos($file, '/') !== false;
        $relativePath = $hasPath ? dirname($file) : '';
        
        // Generate hash
        $hash = substr(md5("$module/$file"), 0, 8);
        $hashedName = $this->pathResolver->getHashedFileName($fileName, $hash);
        
        // IMPORTANT FIX: Check if the file path already contains the asset type directory
        // For example, if file is 'css/dashboard.css', we should not add 'css/' again
        $pathContainsType = $hasPath && (strpos($relativePath, $type) === 0 || strpos($relativePath, $type . '/') !== false);
        
        if ($pathContainsType) {
            // If path already contains type directory, don't add it again, but still hash the filename
            $sourcePath = $this->getSourcePath($module) . "/$file";
            $cachePath = $this->getCachePath($module) . "/" . dirname($file) . "/$hashedName";
            $publicPath = $this->getPublicPath($module) . "/" . dirname($file) . "/$hashedName";
        } else {
            // Ostaje isto kao i prije
            $sourcePath = $this->getSourcePath($module) . "/$type/$file";
            $cachePath = $this->getCachePath($module) . "/$type" . ($relativePath ? "/$relativePath" : "") . "/$hashedName";
            $publicPath = $this->getPublicPath($module) . "/$type" . ($relativePath ? "/$relativePath" : "") . "/$hashedName";
        }
        
        // Log the calculated paths
        $this->log('debug', "Module asset paths for $file in module $module: source=$sourcePath, cache=$cachePath, public=$publicPath");
        
        // Ensure cache directory exists
        $this->pathResolver->ensureDirectoryExists(dirname($cachePath));
        
        return [
            'source' => $sourcePath,
            'cache' => $cachePath,
            'public' => $publicPath
        ];
    }
    
    /**
     * Get the source path for module assets
     */
    public function getSourcePath(?string $module = null): string
    {
        return AssetPathResolver::getModuleSourceBasePath() . "/$module/resources/assets";
    }
    
    /**
     * Get the cache path for module assets
     */
    public function getCachePath(?string $module = null): string
    {
        return AssetPathResolver::getModuleAssetsCachePath() . "/$module";
    }
    
    /**
     * Get the public path for module assets
     */
    public function getPublicPath(?string $module = null): string
    {
        return AssetPathResolver::getModuleAssetsPublicPath() . "/$module";
    }
    
    /**
     * Compile module assets for production
     */
    public function compileForProduction(array &$manifest): array
    {
        $stats = [
            'processed' => 0,
            'size_before' => 0,
            'size_after' => 0
        ];
        
        $moduleStats = [];
        $enabledModules = explode(',', \baseKRIZAN\Config\Config::get('moduli'));
        
        foreach ($enabledModules as $module) {
            if (empty($module)) continue;
            
            // Initialize module in manifest
            if (!isset($manifest[$module])) {
                $manifest[$module] = [];
                foreach (AssetPathResolver::getFileTypesConfig() as $type => $extensions) {
                    $manifest[$module][$type] = [];
                }
            }
            
            // Check if module source directory exists
            $moduleSourcePath = $this->getSourcePath($module);
            
            if (!is_dir($moduleSourcePath)) {
                $this->log('warning', "Module source directory not found: {$moduleSourcePath}");
                continue;
            }
            
            // Create output directory
            $moduleOutputPath = $this->getCachePath($module);
            $this->pathResolver->ensureDirectoryExists($moduleOutputPath);
            
            // Process each asset type
            foreach (AssetPathResolver::getFileTypesConfig() as $type => $extensions) {
                $moduleTypeDir = "$moduleSourcePath/$type";
                
                if (!is_dir($moduleTypeDir)) {
                    continue;
                }
                
                $moduleOutputTypeDir = "$moduleOutputPath/$type";
                $this->pathResolver->ensureDirectoryExists($moduleOutputTypeDir);
                
                // Process files
                $processedStats = $this->processDirectoryForProduction(
                    $moduleTypeDir,
                    $moduleOutputTypeDir,
                    $type,
                    $module,
                    '',
                    $manifest[$module][$type]
                );
                
                $stats['processed'] += $processedStats['processed'];
                $stats['size_before'] += $processedStats['size_before'];
                $stats['size_after'] += $processedStats['size_after'];
                
                $moduleStats[$module] = [
                    'processed' => ($moduleStats[$module]['processed'] ?? 0) + $processedStats['processed'],
                    'size_before' => ($moduleStats[$module]['size_before'] ?? 0) + $processedStats['size_before'],
                    'size_after' => ($moduleStats[$module]['size_after'] ?? 0) + $processedStats['size_after']
                ];
            }
        }
        
        return $stats;
    }
}