<?php

namespace baseKRIZAN\Assets\Strategy;

use baseKRIZAN\Assets\AssetPathResolver;

/**
 * Source Asset Strategy
 * 
 * Handles source assets from app/src/
 */
class SrcAssetStrategy extends AbstractAssetStrategy
{
    /**
     * Check if this strategy can handle the given asset
     */
    public function canHandle(string $file, ?string $module = null): bool
    {
        // Source assets don't have a module
        if ($module !== null) {
            return false;
        }
        
        // Only handle srcassets files
        return strpos($file, 'srcassets/') === 0;
    }
    
    /**
     * Process a source asset path
     */
    public function processAssetPath(string $file, ?string $module = null): string
    {
        $this->log('debug', "SrcAssetStrategy processing: $file");
        
        // Skip if already an absolute path or URL
        if (strpos($file, 'http://') === 0 || strpos($file, 'https://') === 0) {
            return $file;
        }
        
        // In production mode, check manifest
        if (!$this->developmentMode) {
            $manifestPath = \baseKRIZAN\Assets\AssetManifestManager::getInstance($this->logger)
                ->getAssetPathFromManifest($file, null);
            
            if ($manifestPath) {
                $fullPath = $this->pathResolver->applyBasePath($manifestPath);
                
                // Apply CDN if enabled
                if ($this->cdnEnabled && !empty($this->cdnUrl)) {
                    $fullPath = $this->pathResolver->applyCdn($fullPath, $this->cdnUrl);
                }
                
                return $fullPath;
            }
        }
        
        // SRC ASSET - assets from app/src
        $relativeSrcPath = substr($file, 10); // Remove 'srcassets/' prefix
        
        // Get the configured src path and verify it exists
        $srcBasePath = AssetPathResolver::getSrcAssetsSourcePath();
        
        // Construct the full source path
        $sourcePath = $srcBasePath . '/' . $relativeSrcPath;
        
        // Adaptive handling - if file not found at configured path but exists at different path
        if (!file_exists($sourcePath)) {
            // Try alternate path without /assets part (for compatibility)
            $alternativePath = str_replace('/app/src/assets/', '/app/src/', $sourcePath);
            
            if (file_exists($alternativePath)) {
                $this->log('debug', "File not found at configured path but exists at alternate path: $alternativePath");
                $sourcePath = $alternativePath;
            } else {
                $this->log('warning', "Source file not found at $sourcePath or $alternativePath");
            }
        }
        
        // Create paths for asset
        $paths = [
            'source' => $sourcePath,
            'cache' => $this->getCachePath() . '/' . $relativeSrcPath,
            'public' => $this->getPublicPath() . '/' . $relativeSrcPath
        ];
        
        // Ensure cache directory exists
        $this->pathResolver->ensureDirectoryExists(dirname($paths['cache']));
        
        return $this->processAssetCommon($file, null, $paths);
    }
    
    /**
     * Determine asset paths specific to source assets
     */
    protected function determineAssetPaths(string $relativeSrcPath): array
    {
        $ext = pathinfo($relativeSrcPath, PATHINFO_EXTENSION);
        $type = $this->pathResolver->getAssetType($ext);
        $fileName = basename($relativeSrcPath);
        
        // Generate hash
        $hash = substr(md5('srcassets/' . $relativeSrcPath), 0, 8);
        $hashedName = $this->pathResolver->getHashedFileName($fileName, $hash);
        
        // Sada koristimo direktno AssetPathResolver::getSrcAssetsSourcePath()
        // jer je to sad već definirano kao /app/src
        $sourcePath = AssetPathResolver::getSrcAssetsSourcePath() . "/$relativeSrcPath";
        
        // Extract component name (first directory after srcassets/)
        $parts = explode('/', $relativeSrcPath);
        $componentName = $parts[0];
        
        // For files with subdirectories like datatables/css/file.css
        if (count($parts) > 1) {
            $subPath = implode('/', array_slice($parts, 0, -1)); // path without filename
            $cachePath = $this->getCachePath() . "/$subPath/$hashedName";
            $publicPath = $this->getPublicPath() . "/$subPath/$hashedName";
        } else {
            // For simple files like datatables/file.css
            $cachePath = $this->getCachePath() . "/$componentName/$hashedName";
            $publicPath = $this->getPublicPath() . "/$componentName/$hashedName";
        }
        
        // Ensure cache directory exists
        $this->pathResolver->ensureDirectoryExists(dirname($cachePath));
        
        return [
            'source' => $sourcePath,
            'cache' => $cachePath,
            'public' => $publicPath
        ];
    }
    
    /**
     * Get the source path for source assets
     * 
     * Sad koristimo standardni način kroz AssetPathResolver
     */
    public function getSourcePath(?string $module = null): string
    {
        return AssetPathResolver::getSrcAssetsSourcePath();
    }
    
    /**
     * Get the cache path for source assets
     */
    public function getCachePath(?string $module = null): string
    {
        return AssetPathResolver::getSrcAssetsCachePath();
    }
    
    /**
     * Get the public path for source assets
     */
    public function getPublicPath(?string $module = null): string
    {
        return AssetPathResolver::getSrcAssetsPublicPath();
    }
    
    /**
     * Compile source assets for production
     */
    public function compileForProduction(array &$manifest): array
    {
        $stats = [
            'processed' => 0,
            'size_before' => 0,
            'size_after' => 0
        ];
        
        // Initialize src in manifest if not exists
        if (!isset($manifest['src'])) {
            $manifest['src'] = [];
            foreach (AssetPathResolver::getFileTypesConfig() as $type => $extensions) {
                $manifest['src'][$type] = [];
            }
        }
        
        $sourcePath = $this->getSourcePath();
        
        // Check if source directory exists
        if (!is_dir($sourcePath)) {
            $this->log('warning', "Source directory not found: {$sourcePath}");
            return $stats;
        }
        
        // Create output directory
        $outputPath = $this->getCachePath();
        $this->pathResolver->ensureDirectoryExists($outputPath);
        
        // Process component directories - ali sada izravno iz app/src
        $components = glob("{$sourcePath}/*", GLOB_ONLYDIR);
        
        foreach ($components as $component) {
            $componentName = basename($component);
            
            // Preskoči direktorije koji nisu komponente koje želimo kompajlirati
            if (in_array($componentName, ['Controller', 'Model', 'View', 'Entity', 'Repository'])) {
                continue;
            }
            
            $outputComponentDir = "$outputPath/$componentName";
            $this->pathResolver->ensureDirectoryExists($outputComponentDir);
            
            // Process each type of asset in the component
            foreach (AssetPathResolver::getFileTypesConfig() as $type => $extensions) {
                $componentTypeDir = "$component/$type";
                
                if (!is_dir($componentTypeDir)) {
                    continue;
                }
                
                $outputComponentTypeDir = "$outputComponentDir/$type";
                $this->pathResolver->ensureDirectoryExists($outputComponentTypeDir);
                
                // Process files
                $processedStats = $this->processDirectoryForProduction(
                    $componentTypeDir,
                    $outputComponentTypeDir,
                    $type,
                    null,
                    "$componentName/$type",
                    $manifest['src'][$type]
                );
                
                $stats['processed'] += $processedStats['processed'];
                $stats['size_before'] += $processedStats['size_before'];
                $stats['size_after'] += $processedStats['size_after'];
            }
        }
        
        return $stats;
    }
}