<?php

namespace baseKRIZAN\BORNA\Analyzers;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\BORNA\Interfaces\AnalyzerInterface;
use baseKRIZAN\BORNA\Enum\SecurityLevel;
use baseKRIZAN\BORNA\Storage\StorageInterface;

/**
 * Statistical anomaly detection for security analysis
 * Uses Z-score, moving averages and other statistical methods to detect outliers
 */
class AnomalyAnalyzer implements AnalyzerInterface
{
    /**
     * Logger instance
     */
    private Logger $logger;
    
    /**
     * Storage interface
     */
    private StorageInterface $storage;
    
    /**
     * Security level
     */
    private SecurityLevel $securityLevel;
    
    /**
     * Number of standard deviations for anomaly detection
     */
    private float $zScoreThreshold;
    
    /**
     * Monitored metrics
     */
    private array $metrics = [
        'request_rate' => [
            'description' => 'Requests per minute',
            'window_size' => 60,
            'anomaly_threshold' => 3.0
        ],
        'error_rate' => [
            'description' => 'Error responses per minute',
            'window_size' => 60,
            'anomaly_threshold' => 2.5
        ],
        'session_change_rate' => [
            'description' => 'Session changes per hour',
            'window_size' => 3600,
            'anomaly_threshold' => 2.0
        ],
        'path_entropy' => [
            'description' => 'Entropy of accessed paths',
            'window_size' => 3600,
            'anomaly_threshold' => 2.0
        ],
        'parameter_count' => [
            'description' => 'Number of parameters in requests',
            'window_size' => 60,
            'anomaly_threshold' => 2.5
        ],
        'parameter_length' => [
            'description' => 'Average parameter length',
            'window_size' => 60,
            'anomaly_threshold' => 3.0
        ],
        'response_time' => [
            'description' => 'Response time in milliseconds',
            'window_size' => 60,
            'anomaly_threshold' => 2.5
        ]
    ];
    
    /**
     * Cached statistics
     */
    private array $cachedStatistics = [];
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger, 
        StorageInterface $storage,
        SecurityLevel $securityLevel = SecurityLevel::MEDIUM
    ) {
        $this->logger = $logger;
        $this->storage = $storage;
        $this->securityLevel = $securityLevel;
        
        // Set Z-score threshold based on security level
        $this->zScoreThreshold = match($securityLevel) {
            SecurityLevel::LOW => 4.0,    // Very permissive
            SecurityLevel::MEDIUM => 3.0, // Standard
            SecurityLevel::HIGH => 2.0,   // Strict
        };
        
        $this->logger->borna('AnomalyAnalyzer initialized', [
            'security_level' => $this->securityLevel->name,
            'z_score_threshold' => $this->zScoreThreshold
        ]);
    }
    
    /**
     * @inheritDoc
     */
    public function analyze(array $requestData): array
    {
        // Default result with no threat
        $result = [
            'score' => 0,
            'description' => 'No anomalies detected',
            'details' => []
        ];
        
        // Calculate metrics for this request
        $metricValues = $this->calculateMetrics($requestData);
        
        // Store metrics for future analysis
        $this->storeMetrics($requestData['ip'], $metricValues);
        
        // Check for anomalies
        $anomalies = $this->detectAnomalies($requestData['ip'], $metricValues);
        
        if (!empty($anomalies)) {
            // Calculate total anomaly score
            $totalScore = 0;
            $anomalyDetails = [];
            
            foreach ($anomalies as $metric => $anomaly) {
                $totalScore += $anomaly['score'];
                
                $anomalyDetails[] = [
                    'metric' => $metric,
                    'description' => $this->metrics[$metric]['description'],
                    'value' => $anomaly['value'],
                    'expected_range' => $anomaly['expected_range'],
                    'z_score' => $anomaly['z_score'],
                    'score' => $anomaly['score']
                ];
            }
            
            // Cap total score at 100
            $totalScore = min(100, $totalScore);
            
            $result = [
                'score' => $totalScore,
                'description' => 'Statistical anomalies detected',
                'details' => [
                    'anomalies' => $anomalyDetails,
                    'security_level' => $this->securityLevel->name,
                    'z_score_threshold' => $this->zScoreThreshold
                ]
            ];
            
            // Log significant anomalies
            if ($totalScore > 50) {
                $this->logger->borna('Significant statistical anomalies detected', [
                    'ip' => $requestData['ip'],
                    'score' => $totalScore,
                    'anomalies_count' => count($anomalies)
                ]);
            }
        }
        
        return $result;
    }
    
    /**
     * @inheritDoc
     */
    public function getName(): string
    {
        return 'anomaly';
    }
    
    /**
     * @inheritDoc
     */
    public function setSecurityLevel(int $level): void
    {
        $this->securityLevel = SecurityLevel::from($level);
        
        // Update Z-score threshold based on security level
        $this->zScoreThreshold = match($this->securityLevel) {
            SecurityLevel::LOW => 4.0,    // Very permissive
            SecurityLevel::MEDIUM => 3.0, // Standard
            SecurityLevel::HIGH => 2.0,   // Strict
        };
        
        $this->logger->borna('AnomalyAnalyzer security level changed', [
            'new_level' => $this->securityLevel->name,
            'z_score_threshold' => $this->zScoreThreshold
        ]);
    }
    
    /**
     * @inheritDoc
     */
    public function getStatistics(): array
    {
        // Return cached statistics if available and less than 5 minutes old
        if (!empty($this->cachedStatistics) && 
            (time() - ($this->cachedStatistics['timestamp'] ?? 0)) < 300) {
            return $this->cachedStatistics;
        }
        
        $stats = [
            'total_anomalies_detected' => 0,
            'metrics' => [],
            'timestamp' => time()
        ];
        
        // Collect statistics for each metric
        foreach ($this->metrics as $metric => $config) {
            $metricStats = $this->storage->getMetricStatistics($metric);
            $anomalyData = $this->storage->getAnomalyData($metric, 10);
            
            $stats['metrics'][$metric] = [
                'description' => $config['description'],
                'count' => $metricStats['count'] ?? 0,
                'min' => $metricStats['min'] ?? 0,
                'max' => $metricStats['max'] ?? 0,
                'mean' => $metricStats['mean'] ?? 0,
                'stddev' => $metricStats['stddev'] ?? 0,
                'recent_values' => array_map(function($item) {
                    return [
                        'timestamp' => $item['timestamp'],
                        'value' => $item['value']
                    ];
                }, $anomalyData)
            ];
            
            // Count anomalies for this metric
            $anomalyCount = 0;
            foreach ($anomalyData as $data) {
                if (isset($data['metadata']['is_anomaly']) && $data['metadata']['is_anomaly']) {
                    $anomalyCount++;
                }
            }
            
            $stats['metrics'][$metric]['anomalies'] = $anomalyCount;
            $stats['total_anomalies_detected'] += $anomalyCount;
        }
        
        // Cache statistics
        $this->cachedStatistics = $stats;
        
        return $stats;
    }
    
    /**
     * Calculate metrics for the current request
     */
    private function calculateMetrics(array $requestData): array
    {
        $metrics = [];
        
        // Calculate request rate (handled internally through counter)
        $requestRateKey = 'request_rate:' . date('Y-m-d-H-i');
        $metrics['request_rate'] = $this->storage->incrementCounter($requestRateKey, 3600);
        
        // Calculate error rate if status code is available
        if (isset($requestData['status_code']) && $requestData['status_code'] >= 400) {
            $errorRateKey = 'error_rate:' . date('Y-m-d-H-i');
            $metrics['error_rate'] = $this->storage->incrementCounter($errorRateKey, 3600);
        }
        
        // Calculate session change rate if session ID is available
        if (isset($requestData['session_id'])) {
            $sessionKey = 'session:' . $requestData['ip'];
            $sessionData = $this->storage->getRateLimit($sessionKey);
            
            if ($sessionData && isset($sessionData['session_id']) && 
                $sessionData['session_id'] !== $requestData['session_id']) {
                
                $sessionChangeKey = 'session_change_rate:' . date('Y-m-d-H');
                $metrics['session_change_rate'] = $this->storage->incrementCounter($sessionChangeKey, 7200);
                
                // Update stored session
                $this->storage->storeRateLimit($sessionKey, [
                    'session_id' => $requestData['session_id'],
                    'updated_at' => time()
                ], 86400);
            } elseif (!$sessionData) {
                // First seen session
                $this->storage->storeRateLimit($sessionKey, [
                    'session_id' => $requestData['session_id'],
                    'updated_at' => time()
                ], 86400);
            }
        }
        
        // Calculate path entropy
        if (isset($requestData['path'])) {
            $pathKey = 'paths:' . $requestData['ip'];
            $pathData = $this->storage->getRateLimit($pathKey);
            
            $paths = [];
            if ($pathData && isset($pathData['paths'])) {
                $paths = $pathData['paths'];
            }
            
            // Add current path
            $paths[] = $requestData['path'];
            
            // Keep only last 20 paths
            if (count($paths) > 20) {
                $paths = array_slice($paths, -20);
            }
            
            // Update paths
            $this->storage->storeRateLimit($pathKey, [
                'paths' => $paths,
                'updated_at' => time()
            ], 86400);
            
            // Calculate Shannon entropy of paths
            $metrics['path_entropy'] = $this->calculateEntropy($paths);
        }
        
        // Calculate parameter count and average length
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            $metrics['parameter_count'] = count($requestData['query_params']);
            
            $totalLength = 0;
            foreach ($requestData['query_params'] as $param => $value) {
                if (is_string($value)) {
                    $totalLength += strlen($value);
                }
            }
            
            $metrics['parameter_length'] = $metrics['parameter_count'] > 0 
                ? $totalLength / $metrics['parameter_count'] 
                : 0;
        }
        
        // Response time if available
        if (isset($requestData['response_time'])) {
            $metrics['response_time'] = $requestData['response_time'];
        }
        
        return $metrics;
    }
    
    /**
     * Store calculated metrics for future analysis
     */
    private function storeMetrics(string $ip, array $metricValues): void
    {
        foreach ($metricValues as $metric => $value) {
            // Skip if metric not defined
            if (!isset($this->metrics[$metric])) {
                continue;
            }
            
            // Store metric
            $this->storage->storeAnomalyData($metric, $value, [
                'ip' => $ip,
                'timestamp' => time()
            ]);
        }
    }
    
    /**
     * Detect anomalies in metric values
     */
    private function detectAnomalies(string $ip, array $metricValues): array
    {
        $anomalies = [];
        
        foreach ($metricValues as $metric => $value) {
            // Skip if metric not defined
            if (!isset($this->metrics[$metric])) {
                continue;
            }
            
            // Get statistics for this metric
            $stats = $this->storage->getMetricStatistics($metric);
            
            // Skip if not enough data
            if (($stats['count'] ?? 0) < 10) {
                continue;
            }
            
            // Calculate Z-score
            $mean = $stats['mean'] ?? 0;
            $stddev = $stats['stddev'] ?? 1;
            
            // Avoid division by zero
            if ($stddev <= 0) {
                $stddev = 0.0001;
            }
            
            $zScore = abs(($value - $mean) / $stddev);
            
            // Check if value is anomalous
            $anomalyThreshold = $this->metrics[$metric]['anomaly_threshold'] ?? 3.0;
            $threshold = $this->zScoreThreshold * ($anomalyThreshold / 3.0);
            
            if ($zScore > $threshold) {
                // Calculate anomaly score based on how far beyond threshold
                $severity = min(1, ($zScore - $threshold) / $threshold);
                $score = $severity * 25; // Max score per anomaly is 25
                
                $anomalies[$metric] = [
                    'value' => $value,
                    'expected_range' => [$mean - ($stddev * $threshold), $mean + ($stddev * $threshold)],
                    'z_score' => $zScore,
                    'score' => $score
                ];
                
                // Store that this was an anomaly
                $this->storage->storeAnomalyData($metric, $value, [
                    'ip' => $ip,
                    'timestamp' => time(),
                    'is_anomaly' => true,
                    'z_score' => $zScore,
                    'threshold' => $threshold
                ]);
                
                $this->logger->borna('Anomaly detected', [
                    'metric' => $metric,
                    'value' => $value,
                    'mean' => $mean,
                    'stddev' => $stddev,
                    'z_score' => $zScore,
                    'threshold' => $threshold,
                    'ip' => $ip
                ]);
            }
        }
        
        return $anomalies;
    }
    
    /**
     * Calculate Shannon entropy of a set of values
     */
    private function calculateEntropy(array $values): float
    {
        if (empty($values)) {
            return 0;
        }
        
        $valueCount = count($values);
        $frequencies = array_count_values($values);
        $entropy = 0;
        
        foreach ($frequencies as $count) {
            $probability = $count / $valueCount;
            $entropy -= $probability * log($probability, 2);
        }
        
        return $entropy;
    }
}