<?php

namespace baseKRIZAN\BORNA\Analyzers;

use baseKRIZAN\Error\Logger;

/**
 * User and request behavior security analyzer
 * Detects suspicious patterns in user behavior
 */
class BehavioralAnalyzer
{
    private Logger $logger;
    private int $securityLevel;
    private array $userHistory = [];
    private array $requestPatterns = [];
    private array $suspiciousBehaviors = [];
    
    // Thresholds for suspicious behavior
    private array $thresholds = [
        // Level 1 (low) thresholds
        1 => [
            'requests_per_minute' => 60,
            'page_visit_speed' => 0.5, // seconds between page visits
            'session_switching' => 10, // session changes
            'failed_logins' => 5,
            'unusual_hours' => false
        ],
        // Level 2 (medium) thresholds
        2 => [
            'requests_per_minute' => 40,
            'page_visit_speed' => 1.0,
            'session_switching' => 5,
            'failed_logins' => 3,
            'unusual_hours' => true
        ],
        // Level 3 (high) thresholds
        3 => [
            'requests_per_minute' => 20,
            'page_visit_speed' => 1.5,
            'session_switching' => 3,
            'failed_logins' => 2,
            'unusual_hours' => true
        ]
    ];
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, int $securityLevel = 1)
    {
        $this->logger = $logger;
        $this->securityLevel = max(1, min(3, $securityLevel)); // Ensure level is 1-3
        
        // Load suspicious behaviors and request patterns
        $this->loadPatterns();
    }
    
    /**
     * Load suspicious behavior patterns
     */
    private function loadPatterns(): void
    {
        // Common suspicious request patterns
        $this->requestPatterns = [
            'admin_scanning' => [
                'pattern' => '/\/(admin|administrator|wp-admin|backend|console|cms)/',
                'description' => 'Scanning for admin interfaces',
                'score' => 35
            ],
            'vulnerability_scanning' => [
                'pattern' => '/\.(old|bak|backup|php~|php-|temp|git|svn|env|config)/',
                'description' => 'Scanning for backup or configuration files',
                'score' => 40
            ],
            'technology_probing' => [
                'pattern' => '/(phpinfo|server-status|crossdomain|clientaccesspolicy)/',
                'description' => 'Probing for server information',
                'score' => 30
            ],
            'auth_bypassing' => [
                'pattern' => '/(\/user\/login|\/wp-login|\/auth\/signin|\/login\.php)\?.*?(bypass|admin=1|auth=0|admin_user)/',
                'description' => 'Attempting authentication bypasses',
                'score' => 60
            ],
            'file_inclusion' => [
                'pattern' => '/\/(include|require|load|read)\?file=/',
                'description' => 'Potential file inclusion attempts',
                'score' => 50
            ],
            'api_abuse' => [
                'pattern' => '/\/api\/(v\d+\/)?(admin|user|auth|token|key)/',
                'description' => 'Potential API abuse attempts',
                'score' => 25
            ]
        ];
        
        // Define suspicious behavioral patterns
        $this->suspiciousBehaviors = [
            'rapid_requests' => [
                'description' => 'High rate of requests in short time',
                'score' => 30
            ],
            'session_anomalies' => [
                'description' => 'Suspicious session behavior',
                'score' => 40
            ],
            'unusual_navigation' => [
                'description' => 'Unusual site navigation pattern',
                'score' => 20
            ],
            'auth_failures' => [
                'description' => 'Multiple authentication failures',
                'score' => 50
            ],
            'unusual_hours' => [
                'description' => 'Activity during unusual hours',
                'score' => 15
            ],
            'rapid_page_cycling' => [
                'description' => 'Cycling through pages too quickly',
                'score' => 25
            ],
            'parameter_tampering' => [
                'description' => 'Manipulating query parameters',
                'score' => 45
            ]
        ];
    }
    
    /**
     * Analyze a request for behavioral anomalies
     * 
     * @param array $requestData Request data
     * @return array Analysis result with score and description
     */
    public function analyze(array $requestData): array
    {
        // Default result with no threat
        $result = [
            'score' => 0,
            'description' => 'No behavioral anomalies detected',
            'details' => []
        ];
        
        if (empty($requestData['ip'])) {
            return $result;
        }
        
        $detectedAnomalies = [];
        $totalScore = 0;
        
        // Check request patterns
        foreach ($this->requestPatterns as $patternName => $patternInfo) {
            if (isset($requestData['path']) && preg_match($patternInfo['pattern'], $requestData['path'])) {
                $detectedAnomalies[] = [
                    'type' => $patternName,
                    'description' => $patternInfo['description'],
                    'score' => $patternInfo['score'],
                    'path' => $requestData['path']
                ];
                
                $totalScore += $patternInfo['score'];
            }
        }
        
        // Update user history for this IP
        $this->updateUserHistory($requestData);
        
        // Check for rapid requests
        if ($this->checkRapidRequests($requestData['ip'])) {
            $detectedAnomalies[] = [
                'type' => 'rapid_requests',
                'description' => $this->suspiciousBehaviors['rapid_requests']['description'],
                'score' => $this->suspiciousBehaviors['rapid_requests']['score'],
                'details' => [
                    'requests_per_minute' => $this->calculateRequestsPerMinute($requestData['ip']),
                    'threshold' => $this->thresholds[$this->securityLevel]['requests_per_minute']
                ]
            ];
            
            $totalScore += $this->suspiciousBehaviors['rapid_requests']['score'];
        }
        
        // Check for session anomalies if session data is available
        if (isset($requestData['session_id'])) {
            if ($this->checkSessionAnomalies($requestData['ip'], $requestData['session_id'])) {
                $detectedAnomalies[] = [
                    'type' => 'session_anomalies',
                    'description' => $this->suspiciousBehaviors['session_anomalies']['description'],
                    'score' => $this->suspiciousBehaviors['session_anomalies']['score'],
                    'details' => [
                        'session_changes' => $this->countSessionChanges($requestData['ip']),
                        'threshold' => $this->thresholds[$this->securityLevel]['session_switching']
                    ]
                ];
                
                $totalScore += $this->suspiciousBehaviors['session_anomalies']['score'];
            }
        }
        
        // Check for rapid page cycling
        if ($this->checkRapidPageCycling($requestData['ip'], $requestData['path'] ?? '')) {
            $detectedAnomalies[] = [
                'type' => 'rapid_page_cycling',
                'description' => $this->suspiciousBehaviors['rapid_page_cycling']['description'],
                'score' => $this->suspiciousBehaviors['rapid_page_cycling']['score'],
                'details' => [
                    'average_visit_time' => $this->calculateAverageVisitTime($requestData['ip']),
                    'threshold' => $this->thresholds[$this->securityLevel]['page_visit_speed']
                ]
            ];
            
            $totalScore += $this->suspiciousBehaviors['rapid_page_cycling']['score'];
        }
        
        // Check for unusual hours if enabled for this security level
        if ($this->thresholds[$this->securityLevel]['unusual_hours'] && $this->checkUnusualHours()) {
            $detectedAnomalies[] = [
                'type' => 'unusual_hours',
                'description' => $this->suspiciousBehaviors['unusual_hours']['description'],
                'score' => $this->suspiciousBehaviors['unusual_hours']['score'],
                'details' => [
                    'current_hour' => (int)date('H'),
                    'unusual_range' => [0, 5] // 12am-5am
                ]
            ];
            
            $totalScore += $this->suspiciousBehaviors['unusual_hours']['score'];
        }
        
        // Check for parameter tampering
        if ($this->checkParameterTampering($requestData)) {
            $detectedAnomalies[] = [
                'type' => 'parameter_tampering',
                'description' => $this->suspiciousBehaviors['parameter_tampering']['description'],
                'score' => $this->suspiciousBehaviors['parameter_tampering']['score']
            ];
            
            $totalScore += $this->suspiciousBehaviors['parameter_tampering']['score'];
        }
        
        // If authentication failures are stored in session and we have that data
        if (isset($requestData['auth_failures']) && 
            $requestData['auth_failures'] >= $this->thresholds[$this->securityLevel]['failed_logins']) {
            
            $detectedAnomalies[] = [
                'type' => 'auth_failures',
                'description' => $this->suspiciousBehaviors['auth_failures']['description'],
                'score' => $this->suspiciousBehaviors['auth_failures']['score'],
                'details' => [
                    'failure_count' => $requestData['auth_failures'],
                    'threshold' => $this->thresholds[$this->securityLevel]['failed_logins']
                ]
            ];
            
            $totalScore += $this->suspiciousBehaviors['auth_failures']['score'];
        }
        
        // If anomalies were detected, update the result
        if (!empty($detectedAnomalies)) {
            $result['score'] = min(100, $totalScore);
            $result['description'] = 'Behavioral anomalies detected';
            $result['details'] = [
                'anomalies' => $detectedAnomalies,
                'security_level' => $this->securityLevel
            ];
            
            // Log significant behavioral anomalies
            if ($totalScore > 50) {
                $this->logger->borna('Significant behavioral anomalies detected', [
                    'ip' => $requestData['ip'],
                    'path' => $requestData['path'] ?? 'unknown',
                    'score' => $totalScore,
                    'anomalies_count' => count($detectedAnomalies)
                ]);
            }
        }
        
        return $result;
    }
    
    /**
     * Update user history with current request
     */
    private function updateUserHistory(array $requestData): void
    {
        $ip = $requestData['ip'];
        $timestamp = microtime(true);
        
        if (!isset($this->userHistory[$ip])) {
            $this->userHistory[$ip] = [
                'first_seen' => $timestamp,
                'last_seen' => $timestamp,
                'request_count' => 0,
                'requests' => [],
                'sessions' => [],
                'pages' => []
            ];
        }
        
        // Update history
        $this->userHistory[$ip]['last_seen'] = $timestamp;
        $this->userHistory[$ip]['request_count']++;
        
        // Add request to history (limited to last 50)
        $this->userHistory[$ip]['requests'][] = [
            'timestamp' => $timestamp,
            'path' => $requestData['path'] ?? '',
            'method' => $requestData['method'] ?? '',
            'session_id' => $requestData['session_id'] ?? ''
        ];
        
        // Limit history size
        if (count($this->userHistory[$ip]['requests']) > 50) {
            array_shift($this->userHistory[$ip]['requests']);
        }
        
        // Track sessions
        if (isset($requestData['session_id']) && !empty($requestData['session_id'])) {
            $sessionId = $requestData['session_id'];
            
            if (!isset($this->userHistory[$ip]['sessions'][$sessionId])) {
                $this->userHistory[$ip]['sessions'][$sessionId] = [
                    'first_seen' => $timestamp,
                    'last_seen' => $timestamp,
                    'page_count' => 0
                ];
            } else {
                $this->userHistory[$ip]['sessions'][$sessionId]['last_seen'] = $timestamp;
                $this->userHistory[$ip]['sessions'][$sessionId]['page_count']++;
            }
        }
        
        // Track pages visited
        if (isset($requestData['path']) && !empty($requestData['path'])) {
            $path = $requestData['path'];
            
            if (!isset($this->userHistory[$ip]['pages'][$path])) {
                $this->userHistory[$ip]['pages'][$path] = [
                    'first_visit' => $timestamp,
                    'last_visit' => $timestamp,
                    'visit_count' => 1
                ];
            } else {
                $this->userHistory[$ip]['pages'][$path]['last_visit'] = $timestamp;
                $this->userHistory[$ip]['pages'][$path]['visit_count']++;
            }
        }
    }
    
    /**
     * Check for rapid requests from an IP
     */
    private function checkRapidRequests(string $ip): bool
    {
        if (!isset($this->userHistory[$ip])) {
            return false;
        }
        
        $requestsPerMinute = $this->calculateRequestsPerMinute($ip);
        
        return $requestsPerMinute > $this->thresholds[$this->securityLevel]['requests_per_minute'];
    }
    
    /**
     * Calculate requests per minute for an IP
     */
    private function calculateRequestsPerMinute(string $ip): float
    {
        if (!isset($this->userHistory[$ip]) || count($this->userHistory[$ip]['requests']) < 2) {
            return 0;
        }
        
        $history = $this->userHistory[$ip]['requests'];
        $recentRequests = array_filter($history, function($req) {
            return (microtime(true) - $req['timestamp']) <= 60; // Last minute
        });
        
        return count($recentRequests);
    }
    
    /**
     * Check for session anomalies (frequent session switching)
     */
    private function checkSessionAnomalies(string $ip, string $sessionId): bool
    {
        if (!isset($this->userHistory[$ip])) {
            return false;
        }
        
        $sessionChanges = $this->countSessionChanges($ip);
        
        return $sessionChanges > $this->thresholds[$this->securityLevel]['session_switching'];
    }
    
    /**
     * Count session changes for an IP
     */
    private function countSessionChanges(string $ip): int
    {
        if (!isset($this->userHistory[$ip]) || count($this->userHistory[$ip]['requests']) < 2) {
            return 0;
        }
        
        $history = $this->userHistory[$ip]['requests'];
        $sessionChanges = 0;
        $lastSessionId = null;
        
        foreach ($history as $req) {
            if (isset($req['session_id']) && $req['session_id'] !== $lastSessionId) {
                if ($lastSessionId !== null) {
                    $sessionChanges++;
                }
                $lastSessionId = $req['session_id'];
            }
        }
        
        return $sessionChanges;
    }
    
    /**
     * Check for rapid page cycling behavior
     */
    private function checkRapidPageCycling(string $ip, string $path): bool
    {
        if (!isset($this->userHistory[$ip]) || count($this->userHistory[$ip]['requests']) < 3) {
            return false;
        }
        
        $avgVisitTime = $this->calculateAverageVisitTime($ip);
        
        return $avgVisitTime < $this->thresholds[$this->securityLevel]['page_visit_speed'];
    }
    
    /**
     * Calculate average visit time between pages
     */
    private function calculateAverageVisitTime(string $ip): float
    {
        if (!isset($this->userHistory[$ip]) || count($this->userHistory[$ip]['requests']) < 2) {
            return 999; // High value to indicate no issue
        }
        
        $history = $this->userHistory[$ip]['requests'];
        $visitTimes = [];
        
        for ($i = 1; $i < count($history); $i++) {
            // Only count different pages
            if ($history[$i]['path'] !== $history[$i-1]['path']) {
                $visitTimes[] = $history[$i]['timestamp'] - $history[$i-1]['timestamp'];
            }
        }
        
        if (empty($visitTimes)) {
            return 999; // No different page visits
        }
        
        return array_sum($visitTimes) / count($visitTimes);
    }
    
    /**
     * Check for unusual hour activity
     */
    private function checkUnusualHours(): bool
    {
        $hour = (int)date('H');
        
        // Consider 12am to 5am as unusual hours (adjust as needed)
        return ($hour >= 0 && $hour < 5);
    }
    
    /**
     * Check for parameter tampering
     */
    private function checkParameterTampering(array $requestData): bool
    {
        // Check for typical parameter tampering patterns
        $suspiciousParams = [
            'id' => '/^-?\d+$/', // Negative IDs
            'price' => '/^0(\.0+)?$/', // Zero price
            'quantity' => '/^-\d+$|^0$/', // Negative or zero quantity
            'admin' => '/^(true|1|yes)$/i', // Forcing admin flags
            'debug' => '/^(true|1|yes)$/i', // Debug flags
            'role' => '/^(admin|administrator|root|superuser)$/i' // Role tampering
        ];
        
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            foreach ($suspiciousParams as $param => $pattern) {
                if (isset($requestData['query_params'][$param]) && 
                    preg_match($pattern, $requestData['query_params'][$param])) {
                    return true;
                }
            }
        }
        
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($suspiciousParams as $param => $pattern) {
                if (isset($requestData['post_params'][$param]) && 
                    preg_match($pattern, $requestData['post_params'][$param])) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Set security level
     */
    public function setSecurityLevel(int $level): void
    {
        $this->securityLevel = max(1, min(3, $level));
    }
    
    /**
     * Get security level
     */
    public function getSecurityLevel(): int
    {
        return $this->securityLevel;
    }
    
    /**
     * Get history for an IP address
     */
    public function getIPHistory(string $ip): ?array
    {
        return $this->userHistory[$ip] ?? null;
    }
    
    /**
     * Clear history for an IP address
     */
    public function clearIPHistory(string $ip): void
    {
        if (isset($this->userHistory[$ip])) {
            unset($this->userHistory[$ip]);
        }
    }
    
    /**
     * Get all user histories
     */
    public function getAllHistories(): array
    {
        return $this->userHistory;
    }
    
    /**
     * Clean up old history entries
     */
    public function cleanupHistories(int $maxAgeMinutes = 60): int
    {
        $cutoffTime = microtime(true) - ($maxAgeMinutes * 60);
        $removed = 0;
        
        foreach ($this->userHistory as $ip => $history) {
            if ($history['last_seen'] < $cutoffTime) {
                unset($this->userHistory[$ip]);
                $removed++;
            }
        }
        
        return $removed;
    }
}