<?php

namespace baseKRIZAN\BORNA\Analyzers;

use baseKRIZAN\Error\Logger;

/**
 * Geolocation-based security analyzer
 * Checks IP addresses against country lists
 */
class GeolocationAnalyzer
{
    private Logger $logger;
    private array $allowedCountries = [];
    private array $blockedCountries = [];
    private array $geoCache = [];
    private bool $enabled = false;
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, array $allowedCountries = [], array $blockedCountries = [])
    {
        $this->logger = $logger;
        $this->allowedCountries = array_map('strtoupper', array_filter($allowedCountries));
        $this->blockedCountries = array_map('strtoupper', array_filter($blockedCountries));
        
        // Enable only if we have countries configured
        $this->enabled = !empty($this->allowedCountries) || !empty($this->blockedCountries);
        
        if ($this->enabled) {
            $this->logger->borna('Geolocation analyzer initialized', [
                'allowed_countries' => $this->allowedCountries,
                'blocked_countries' => $this->blockedCountries
            ]);
        }
    }
    
    /**
     * Analyze a request based on geolocation
     * 
     * @param array $requestData Request data
     * @return array Analysis result with score and description
     */
    public function analyze(array $requestData): array
    {
        // Default result with no threat
        $result = [
            'score' => 0,
            'description' => 'No geo-location threats detected',
            'details' => []
        ];
        
        // Skip if not enabled or no IP provided
        if (!$this->enabled || empty($requestData['ip'])) {
            return $result;
        }
        
        // Skip localhost or internal IPs
        if ($this->isInternalIP($requestData['ip'])) {
            $result['description'] = 'Internal IP address, skipping geolocation check';
            return $result;
        }
        
        // Get country code for IP
        $geoData = $this->getIPGeoLocation($requestData['ip']);
        
        if (!$geoData || empty($geoData['country_code'])) {
            $result['description'] = 'Unable to determine IP location';
            return $result;
        }
        
        $countryCode = strtoupper($geoData['country_code']);
        
        // Store geolocation data in result
        $result['details'] = [
            'ip' => $requestData['ip'],
            'country_code' => $countryCode,
            'country_name' => $geoData['country_name'] ?? 'Unknown',
            'city' => $geoData['city'] ?? 'Unknown',
            'latitude' => $geoData['latitude'] ?? 0,
            'longitude' => $geoData['longitude'] ?? 0,
        ];
        
        // Check against blocked countries
        if (!empty($this->blockedCountries) && in_array($countryCode, $this->blockedCountries)) {
            $result['score'] = 100;
            $result['description'] = 'IP originates from blocked country';
            
            $this->logger->borna('Access attempt from blocked country', [
                'ip' => $requestData['ip'],
                'country_code' => $countryCode,
                'path' => $requestData['path'] ?? 'unknown'
            ]);
            
            return $result;
        }
        
        // Check against allowed countries (whitelist approach)
        if (!empty($this->allowedCountries) && !in_array($countryCode, $this->allowedCountries)) {
            $result['score'] = 80;
            $result['description'] = 'IP originates from non-allowed country';
            
            $this->logger->borna('Access attempt from non-allowed country', [
                'ip' => $requestData['ip'],
                'country_code' => $countryCode,
                'path' => $requestData['path'] ?? 'unknown'
            ]);
            
            return $result;
        }
        
        return $result;
    }
    
    /**
     * Check if an IP is internal/private
     * 
     * @param string $ip IP address to check
     * @return bool True if internal
     */
    private function isInternalIP(string $ip): bool
    {
        // Check for localhost
        if ($ip === '127.0.0.1' || $ip === '::1' || $ip === 'localhost') {
            return true;
        }
        
        // Check for IPv4 private ranges
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $privateRanges = [
                '10.0.0.0|10.255.255.255',     // 10.0.0.0/8
                '172.16.0.0|172.31.255.255',   // 172.16.0.0/12
                '192.168.0.0|192.168.255.255', // 192.168.0.0/16
                '169.254.0.0|169.254.255.255', // 169.254.0.0/16
                '127.0.0.0|127.255.255.255'    // 127.0.0.0/8
            ];
            
            $ipLong = ip2long($ip);
            
            foreach ($privateRanges as $range) {
                list($start, $end) = explode('|', $range);
                if ($ipLong >= ip2long($start) && $ipLong <= ip2long($end)) {
                    return true;
                }
            }
        }
        
        // Check for IPv6 private ranges
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            // Check if it's a local IPv6 address
            if (substr($ip, 0, 2) === 'fc' || substr($ip, 0, 2) === 'fd') {
                return true; // fc00::/7 (Unique Local Addresses)
            }
            
            // Check for link-local addresses
            if (substr($ip, 0, 4) === 'fe80') {
                return true; // fe80::/10 (Link-local addresses)
            }
        }
        
        return false;
    }
    
    /**
     * Get geolocation data for an IP address
     * 
     * @param string $ip IP address
     * @return array|null Geolocation data or null if not available
     */
    private function getIPGeoLocation(string $ip): ?array
    {
        // Return from cache if available
        if (isset($this->geoCache[$ip])) {
            return $this->geoCache[$ip];
        }
        
        // Try to use a simplified approach based on IP ranges
        // This is a fallback since geoip functions might not be available
        
        // Create a basic result with defaults
        $geoData = [
            'country_code' => 'XX', // Unknown
            'country_name' => 'Unknown',
            'city' => 'Unknown',
            'latitude' => 0,
            'longitude' => 0
        ];
        
        // Detect common IP patterns for major regions
        if (preg_match('/^192\.168\./', $ip) || preg_match('/^10\./', $ip) || preg_match('/^172\.(1[6-9]|2[0-9]|3[0-1])\./', $ip)) {
            $geoData['country_code'] = 'LO'; // Local
            $geoData['country_name'] = 'Local Network';
        }
        
        // Cache result
        $this->geoCache[$ip] = $geoData;
        return $geoData;
    }
    
    /**
     * Set allowed countries
     */
    public function setAllowedCountries(array $countries): void
    {
        $this->allowedCountries = array_map('strtoupper', array_filter($countries));
        $this->enabled = !empty($this->allowedCountries) || !empty($this->blockedCountries);
    }
    
    /**
     * Set blocked countries
     */
    public function setBlockedCountries(array $countries): void
    {
        $this->blockedCountries = array_map('strtoupper', array_filter($countries));
        $this->enabled = !empty($this->allowedCountries) || !empty($this->blockedCountries);
    }
    
    /**
     * Get allowed countries
     */
    public function getAllowedCountries(): array
    {
        return $this->allowedCountries;
    }
    
    /**
     * Get blocked countries
     */
    public function getBlockedCountries(): array
    {
        return $this->blockedCountries;
    }
}