<?php

namespace baseKRIZAN\BORNA\Analyzers;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\BORNA\Interfaces\AnalyzerInterface;
use baseKRIZAN\BORNA\Enum\SecurityLevel;
use baseKRIZAN\BORNA\Storage\StorageInterface;

/**
 * File and system integrity monitoring
 * Detects unauthorized changes to critical files and configurations
 */
class IntegrityAnalyzer implements AnalyzerInterface
{
    /**
     * Logger instance
     */
    private Logger $logger;
    
    /**
     * Storage interface
     */
    private StorageInterface $storage;
    
    /**
     * Security level
     */
    private SecurityLevel $securityLevel = SecurityLevel::MEDIUM;
    
    /**
     * Critical file patterns to monitor
     */
    private array $criticalFilePaths = [
        '*/config/*.php',
        '*/app/Controllers/*.php',
        '*/app/Models/*.php',
        '*/app/baseKRIZAN/Security/*.php',
        '*/app/baseKRIZAN/Http/Middleware/*.php',
        '*/bootstrap.php',
        '*/index.php',
        '*/composer.json',
        '*/composer.lock',
        '.htaccess',
        'web.config'
    ];
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger, 
        StorageInterface $storage,
        SecurityLevel $securityLevel = SecurityLevel::MEDIUM
    ) {
        $this->logger = $logger;
        $this->storage = $storage;
        $this->securityLevel = $securityLevel;
        
        $this->adjustMonitoredFiles();
        
        $this->logger->borna('IntegrityAnalyzer initialized', [
            'monitored_patterns' => count($this->criticalFilePaths),
            'security_level' => $this->securityLevel->name
        ]);
    }
    
    /**
     * @inheritDoc
     */
    public function analyze(array $requestData): array
    {
        // Default result with no threat
        $result = [
            'score' => 0,
            'description' => 'No integrity issues detected',
            'details' => []
        ];
        
        // Check if last check was too recent
        $lastCheck = $this->getLastCheckTimestamp();
        if (time() - $lastCheck < 3600) {
            return $result;
        }
        
        // Run integrity check
        $integrityIssues = $this->runIntegrityCheck();
        
        if (!empty($integrityIssues)) {
            // Calculate severity based on number and importance of changed files
            $totalScore = min(100, count($integrityIssues) * 15);
            
            $result = [
                'score' => $totalScore,
                'description' => 'File integrity issues detected',
                'details' => [
                    'issues' => $integrityIssues,
                    'check_time' => date('Y-m-d H:i:s'),
                    'next_check' => date('Y-m-d H:i:s', time() + 3600)
                ]
            ];
            
            // Log integrity issues
            $this->logger->borna('File integrity issues detected', [
                'issues_count' => count($integrityIssues),
                'files' => array_column($integrityIssues, 'file'),
                'score' => $totalScore
            ]);
        }
        
        // Update last check timestamp
        $this->updateLastCheckTimestamp();
        
        return $result;
    }
    
    /**
     * @inheritDoc
     */
    public function getName(): string
    {
        return 'integrity';
    }
    
    /**
     * @inheritDoc
     */
    public function setSecurityLevel(int $level): void
    {
        $this->securityLevel = SecurityLevel::from($level);
        
        // Adjust monitored files based on security level
        $this->adjustMonitoredFiles();
        
        $this->logger->borna('IntegrityAnalyzer security level changed', [
            'new_level' => $this->securityLevel->name,
            'monitored_patterns' => count($this->criticalFilePaths)
        ]);
    }
    
    /**
     * @inheritDoc
     */
    public function getStatistics(): array
    {
        return [
            'monitored_file_patterns' => $this->criticalFilePaths,
            'last_check' => date('Y-m-d H:i:s', $this->getLastCheckTimestamp()),
            'next_check' => date('Y-m-d H:i:s', $this->getLastCheckTimestamp() + 3600),
            'security_level' => $this->securityLevel->name
        ];
    }
    
    /**
     * Adjust monitored files based on security level
     */
    private function adjustMonitoredFiles(): void
    {
        // Basic monitoring for low security level
        $basePatterns = [
            '*/config/*.php',
            '*/app/baseKRIZAN/Security/*.php',
            '*/bootstrap.php',
            '*/index.php'
        ];
        
        // Add more files for medium security level
        $mediumPatterns = [
            '*/app/Controllers/*.php',
            '*/app/Models/*.php',
            '*/app/baseKRIZAN/Http/Middleware/*.php',
            '*/composer.json',
            '*/composer.lock',
        ];
        
        // Add even more files for high security level
        $highPatterns = [
            '*/app/Views/*.php',
            '*/app/*.php',
            '*/public/*.php',
            '.htaccess',
            'web.config',
            '*/vendor/autoload.php'
        ];
        
        // Set patterns based on security level
        $this->criticalFilePaths = match($this->securityLevel) {
            SecurityLevel::LOW => $basePatterns,
            SecurityLevel::MEDIUM => array_merge($basePatterns, $mediumPatterns),
            SecurityLevel::HIGH => array_merge($basePatterns, $mediumPatterns, $highPatterns),
        };
    }
    
    /**
     * Get last integrity check timestamp from storage
     */
    private function getLastCheckTimestamp(): int
    {
        $data = $this->storage->getRateLimit('integrity_last_check');
        
        return $data['timestamp'] ?? 0;
    }
    
    /**
     * Update last check timestamp in storage
     */
    private function updateLastCheckTimestamp(): void
    {
        $timestamp = time();
        
        $this->storage->storeRateLimit('integrity_last_check', [
            'timestamp' => $timestamp
        ], 86400 * 7); // Čuvaj 7 dana
    }
    
    /**
     * Run integrity check of critical files
     * 
     * @return array Detected integrity issues
     */
    private function runIntegrityCheck(): array
    {
        $issues = [];
        $appRoot = defined('APP_ROOT') ? APP_ROOT : dirname(__DIR__, 5);
        
        foreach ($this->criticalFilePaths as $pattern) {
            // Convert glob pattern to absolute paths
            $files = glob($appRoot . '/' . $pattern);
            
            if (!$files) {
                // Try without appRoot if no matches
                $files = glob($pattern);
            }
            
            if (!$files) {
                continue;
            }
            
            foreach ($files as $file) {
                if (!file_exists($file) || !is_file($file)) {
                    continue;
                }
                
                $currentHash = $this->calculateFileHash($file);
                $storedHash = $this->storage->getIntegrityHash($file);
                
                if ($storedHash === null) {
                    // First time seeing this file, store its hash
                    $this->storage->storeIntegrityHash($file, $currentHash);
                } else if ($currentHash !== $storedHash) {
                    // Hash mismatch, file has changed
                    $issues[] = [
                        'file' => $file,
                        'type' => 'modified',
                        'stored_hash' => $storedHash,
                        'current_hash' => $currentHash,
                        'modified_time' => date('Y-m-d H:i:s', filemtime($file))
                    ];
                    
                    // Update stored hash
                    $this->storage->storeIntegrityHash($file, $currentHash);
                }
            }
        }
        
        return $issues;
    }
    
    /**
     * Perform a full system integrity check
     * 
     * @return array Results of the integrity check
     */
    public function performFullSystemCheck(): array
    {
        $this->logger->borna('Starting full system integrity check');
        
        $startTime = microtime(true);
        $appRoot = defined('APP_ROOT') ? APP_ROOT : dirname(__DIR__, 5);
        
        $checkedFiles = 0;
        $newFiles = 0;
        $modifiedFiles = 0;
        $missingFiles = 0;
        $issues = [];
        
        // Check all existing files
        foreach ($this->criticalFilePaths as $pattern) {
            // Convert glob pattern to absolute paths
            $files = glob($appRoot . '/' . $pattern);
            
            if (!$files) {
                // Try without appRoot if no matches
                $files = glob($pattern);
            }
            
            if (!$files) {
                continue;
            }
            
            foreach ($files as $file) {
                if (!file_exists($file) || !is_file($file)) {
                    continue;
                }
                
                $checkedFiles++;
                $currentHash = $this->calculateFileHash($file);
                $storedHash = $this->storage->getIntegrityHash($file);
                
                if ($storedHash === null) {
                    // First time seeing this file, store its hash
                    $this->storage->storeIntegrityHash($file, $currentHash);
                    $newFiles++;
                    
                    $issues[] = [
                        'file' => $file,
                        'type' => 'new',
                        'current_hash' => $currentHash,
                        'modified_time' => date('Y-m-d H:i:s', filemtime($file))
                    ];
                } else if ($currentHash !== $storedHash) {
                    // Hash mismatch, file has changed
                    $modifiedFiles++;
                    
                    $issues[] = [
                        'file' => $file,
                        'type' => 'modified',
                        'stored_hash' => $storedHash,
                        'current_hash' => $currentHash,
                        'modified_time' => date('Y-m-d H:i:s', filemtime($file))
                    ];
                    
                    // Update the hash to the new value
                    $this->storage->storeIntegrityHash($file, $currentHash);
                }
            }
        }
        
        $duration = microtime(true) - $startTime;
        
        $this->logger->borna('Completed full system integrity check', [
            'duration_seconds' => round($duration, 2),
            'checked_files' => $checkedFiles,
            'new_files' => $newFiles,
            'modified_files' => $modifiedFiles,
            'missing_files' => $missingFiles
        ]);
        
        // Update last check timestamp
        $this->updateLastCheckTimestamp();
        
        return [
            'status' => 'completed',
            'duration_seconds' => round($duration, 2),
            'checked_files' => $checkedFiles,
            'new_files' => $newFiles,
            'modified_files' => $modifiedFiles,
            'missing_files' => $missingFiles,
            'issues' => $issues,
            'timestamp' => date('Y-m-d H:i:s'),
            'security_level' => $this->securityLevel->name
        ];
    }
    
    /**
     * Calculate a secure hash for a file
     */
    private function calculateFileHash(string $filePath): string
    {
        if (!file_exists($filePath) || !is_readable($filePath)) {
            return '';
        }
        
        return hash_file('sha256', $filePath);
    }
}