<?php

namespace baseKRIZAN\BORNA\Analyzers;

use baseKRIZAN\Error\Logger;

/**
 * Security analyzer for detecting specific attack patterns
 * Focuses on common attack vectors like XSS, SQLi, etc.
 */
class PatternAnalyzer
{
    private Logger $logger;
    private bool $xssProtection;
    private bool $sqliProtection;
    private array $patterns = [];
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, bool $xssProtection = true, bool $sqliProtection = true)
    {
        $this->logger = $logger;
        $this->xssProtection = $xssProtection;
        $this->sqliProtection = $sqliProtection;
        
        // Initialize attack patterns
        $this->initializePatterns();
    }
    
    /**
     * Initialize attack pattern definitions
     */
    private function initializePatterns(): void
    {
        // Define attack patterns for different attack types
        $this->patterns = [
            // XSS (Cross-Site Scripting) patterns
            'xss' => [
                // Basic script tags and javascript: protocol
                '/(<script[^>]*>.*?<\/script>)/is',
                '/(javascript\s*:)/i',
                
                // Event handlers
                '/\b(on(blur|change|click|dblclick|focus|keydown|keypress|keyup|load|mousedown|mousemove|mouseout|mouseover|mouseup|reset|select|submit|unload))\s*=/i',
                
                // Common JS functions
                '/\b(alert|confirm|prompt|console\.(log|error)|document\.cookie|document\.write|eval|setTimeout|setInterval|new\s+Function)\s*\(/i',
                
                // DOM manipulation
                '/((document|window)\.(location|open))/i',
                '/((document|element)\.inner(HTML|Text))/i',
                
                // Encoded attacks
                '/(&#x[0-9a-f]+;)/i', // hex encoding
                '/(&#[0-9]+;)/i', // decimal encoding
                
                // Data URI with script content
                '/(data\s*:\s*text\/html)/i',
                
                // Less common vectors
                '/(expression|behavior|moz-binding)/i'
            ],
            
            // SQL Injection patterns
            'sqli' => [
                // Common SQL commands
                '/(UNION\s+SELECT|SELECT\s+.*?\s+FROM|INSERT\s+INTO|UPDATE\s+.*?\s+SET|DELETE\s+FROM|DROP\s+TABLE|ALTER\s+TABLE|EXECUTE\s+.*?|EXEC\s+.*?|UNION\s+ALL\s+SELECT)/i',
                
                // SQL comments
                '/(--\s|\/\*.*?\*\/|#.*?$)/i',
                
                // Authentication bypass
                "/('\s*OR\s*'1'\s*=\s*'1)/i",
                '/(1\s*=\s*1)/i',
                
                // Stacked queries
                '/(;\s*SELECT|;\s*INSERT|;\s*UPDATE|;\s*DELETE|;\s*DROP)/i',
                
                // Blind SQL injection time delays
                '/(SLEEP\s*\(|BENCHMARK\s*\(|PG_SLEEP\s*\(|WAITFOR\s+DELAY)/i',
                
                // DB specific functions
                '/(LOAD_FILE\s*\(|INTO\s+OUTFILE|INTO\s+DUMPFILE)/i'
            ],
            
            // Remote/Local File Inclusion
            'file_inclusion' => [
                // Path traversal
                '/(\.\.\/|\.\.\\\\|%2e%2e%2f|%252e%252e%252f)/i',
                
                // Remote file inclusion
                '/(php|data|zlib|phar|ssh2|file|glob|expect):\/\//i',
                
                // Common target files
                '/(\/etc\/passwd|boot\.ini|win\.ini|system\.ini|config\.sys)/i',
                
                // PHP wrappers
                '/(php:\/\/filter|php:\/\/input|phar:\/\/|zip:\/\/)/i'
            ],
            
            // Command Injection
            'command_injection' => [
                // Command separators
                '/\s[;&|`]\s|\$\(|\$\{/i',
                
                // Common commands
                '/(ping|wget|curl|nc|netcat|bash|sh|cmd|powershell)(\s+-|\s+\/)/i',
                
                // Command options
                '/(\/bin\/sh|\/bin\/bash|cmd\.exe)/i',
                
                // Piping and redirection
                '/(\s*>\s*[^\s]+|\s*>>\s*[^\s]+|\s*<\s*[^\s]+|\s*\|\s*[^\s]+)/i'
            ],
            
            // SSRF (Server-Side Request Forgery)
            'ssrf' => [
                // Internal addresses
                '/(localhost|127\.0\.0\.1|0\.0\.0\.0|::1)/i',
                
                // Internal networks
                '/(^10\.|^172\.(1[6-9]|2[0-9]|3[0-1])\.|^192\.168\.|^169\.254\.)/i',
                
                // Alternative representations
                '/(2130706433|3232235521|0x7f000001)/i',
                
                // Domain + port specifications
                '/(^|\.)internal\.(com|net|org)|example\.(com|net|org)/i'
            ],
            
            // CSRF (Cross-Site Request Forgery)
            'csrf' => [
                // Hidden form elements for CSRF
                '/(<input[^>]*type\s*=\s*[\'"]hidden[\'"][^>]*value\s*=\s*[\'"][^"\']*(?:http|www)[^"\']*[\'"][^>]*>)/i'
            ],
            
            // Open Redirect
            'open_redirect' => [
                // URL parameters that might lead to open redirects
                '/(redirect|return|returnTo|return_to|goto|url|next|target|dest|destination)=/i'
            ],
            
            // Directory Traversal
            'directory_traversal' => [
                // Path traversal sequences
                '/(\.\.\/|\.\.\\\\|%2e%2e%2f|%2e%2e\/|\.\.%2f|%2e%2e%5c|\.\.\\\\|\.\.%5c|\.\.%255c)/i',
                
                // Encoded variants (pojednostavljeno za PCRE2 kompatibilnost)
                '/(\.\.u2215|\.\.u2216|\.\.%u2215|\.\.%u2216)/i'
            ]
        ];
    }
    
    /**
     * Analyze a request for attack patterns
     * 
     * @param array $requestData Request data
     * @return array Analysis result with score and description
     */
    public function analyze(array $requestData): array
    {
        // Default result with no threat
        $result = [
            'score' => 0,
            'description' => 'No attack patterns detected',
            'details' => []
        ];
        
        $matches = [];
        $totalScore = 0;
        
        // Skip disabled protections
        $activePatterns = $this->patterns;
        if (!$this->xssProtection) {
            unset($activePatterns['xss']);
        }
        if (!$this->sqliProtection) {
            unset($activePatterns['sqli']);
        }
        
        // Analyze query parameters
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            foreach ($requestData['query_params'] as $param => $value) {
                if (is_string($value)) {
                    $this->scanForPatterns($activePatterns, $value, 'query', $param, $matches);
                }
            }
        }
        
        // Analyze post parameters
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($requestData['post_params'] as $param => $value) {
                if (is_string($value)) {
                    $this->scanForPatterns($activePatterns, $value, 'post', $param, $matches);
                }
            }
        }
        
        // Analyze cookie values
        if (isset($requestData['cookies']) && is_array($requestData['cookies'])) {
            foreach ($requestData['cookies'] as $name => $value) {
                if (is_string($value)) {
                    $this->scanForPatterns($activePatterns, $value, 'cookie', $name, $matches);
                }
            }
        }
        
        // Analyze headers for potential attacks
        if (isset($requestData['headers']) && is_array($requestData['headers'])) {
            $sensitiveHeaders = ['Referer', 'User-Agent', 'X-Forwarded-For', 'Origin'];
            
            foreach ($sensitiveHeaders as $header) {
                if (isset($requestData['headers'][$header]) && is_string($requestData['headers'][$header])) {
                    $this->scanForPatterns($activePatterns, $requestData['headers'][$header], 'header', $header, $matches);
                }
            }
        }
        
        // Analyze request path
        if (isset($requestData['path']) && is_string($requestData['path'])) {
            $this->scanForPatterns($activePatterns, $requestData['path'], 'path', 'path', $matches);
        }
        
        // Calculate score based on matched patterns
        $attackTypes = [];
        
        foreach ($matches as $match) {
            switch ($match['attack_type']) {
                case 'xss':
                    if (!isset($attackTypes['xss'])) {
                        $attackTypes['xss'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['xss']['score'] += 15;
                    $attackTypes['xss']['count']++;
                    $attackTypes['xss']['matches'][] = $match;
                    break;
                    
                case 'sqli':
                    if (!isset($attackTypes['sqli'])) {
                        $attackTypes['sqli'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['sqli']['score'] += 20;
                    $attackTypes['sqli']['count']++;
                    $attackTypes['sqli']['matches'][] = $match;
                    break;
                    
                case 'file_inclusion':
                    if (!isset($attackTypes['file_inclusion'])) {
                        $attackTypes['file_inclusion'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['file_inclusion']['score'] += 25;
                    $attackTypes['file_inclusion']['count']++;
                    $attackTypes['file_inclusion']['matches'][] = $match;
                    break;
                    
                case 'command_injection':
                    if (!isset($attackTypes['command_injection'])) {
                        $attackTypes['command_injection'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['command_injection']['score'] += 30;
                    $attackTypes['command_injection']['count']++;
                    $attackTypes['command_injection']['matches'][] = $match;
                    break;
                    
                case 'ssrf':
                    if (!isset($attackTypes['ssrf'])) {
                        $attackTypes['ssrf'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['ssrf']['score'] += 20;
                    $attackTypes['ssrf']['count']++;
                    $attackTypes['ssrf']['matches'][] = $match;
                    break;
                    
                case 'csrf':
                    if (!isset($attackTypes['csrf'])) {
                        $attackTypes['csrf'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['csrf']['score'] += 15;
                    $attackTypes['csrf']['count']++;
                    $attackTypes['csrf']['matches'][] = $match;
                    break;
                    
                case 'open_redirect':
                    if (!isset($attackTypes['open_redirect'])) {
                        $attackTypes['open_redirect'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['open_redirect']['score'] += 10;
                    $attackTypes['open_redirect']['count']++;
                    $attackTypes['open_redirect']['matches'][] = $match;
                    break;
                    
                case 'directory_traversal':
                    if (!isset($attackTypes['directory_traversal'])) {
                        $attackTypes['directory_traversal'] = [
                            'score' => 0,
                            'count' => 0,
                            'matches' => []
                        ];
                    }
                    $attackTypes['directory_traversal']['score'] += 20;
                    $attackTypes['directory_traversal']['count']++;
                    $attackTypes['directory_traversal']['matches'][] = $match;
                    break;
            }
        }
        
        // Calculate total score and prepare result
        if (!empty($attackTypes)) {
            $detectedAttacks = [];
            
            foreach ($attackTypes as $type => $data) {
                // Cap score for each attack type at certain levels to prevent single attack type from maxing out
                $cappedScore = min($data['score'], 60); // Cap at 60 points per attack type
                $totalScore += $cappedScore;
                
                $detectedAttacks[] = [
                    'type' => $type,
                    'score' => $cappedScore,
                    'count' => $data['count'],
                    'matches' => array_slice($data['matches'], 0, 5) // Limit matches in response
                ];
            }
            
            // Cap total score at 100
            $totalScore = min($totalScore, 100);
            
            $result['score'] = $totalScore;
            $result['description'] = 'Attack patterns detected';
            $result['details'] = [
                'attacks' => $detectedAttacks,
                'match_count' => count($matches)
            ];
            
            // Log high severity attacks
            if ($totalScore >= 50) {
                $this->logger->borna('High severity attack patterns detected', [
                    'ip' => $requestData['ip'] ?? 'unknown',
                    'path' => $requestData['path'] ?? 'unknown',
                    'score' => $totalScore,
                    'attack_types' => array_keys($attackTypes)
                ]);
            }
        }
        
        return $result;
    }
    
    /**
     * Scan a string value for attack patterns
     * 
     * @param array $patterns Patterns to check against
     * @param string $value Value to scan
     * @param string $source Source of the value (query, post, cookie, etc.)
     * @param string $name Name of the parameter
     * @param array &$matches Array to store matches
     */
    private function scanForPatterns(array $patterns, string $value, string $source, string $name, array &$matches): void
    {
        foreach ($patterns as $attackType => $patternList) {
            foreach ($patternList as $pattern) {
                try {
                    // Validiraj pattern prije korištenja
                    if (@preg_match($pattern, 'test') === false) {
                        $this->logger->borna('Invalid regex pattern skipped', [
                            'pattern' => $pattern,
                            'error' => preg_last_error_msg()
                        ]);
                        continue; // Preskoči neispravne patterne
                    }
                    
                    if (preg_match($pattern, $value, $patternMatches)) {
                        $matches[] = [
                            'attack_type' => $attackType,
                            'source' => $source,
                            'name' => $name,
                            'pattern' => $pattern,
                            'matched' => $patternMatches[0],
                            'value' => $this->truncateValue($value)
                        ];
                    }
                } catch (\Exception $e) {
                    $this->logger->error('Error in pattern scanning', [
                        'pattern' => $pattern,
                        'error' => $e->getMessage()
                    ]);
                    continue; // Preskoči u slučaju greške
                }
            }
        }
    }
    
    /**
     * Truncate a value for safe logging
     * 
     * @param string $value Value to truncate
     * @param int $maxLength Maximum length
     * @return string Truncated value
     */
    private function truncateValue(string $value, int $maxLength = 100): string
    {
        if (strlen($value) <= $maxLength) {
            return $value;
        }
        
        return substr($value, 0, $maxLength) . '...';
    }
    
    /**
     * Enable or disable XSS protection
     */
    public function setXssProtection(bool $enabled): void
    {
        $this->xssProtection = $enabled;
    }
    
    /**
     * Enable or disable SQLi protection
     */
    public function setSqliProtection(bool $enabled): void
    {
        $this->sqliProtection = $enabled;
    }
    
    /**
     * Check if XSS protection is enabled
     */
    public function isXssProtectionEnabled(): bool
    {
        return $this->xssProtection;
    }
    
    /**
     * Check if SQLi protection is enabled
     */
    public function isSqliProtectionEnabled(): bool
    {
        return $this->sqliProtection;
    }
    
    /**
     * Add a custom pattern for a specific attack type
     * 
     * @param string $attackType Attack type
     * @param string $pattern Regex pattern
     * @return bool Success status
     */
    public function addCustomPattern(string $attackType, string $pattern): bool
    {
        if (!isset($this->patterns[$attackType])) {
            return false;
        }
        
        // Validate pattern by testing it
        try {
            preg_match($pattern, 'test');
            $this->patterns[$attackType][] = $pattern;
            
            $this->logger->borna('Added custom pattern', [
                'attack_type' => $attackType,
                'pattern' => $pattern
            ]);
            
            return true;
        } catch (\Exception $e) {
            $this->logger->error('Invalid pattern format', [
                'pattern' => $pattern,
                'error' => $e->getMessage()
            ]);
            
            return false;
        }
    }
    
    /**
     * Get all patterns for a specific attack type
     */
    public function getPatterns(string $attackType = null): array
    {
        if ($attackType !== null) {
            return $this->patterns[$attackType] ?? [];
        }
        
        return $this->patterns;
    }
}