<?php

namespace baseKRIZAN\BORNA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Services\Container;
use baseKRIZAN\Http\Middleware\MiddlewareRegistry;
use baseKRIZAN\BORNA\ThreatDetector;
use baseKRIZAN\BORNA\FirewallRules;
use baseKRIZAN\BORNA\SecurityEvents;
use baseKRIZAN\BORNA\Analyzers\GeolocationAnalyzer;
use baseKRIZAN\BORNA\Analyzers\BehavioralAnalyzer;
use baseKRIZAN\BORNA\Analyzers\PatternAnalyzer;
use baseKRIZAN\BORNA\Analyzers\AnomalyAnalyzer;
use baseKRIZAN\BORNA\Analyzers\IntegrityAnalyzer;
use baseKRIZAN\BORNA\Storage\StorageInterface;
use baseKRIZAN\BORNA\Storage\DatabaseStorage;
use baseKRIZAN\BORNA\Storage\RedisStorage;
use baseKRIZAN\BORNA\Enum\SecurityLevel;
use baseKRIZAN\BORNA\MachineLearning\ThreatPredictor;
use baseKRIZAN\BORNA\Interfaces\AnalyzerInterface;

/**
 * Core security management class for the BORNA Security Module
 * Next-generation security framework with advanced threat detection
 */
class BornaManager
{
    /**
     * Storage provider for security data
     */
    private StorageInterface $storage;
    
    /**
     * Logger for security events
     */
    private Logger $logger;
    
    /**
     * Service container
     */
    private Container $container;
    
    /**
     * Core threat detection engine
     */
    private ThreatDetector $threatDetector;
    
    /**
     * Firewall rules manager
     */
    private FirewallRules $firewallRules;
    
    /**
     * Security events recorder
     */
    private SecurityEvents $securityEvents;
    
    /**
     * Module enabled state
     */
    private bool $isEnabled = false;
    
    /**
     * Configuration settings
     */
    private array $config = [];
    
    /**
     * Security analyzers collection
     * 
     * @var array<string, AnalyzerInterface>
     */
    private array $analyzers = [];
    
    /**
     * Blocked IP addresses
     */
    private array $blockedIPs = [];
    
    /**
     * Current security level
     */
    private SecurityLevel $securityLevel;
    
    /**
     * ML-based threat predictor
     */
    private ?ThreatPredictor $threatPredictor = null;
    
    public static function getInstance(
        Logger $logger, 
        Container $container,
        ?StorageInterface $storage = null
    ): self {
        // Create a new instance each time since it's not a true singleton
        return new self($logger, $container, $storage);
    }

    /**
     * Constructor
     */
    public function __construct(
        Logger $logger, 
        Container $container,
        ?StorageInterface $storage = null
    ) {
        $this->logger = $logger;
        $this->container = $container;
        
        // Load configuration
        $this->loadConfig();
        
        // Configure storage
        $this->storage = $this->configureStorage($storage);
        
        // Initialize components if enabled
        if ($this->isEnabled()) {
            $this->initializeComponents();
        }
        
        $this->logger->borna('BORNA Security Module ' . ($this->isEnabled() ? 'enabled' : 'disabled'), [
            'security_level' => $this->securityLevel->name
        ]);
    
        // Uklonite ovaj dio - registracija middleware-a će se pozvati eksplicitno iz CoreModuleServiceProvider
        /*
        if ($this->isEnabled() && $this->container->has('middlewareRegistry')) {
            $middlewareRegistry = $this->container->get('middlewareRegistry');
            $this->registerMiddleware($middlewareRegistry);
        }
        */
    
        // Register module-specific components if enabled
        if ($this->isEnabled()) {
            $this->initializeModuleSpecificComponents();
        }
    }

    /**
     * Register middleware for security filtering
     * 
     * @param MiddlewareRegistry $registry MiddlewareRegistry instanca
     * @return void
     */
    public function registerMiddleware(MiddlewareRegistry $registry): void 
    {
        if (!$this->isEnabled()) {
            return;
        }

        $registry->addMiddleware(
            \baseKRIZAN\Http\Middleware\BornaMiddleware::class, 
            [
                'group' => 'global', 
                'priority' => 45,
                'requiredServices' => ['logger'],
                'conditionalLoad' => true
            ]
        );

        $this->logger->borna('BORNA middleware registration completed');
    }
    
    /**
     * Load configuration settings
     */
    private function loadConfig(): void
    {
        $this->isEnabled = \baseKRIZAN\Config\Config::get('borna.enabled');
        
        // Parse security level from configuration
        $securityLevelValue = (int)\baseKRIZAN\Config\Config::get('borna.security_level');
        $this->securityLevel = SecurityLevel::from($securityLevelValue);
        
        $this->config = [
             // Putanje
            'dashboard_path' => \baseKRIZAN\Config\Config::get('borna.dashboard_path'),
            // Sigurnosne postavke
            'block_threshold' => \baseKRIZAN\Config\Config::get('borna.block_threshold'),
            'block_duration' => \baseKRIZAN\Config\Config::get('borna.block_duration'),
            'alert_email' => \baseKRIZAN\Config\Config::get('borna.alert_email'),
            // Logging postavke
            'log_attacks' => \baseKRIZAN\Config\Config::get('borna.log_attacks'),
            // Rate limiting postavke
            'rate_limit_requests' => \baseKRIZAN\Config\Config::get('borna.rate_limit.requests'),
            'rate_limit_period' => \baseKRIZAN\Config\Config::get('borna.rate_limit.period'),
            // Sesijske postavke
            'max_sessions_per_user' => \baseKRIZAN\Config\Config::get('borna.session.max_per_user'),
            'session_guard_enabled' => \baseKRIZAN\Config\Config::get('borna.session.guard_enabled'),
            // Zaštita od napada
            'xss_protection' => \baseKRIZAN\Config\Config::get('borna.protection.xss'),
            'sqli_protection' => \baseKRIZAN\Config\Config::get('borna.protection.sqli'),
            'csrf_protection' => \baseKRIZAN\Config\Config::get('borna.protection.csrf'),
            // Napredne postavke
            'firewall_enabled' => \baseKRIZAN\Config\Config::get('borna.advanced.firewall_enabled'),
            'auto_block_enabled' => \baseKRIZAN\Config\Config::get('borna.advanced.auto_block_enabled'),
            'storage_type' => \baseKRIZAN\Config\Config::get('borna.advanced.storage_type'),
            'ml_enabled' => \baseKRIZAN\Config\Config::get('borna.advanced.ml_enabled'),
            'anomaly_detection' => \baseKRIZAN\Config\Config::get('borna.advanced.anomaly_detection'),
            'integrity_monitoring' => \baseKRIZAN\Config\Config::get('borna.advanced.integrity_monitoring'),
            // Geolokacijske postavke
            'geo_protection' => \baseKRIZAN\Config\Config::get('borna.geolocation.protection_enabled'),
            'allowed_countries' => \baseKRIZAN\Config\Config::get('borna.geolocation.allowed_countries'),
            'blocked_countries' => \baseKRIZAN\Config\Config::get('borna.geolocation.blocked_countries'),
            // Redis postavke
            'redis' => [
                'host' => \baseKRIZAN\Config\Config::get('borna.redis.host'),
                'port' => \baseKRIZAN\Config\Config::get('borna.redis.port'),
                'password' => \baseKRIZAN\Config\Config::get('borna.redis.password'),
                'database' => \baseKRIZAN\Config\Config::get('borna.redis.database'),
            ],
            // API zaštita
            'api_protection' => [
                'rate_limiting' => \baseKRIZAN\Config\Config::get('borna.api_protection.rate_limiting', true),
                'jwt_verification' => \baseKRIZAN\Config\Config::get('borna.api_protection.jwt_verification', true),
                'scope_validation' => \baseKRIZAN\Config\Config::get('borna.api_protection.scope_validation', true),
            ],
            // Napredne zaštite
            'advanced_protection' => [
                'client_fingerprinting' => \baseKRIZAN\Config\Config::get('borna.advanced_protection.client_fingerprinting', true),
                'behavioral_biometrics' => \baseKRIZAN\Config\Config::get('borna.advanced_protection.behavioral_biometrics', false),
                'polymorphic_tokens' => \baseKRIZAN\Config\Config::get('borna.advanced_protection.polymorphic_tokens', true),
            ],
            'sessionencrypt' => \baseKRIZAN\Config\Config::get('sessionencrypt'),
        ];
    }

    /**
     * Initialize module-specific components
     *
     * @return void
     */
    private function initializeModuleSpecificComponents(): void
    {
        $this->logger->borna('Initializing BORNA specific components');

        // Specifično za BORNA modul
        // Stvori odgovarajući storage
        $storageType = \baseKRIZAN\Config\Config::get('borna.storage_type', 'database');
        $bornaStorage = null;
        
        if ($storageType === 'redis' && class_exists('\\baseKRIZAN\\BORNA\\Storage\\RedisStorage')) {
            $redisConfig = [
                'host' => \baseKRIZAN\Config\Config::get('BORNA_REDIS_HOST', '127.0.0.1'),
                'port' => \baseKRIZAN\Config\Config::get('BORNA_REDIS_PORT', 6379),
                'password' => \baseKRIZAN\Config\Config::get('BORNA_REDIS_PASSWORD', null),
                'database' => \baseKRIZAN\Config\Config::get('BORNA_REDIS_DATABASE', 0)
            ];
            
            $bornaStorage = new \baseKRIZAN\BORNA\Storage\RedisStorage(
                $this->logger,
                $redisConfig['host'],
                $redisConfig['port'],
                $redisConfig['password'],
                $redisConfig['database']
            );
        } else {
            // Default to database storage
            $bornaStorage = new \baseKRIZAN\BORNA\Storage\DatabaseStorage($this->logger);
        }
        
        // Registriraj storage
        $this->container->register('bornaStorage', $bornaStorage);
        
        // Registriraj firewall rules
        $firewallRules = new \baseKRIZAN\BORNA\FirewallRules($this->logger);
        $this->container->register('bornaFirewallRules', $firewallRules);
        
        // Registriraj kontroler ako BORNA postoji
        if (!$this->container->has('bornaController') && class_exists('\\Controllers\\BORNA')) {
            $this->container->register('bornaController', function($container) {
                return new \Controllers\BORNA($container);
            });
        }

        $this->logger->borna('BORNA module specific components initialized');
    }
    
    /**
     * Initialize core components
     */
    private function initializeComponents(): void
    {
        // Ukloni provjeru storage_path
        
        // Initialize core components
        $this->threatDetector = new ThreatDetector($this->logger, $this->config);
        $this->container->register('bornaThreatDetector', $this->threatDetector);
        
        $this->firewallRules = new FirewallRules($this->logger);
        $this->container->register('bornaFirewallRules', $this->firewallRules);
        
        $this->securityEvents = new SecurityEvents($this->logger, $this->storage);
        $this->container->register('bornaSecurityEvents', $this->securityEvents);
        
        // Initialize analyzers
        $this->initializeAnalyzers();
    }
    
    /**
     * Configure storage based on configuration
     */
    private function configureStorage(?StorageInterface $storage = null): StorageInterface
    {
        // Use provided storage if available
        if ($storage) {
            return $storage;
        }
        
        // Create storage based on configuration
        $storageType = $this->config['storage_type'] ?? 'database'; // Database je default
        
        return match ($storageType) {
            'redis' => $this->createRedisStorage(),
            default => $this->createDatabaseStorage() // Default je database
        };
    }
    
    /**
     * Create Redis-based storage
     */
    private function createRedisStorage(): RedisStorage
    {
        $redisConfig = $this->config['redis'] ?? [];
        return new RedisStorage(
            $this->logger,
            $redisConfig['host'] ?? '127.0.0.1',
            $redisConfig['port'] ?? 6379,
            $redisConfig['password'] ?? null,
            $redisConfig['database'] ?? 0
        );
    }

    /**
     * Create Database storage
     */
    private function createDatabaseStorage(): DatabaseStorage
    {
        return new DatabaseStorage($this->logger);
    }
    
    /**
     * Initialize security analyzers
     */
    private function initializeAnalyzers(): void
    {
        // Geolocation analyzer
        if ($this->config['geo_protection']) {
            $geoAnalyzer = new GeolocationAnalyzer(
                $this->logger, 
                $this->config['allowed_countries'],
                $this->config['blocked_countries']
            );
            $this->analyzers['geo'] = $geoAnalyzer;
            $this->container->register('bornaGeoAnalyzer', $geoAnalyzer);
        }
        
        // Behavioral analyzer
        $behavioralAnalyzer = new BehavioralAnalyzer($this->logger, $this->securityLevel->value);
        $this->analyzers['behavioral'] = $behavioralAnalyzer;
        $this->container->register('bornaBehavioralAnalyzer', $behavioralAnalyzer);
        
        // Pattern analyzer
        $patternAnalyzer = new PatternAnalyzer(
            $this->logger, 
            $this->config['xss_protection'],
            $this->config['sqli_protection']
        );
        $this->analyzers['pattern'] = $patternAnalyzer;
        $this->container->register('bornaPatternAnalyzer', $patternAnalyzer);
        
        // Advanced analyzers - only in BORNA
        
        // Anomaly analyzer for statistical outlier detection
        if ($this->config['anomaly_detection']) {
            $anomalyAnalyzer = new AnomalyAnalyzer($this->logger, $this->storage, $this->securityLevel);
            $this->analyzers['anomaly'] = $anomalyAnalyzer;
            $this->container->register('bornaAnomalyAnalyzer', $anomalyAnalyzer);
        }
        
        // Integrity analyzer for file and data integrity monitoring
        if ($this->config['integrity_monitoring']) {
            $integrityAnalyzer = new IntegrityAnalyzer($this->logger, $this->storage, $this->config['storage_path']);
            $this->analyzers['integrity'] = $integrityAnalyzer;
            $this->container->register('bornaIntegrityAnalyzer', $integrityAnalyzer);
        }
    }
    
    /**
     * Initialize machine learning components
     */
    private function initializeMachineLearning(): void
    {
        try {
            $this->threatPredictor = new ThreatPredictor(
                $this->logger,
                $this->config['ml_model_path'],
                $this->storage
            );
            
            $this->container->register('bornaThreatPredictor', $this->threatPredictor);
            
            $this->logger->borna('BORNA ML threat prediction engine initialized');
        } catch (\Throwable $e) {
            $this->logger->error('Failed to initialize ML threat prediction', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
    
    /**
     * Load blocked IPs from storage
     */
    private function loadBlockedIPs(): void
    {
        $this->blockedIPs = $this->storage->getBlockedIPs();
        
        // Filter out expired blocks
        $currentTime = time();
        $this->blockedIPs = array_filter($this->blockedIPs, function($item) use ($currentTime) {
            return isset($item['expires']) && $item['expires'] > $currentTime;
        });
        
        $this->logger->borna('Loaded ' . count($this->blockedIPs) . ' blocked IPs');
    }
    
    /**
     * Save blocked IPs to storage
     */
    private function saveBlockedIPs(): void
    {
        $this->storage->saveBlockedIPs($this->blockedIPs);
    }
    
    /**
     * Get all blocked IP addresses
     */
    public function getBlockedIPs(): array
    {
        return $this->blockedIPs;
    }
    
    /**
     * Check if an IP is blocked
     */
    public function isIPBlocked(string $ip): bool
    {
        return isset($this->blockedIPs[$ip]) && $this->blockedIPs[$ip]['expires'] > time();
    }
    
    /**
     * Block an IP address
     */
    public function blockIP(string $ip, string $reason, ?int $durationHours = null): void
    {
        if (empty($ip)) {
            return;
        }
    
        // Skip blocking localhost IPs
        if ($ip === '127.0.0.1' || $ip === '::1' || $ip === 'localhost') {
            $this->logger->borna('Skipping block for localhost IP', [
                'ip' => $ip,
                'reason' => $reason
            ]);
            return;
        }
        
        // Use default duration if not specified
        if ($durationHours === null) {
            $durationHours = $this->config['block_duration'];
        }
        
        $this->blockedIPs[$ip] = [
            'ip' => $ip,
            'reason' => $reason,
            'timestamp' => time(),
            'expires' => time() + ($durationHours * 3600)
        ];
        
        $this->saveBlockedIPs();
        
        $this->logger->borna('IP address blocked', [
            'ip' => $ip,
            'reason' => $reason,
            'duration_hours' => $durationHours,
            'expires' => date('Y-m-d H:i:s', $this->blockedIPs[$ip]['expires'])
        ]);
        
        // Record security event
        $this->securityEvents->recordEvent('ip_blocked', [
            'ip' => $ip,
            'reason' => $reason,
            'duration_hours' => $durationHours
        ]);
    }
    
    /**
     * Unblock an IP address
     */
    public function unblockIP(string $ip): bool
    {
        if (isset($this->blockedIPs[$ip])) {
            unset($this->blockedIPs[$ip]);
            $this->saveBlockedIPs();
            
            $this->logger->borna('IP address unblocked', [
                'ip' => $ip
            ]);
            
            // Record security event
            $this->securityEvents->recordEvent('ip_unblocked', [
                'ip' => $ip
            ]);
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Analyze a request for security threats using all available analyzers
     * 
     * @param array $requestData Request data including path, method, ip, params, etc.
     * @return array Analysis results with threat score and details
     */
    public function analyzeRequest(array $requestData): array
    {
        if (!$this->isEnabled) {
            return ['threat_score' => 0, 'threats' => []];
        }
        
        $threats = [];
        $totalScore = 0;
        
        // Check if IP is already blocked
        if ($this->isIPBlocked($requestData['ip'])) {
            return [
                'threat_score' => 100,
                'threats' => [
                    [
                        'type' => 'blocked_ip',
                        'score' => 100,
                        'description' => 'IP address is blocked',
                        'details' => $this->blockedIPs[$requestData['ip']]
                    ]
                ]
            ];
        }
        
        // Run standard analyzers first
        foreach ($this->analyzers as $name => $analyzer) {
            $result = $analyzer->analyze($requestData);
            
            if ($result['score'] > 0) {
                $threats[] = [
                    'type' => $name,
                    'score' => $result['score'],
                    'description' => $result['description'],
                    'details' => $result['details'] ?? []
                ];
                
                $totalScore += $result['score'];
            }
        }
        
        // Advanced: Run ML-based threat prediction if available
        if ($this->threatPredictor && $this->config['ml_enabled']) {
            $mlPrediction = $this->threatPredictor->predictThreat($requestData);
            
            if ($mlPrediction['score'] > 0) {
                $threats[] = [
                    'type' => 'ml_prediction',
                    'score' => $mlPrediction['score'],
                    'description' => 'Machine learning threat prediction',
                    'details' => $mlPrediction['details'] ?? []
                ];
                
                // ML predictions can have higher weight based on confidence
                $mlWeight = $mlPrediction['confidence'] ?? 1.0;
                $totalScore += $mlPrediction['score'] * $mlWeight;
            }
        }
        
        // Normalize total score to 0-100 range
        $totalScore = min(100, $totalScore);
        
        // Auto-block if score is above threshold and auto-block is enabled
        if ($totalScore >= $this->config['block_threshold'] && $this->config['auto_block_enabled']) {
            $this->blockIP(
                $requestData['ip'],
                'Automatic block due to high threat score: ' . $totalScore,
                $this->config['block_duration']
            );
        }
        
        // Log attack if score is high enough
        if ($totalScore > 50 && $this->config['log_attacks']) {
            $this->securityEvents->recordEvent('attack_detected', [
                'ip' => $requestData['ip'],
                'path' => $requestData['path'],
                'method' => $requestData['method'],
                'score' => $totalScore,
                'threats' => $threats
            ]);
            
            $this->logger->borna('Potential attack detected', [
                'ip' => $requestData['ip'],
                'path' => $requestData['path'],
                'threat_score' => $totalScore,
                'threats_count' => count($threats)
            ]);
        }
        
        return [
            'threat_score' => $totalScore,
            'threats' => $threats
        ];
    }
    
    /**
     * Get security events for a time period
     */
    public function getSecurityEvents(\DateTime $from, \DateTime $to, ?string $eventType = null): array
    {
        return $this->securityEvents->getEvents($from, $to, $eventType);
    }
    
    /**
     * Get security stats for the dashboard
     */
    public function getSecurityStats(): array
    {
        if (!$this->isEnabled) {
            return ['status' => 'BORNA Security Module disabled'];
        }
        
        // Get last 24 hours of events
        $from = new \DateTime('-24 hours');
        $to = new \DateTime();
        
        $recentEvents = $this->securityEvents->getEvents($from, $to);
        $attackEvents = array_filter($recentEvents, function($event) {
            return $event['event_type'] === 'attack_detected';
        });
        
        $blockedIpsCount = count($this->blockedIPs);
        $attackCount = count($attackEvents);
        
        // Get top attacked paths
        $attackedPaths = [];
        foreach ($attackEvents as $event) {
            $path = $event['data']['path'] ?? 'unknown';
            if (!isset($attackedPaths[$path])) {
                $attackedPaths[$path] = 0;
            }
            $attackedPaths[$path]++;
        }
        
        // Sort by most attacked
        arsort($attackedPaths);
        
        // Get top attacker IPs
        $attackerIPs = [];
        foreach ($attackEvents as $event) {
            $ip = $event['data']['ip'] ?? 'unknown';
            if (!isset($attackerIPs[$ip])) {
                $attackerIPs[$ip] = 0;
            }
            $attackerIPs[$ip]++;
        }
        
        // Sort by most attacks
        arsort($attackerIPs);
        
        // Basic stats
        $stats = [
            'status' => 'active',
            'security_level' => $this->securityLevel->value,
            'security_level_name' => $this->securityLevel->name,
            'blocked_ips_count' => $blockedIpsCount,
            'attack_count_24h' => $attackCount,
            'firewall_status' => $this->config['firewall_enabled'] ? 'active' : 'disabled',
            'auto_block' => $this->config['auto_block_enabled'] ? 'enabled' : 'disabled',
            'top_attacked_paths' => array_slice($attackedPaths, 0, 5),
            'top_attacker_ips' => array_slice($attackerIPs, 0, 5),
            'recent_attacks' => array_slice($attackEvents, 0, 10),
            'geo_protection' => $this->config['geo_protection'] ? 'enabled' : 'disabled'
        ];
        
        // Add advanced protection stats
        $stats['advanced_protection'] = [
            'ml_enabled' => $this->config['ml_enabled'],
            'anomaly_detection' => $this->config['anomaly_detection'],
            'integrity_monitoring' => $this->config['integrity_monitoring'],
            'client_fingerprinting' => $this->config['advanced_protection']['client_fingerprinting'],
            'behavioral_biometrics' => $this->config['advanced_protection']['behavioral_biometrics'],
        ];
        
        // Add analyzer statistics if available
        $analyzerStats = [];
        foreach ($this->analyzers as $name => $analyzer) {
            if (method_exists($analyzer, 'getStatistics')) {
                $analyzerStats[$name] = $analyzer->getStatistics();
            }
        }
        
        if (!empty($analyzerStats)) {
            $stats['analyzer_statistics'] = $analyzerStats;
        }
        
        return $stats;
    }
    
    /**
     * Get the current security level
     */
    public function getSecurityLevel(): SecurityLevel
    {
        return $this->securityLevel;
    }
    
    /**
     * Set the security level
     */
    public function setSecurityLevel(SecurityLevel $level): void
    {
        $previousLevel = $this->securityLevel;
        
        $this->logger->borna('Security level changed', [
            'previous_level' => $previousLevel->name,
            'new_level' => $level->name
        ]);
        
        // Update analyzers security level
        foreach ($this->analyzers as $analyzer) {
            if (method_exists($analyzer, 'setSecurityLevel')) {
                $analyzer->setSecurityLevel($level->value);
            }
        }
        
        // Record event
        $this->securityEvents->recordEvent('security_level_changed', [
            'previous_level' => $previousLevel->value,
            'previous_level_name' => $previousLevel->name,
            'new_level' => $level->value,
            'new_level_name' => $level->name
        ]);
        
        $this->securityLevel = $level;
    }
    
    /**
     * Run integrity check on critical system files
     */
    public function checkSystemIntegrity(): array
    {
        if (!isset($this->analyzers['integrity'])) {
            return [
                'status' => 'error',
                'message' => 'Integrity analyzer not available'
            ];
        }
        
        /** @var IntegrityAnalyzer $integrityAnalyzer */
        $integrityAnalyzer = $this->analyzers['integrity'];
        
        return $integrityAnalyzer->performFullSystemCheck();
    }
    
    /**
     * Get analytics on recent security threats
     */
    public function getThreatAnalytics(int $days = 7): array
    {
        $from = new \DateTime("-{$days} days");
        $to = new \DateTime();
        
        // Get events
        $events = $this->securityEvents->getEvents($from, $to);
        
        // Prepare analysis
        $threatsByDay = [];
        $threatsByType = [];
        $threatsByHour = array_fill(0, 24, 0);
        $ipStats = [];
        
        // Initialize threatsByDay
        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', strtotime("-{$i} days"));
            $threatsByDay[$date] = 0;
        }
        
        // Process events
        foreach ($events as $event) {
            if ($event['event_type'] !== 'attack_detected') {
                continue;
            }
            
            // Process by day
            $date = substr($event['timestamp'], 0, 10);
            if (isset($threatsByDay[$date])) {
                $threatsByDay[$date]++;
            }
            
            // Process by hour
            $hour = (int)substr($event['timestamp'], 11, 2);
            $threatsByHour[$hour]++;
            
            // Process by type
            $threats = $event['data']['threats'] ?? [];
            foreach ($threats as $threat) {
                $type = $threat['type'] ?? 'unknown';
                if (!isset($threatsByType[$type])) {
                    $threatsByType[$type] = 0;
                }
                $threatsByType[$type]++;
            }
            
            // Process by IP
            $ip = $event['data']['ip'] ?? 'unknown';
            if (!isset($ipStats[$ip])) {
                $ipStats[$ip] = [
                    'count' => 0,
                    'paths' => [],
                    'threat_types' => [],
                    'max_score' => 0
                ];
            }
            
            $ipStats[$ip]['count']++;
            
            // Record path
            $path = $event['data']['path'] ?? 'unknown';
            if (!in_array($path, $ipStats[$ip]['paths'])) {
                $ipStats[$ip]['paths'][] = $path;
            }
            
            // Record threat types
            foreach ($threats as $threat) {
                $type = $threat['type'] ?? 'unknown';
                if (!in_array($type, $ipStats[$ip]['threat_types'])) {
                    $ipStats[$ip]['threat_types'][] = $type;
                }
                
                // Track max score
                $score = $threat['score'] ?? 0;
                if ($score > $ipStats[$ip]['max_score']) {
                    $ipStats[$ip]['max_score'] = $score;
                }
            }
        }
        
        // Sort IP stats by count
        uasort($ipStats, function($a, $b) {
            return $b['count'] <=> $a['count'];
        });
        
        // Limit to top 10 IPs
        $ipStats = array_slice($ipStats, 0, 10, true);
        
        return [
            'days_analyzed' => $days,
            'total_events' => count($events),
            'total_attacks' => array_sum($threatsByDay),
            'threats_by_day' => $threatsByDay,
            'threats_by_type' => $threatsByType,
            'threats_by_hour' => $threatsByHour,
            'top_threatening_ips' => $ipStats
        ];
    }
    
    /**
     * Generate a client security fingerprint token
     * Uses polymorphic token generation for enhanced client verification
     */
    public function generateClientFingerprint(array $clientData): string
    {
        // Only available in advanced mode
        if (!($this->config['advanced_protection']['client_fingerprinting'] ?? false)) {
            return '';
        }
        
        // Extract client attributes
        $userAgent = $clientData['user_agent'] ?? '';
        $ip = $clientData['ip'] ?? '';
        $screenData = $clientData['screen'] ?? '';
        $timeZone = $clientData['timezone'] ?? '';
        $plugins = $clientData['plugins'] ?? '';
        $canvas = $clientData['canvas'] ?? '';
        $languages = $clientData['languages'] ?? '';
        
        // Create base fingerprint
        $baseFingerprint = hash('sha256', $userAgent . $ip . $screenData . $timeZone . $plugins . $canvas . $languages . microtime(true));
        
        // Add time-based component for polymorphic tokens
        $timeComponent = floor(time() / 3600); // Changes every hour
        
        // Add session-based component if available
        $sessionComponent = '';
        if (isset($clientData['session_id']) && !empty($clientData['session_id'])) {
            $sessionComponent = substr(hash('sha256', $clientData['session_id']), 0, 16);
        }
        
        // Generate final fingerprint token
        // OVDJE JE PROMJENA - dodana provjera za sessionencrypt
        $finalFingerprint = hash('sha256', $baseFingerprint . $timeComponent . $sessionComponent . ($this->config['sessionencrypt'] ?? ''));
        
        // Store fingerprint association for verification
        $this->storage->storeClientFingerprint($finalFingerprint, [
            'ip' => $ip,
            'user_agent' => $userAgent,
            'created' => time(),
            'expires' => time() + 86400, // 24 hours
            'session_id' => $clientData['session_id'] ?? null,
            'user_id' => $clientData['user_id'] ?? null
        ]);
        
        return $finalFingerprint;
    }
    
    /**
     * Verify a client security fingerprint token
     */
    public function verifyClientFingerprint(string $fingerprint, array $clientData): bool
    {
        // Only available in advanced mode
        if (!($this->config['advanced_protection']['client_fingerprinting'] ?? false)) {
            return true; // Skip verification if disabled
        }
        
        // Retrieve stored fingerprint data
        $storedData = $this->storage->getClientFingerprint($fingerprint);
        
        // Unknown fingerprint
        if (!$storedData) {
            $this->logger->borna('Unknown client fingerprint', [
                'fingerprint' => substr($fingerprint, 0, 16) . '...',
                'ip' => $clientData['ip'] ?? 'unknown'
            ]);
            return false;
        }
        
        // Expired fingerprint
        if ($storedData['expires'] < time()) {
            $this->logger->borna('Expired client fingerprint', [
                'fingerprint' => substr($fingerprint, 0, 16) . '...',
                'ip' => $clientData['ip'] ?? 'unknown'
            ]);
            return false;
        }
        
        // Basic checks
        $ipMatch = ($storedData['ip'] === ($clientData['ip'] ?? ''));
        $userAgentMatch = ($storedData['user_agent'] === ($clientData['user_agent'] ?? ''));
        
        // Session check if applicable
        $sessionMatch = true;
        if (isset($storedData['session_id']) && isset($clientData['session_id'])) {
            $sessionMatch = ($storedData['session_id'] === $clientData['session_id']);
        }
        
        // User check if applicable
        $userMatch = true;
        if (isset($storedData['user_id']) && isset($clientData['user_id'])) {
            $userMatch = ($storedData['user_id'] === $clientData['user_id']);
        }
        
        $verified = $ipMatch && $userAgentMatch && $sessionMatch && $userMatch;
        
        // Log suspicious mismatch
        if (!$verified) {
            $this->logger->borna('Client fingerprint verification failed', [
                'fingerprint' => substr($fingerprint, 0, 16) . '...',
                'ip_match' => $ipMatch,
                'user_agent_match' => $userAgentMatch,
                'session_match' => $sessionMatch,
                'user_match' => $userMatch,
                'client_ip' => $clientData['ip'] ?? 'unknown',
                'stored_ip' => $storedData['ip']
            ]);
        }
        
        return $verified;
    }
    
    /**
     * Check if BORNA is enabled
     */
    public function isEnabled(): bool
    {
        return $this->isEnabled;
    }
    
    /**
     * Get configuration settings
     */
    public function getConfig(): array
    {
        return $this->config;
    }
}