<?php

namespace baseKRIZAN\BORNA\Enum;

/**
 * Security level enum defines the different security protection levels
 */
enum SecurityLevel: int
{
    case LOW = 1;
    case MEDIUM = 2;
    case HIGH = 3;
    
    /**
     * Get a human-readable description of the security level
     */
    public function getDescription(): string
    {
        return match($this) {
            self::LOW => 'Basic security controls for standard applications',
            self::MEDIUM => 'Enhanced security for sensitive applications',
            self::HIGH => 'Maximum security for critical applications'
        };
    }
    
    /**
     * Get configuration values appropriate for this security level
     */
    public function getConfigDefaults(): array
    {
        return match($this) {
            self::LOW => [
                'request_sampling_rate' => 0.1, // 10% of requests
                'auto_block_duration' => 1, // 1 hour
                'request_threshold' => 60, // 60 requests per minute
                'anomaly_sensitivity' => 0.3, // low sensitivity
                'pattern_matching' => 'basic'
            ],
            self::MEDIUM => [
                'request_sampling_rate' => 0.5, // 50% of requests
                'auto_block_duration' => 6, // 6 hours
                'request_threshold' => 40, // 40 requests per minute
                'anomaly_sensitivity' => 0.6, // medium sensitivity
                'pattern_matching' => 'advanced'
            ],
            self::HIGH => [
                'request_sampling_rate' => 1.0, // 100% of requests
                'auto_block_duration' => 24, // 24 hours
                'request_threshold' => 20, // 20 requests per minute
                'anomaly_sensitivity' => 0.9, // high sensitivity
                'pattern_matching' => 'comprehensive'
            ]
        };
    }
    
    /**
     * Check if at least this security level
     */
    public function isAtLeast(self $level): bool
    {
        return $this->value >= $level->value;
    }
}