<?php

namespace baseKRIZAN\BORNA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * Firewall rules management for BORNA security module using database storage
 * Handles creation, evaluation, and storage of firewall rules
 */
class FirewallRules
{
    private Logger $logger;
    private $dbConnection;
    private string $rulesTable = 'security_borna_firewall_rules';
    private array $rules = [];
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger, 
        ?DatabaseConnection $dbConnection = null
    ) {
        $this->logger = $logger;
        
        // Inicijalizacija DatabaseConnection
        if ($dbConnection !== null) {
            $this->dbConnection = $dbConnection;
        } else {
            try {
                $this->dbConnection = DatabaseConnection::getInstance();
            } catch (\Throwable $e) {
                $this->logger->error('Failed to create database connection for FirewallRules', [
                    'error' => $e->getMessage()
                ]);
                throw $e;
            }
        }
        
        $this->ensureTableExists();
        $this->loadRules();
    }
    
    /**
     * Ensure the rules table exists
     */
    private function ensureTableExists(): void
    {
        try {
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
                ['table' => $this->rulesTable]
            );
            
            if ($tableExists === null) {
                // Create the table
                $sql = "CREATE TABLE `{$this->rulesTable}` (
                    `id` VARCHAR(36) NOT NULL PRIMARY KEY,
                    `name` VARCHAR(255) NOT NULL,
                    `type` VARCHAR(50) NOT NULL,
                    `pattern` TEXT NOT NULL,
                    `action` VARCHAR(50) NOT NULL,
                    `enabled` BOOLEAN NOT NULL DEFAULT TRUE,
                    `description` TEXT,
                    `created_at` DATETIME NOT NULL,
                    `updated_at` DATETIME
                )";
                
                $this->dbConnection->execute($sql);
                $this->logger->borna('Created firewall rules table');
            }
        } catch (\PDOException $e) {
            $this->logger->error('Error creating firewall rules table', [
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Load firewall rules from database
     */
    private function loadRules(): void
    {
        try {
            $dbRules = $this->dbConnection->queryAndFetchAllAssoc(
                "SELECT * FROM `{$this->rulesTable}` ORDER BY `name`"
            );
            
            if ($dbRules) {
                $this->rules = $dbRules;
                $this->logger->borna('Loaded ' . count($this->rules) . ' firewall rules from database');
            } else {
                // Create default rules if none exist
                $this->createDefaultRules();
            }
        } catch (\PDOException $e) {
            $this->logger->error('Error loading firewall rules', [
                'error' => $e->getMessage()
            ]);
            
            // Create default rules as fallback
            $this->createDefaultRules();
        }
    }
    
    /**
     * Save a rule to the database
     */
    private function saveRule(array $rule): bool
    {
        try {
            if (isset($rule['id'])) {
                // Update existing rule
                $sql = "
                    UPDATE `{$this->rulesTable}` SET
                    `name` = :name,
                    `type` = :type,
                    `pattern` = :pattern,
                    `action` = :action,
                    `enabled` = :enabled,
                    `description` = :description,
                    `updated_at` = :updated_at
                    WHERE `id` = :id
                ";
                
                $params = [
                    'id' => $rule['id'],
                    'name' => $rule['name'],
                    'type' => $rule['type'],
                    'pattern' => $rule['pattern'],
                    'action' => $rule['action'],
                    'enabled' => $rule['enabled'] ? 1 : 0,
                    'description' => $rule['description'],
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                $this->dbConnection->execute($sql, $params);
                return true;
            } else {
                // Insert new rule
                $sql = "
                    INSERT INTO `{$this->rulesTable}`
                    (`id`, `name`, `type`, `pattern`, `action`, `enabled`, `description`, `created_at`)
                    VALUES
                    (:id, :name, :type, :pattern, :action, :enabled, :description, :created_at)
                ";
                
                $params = [
                    'id' => $rule['id'],
                    'name' => $rule['name'],
                    'type' => $rule['type'],
                    'pattern' => $rule['pattern'],
                    'action' => $rule['action'],
                    'enabled' => $rule['enabled'] ? 1 : 0,
                    'description' => $rule['description'],
                    'created_at' => date('Y-m-d H:i:s')
                ];
                
                $this->dbConnection->execute($sql, $params);
                return true;
            }
        } catch (\PDOException $e) {
            $this->logger->error('Error saving firewall rule', [
                'error' => $e->getMessage(),
                'rule' => $rule['name']
            ]);
            return false;
        }
    }
    
    /**
     * Create default firewall rules
     */
    private function createDefaultRules(): void
    {
        $defaultRules = [
            [
                'id' => $this->generateRuleId(),
                'name' => 'Block known malicious paths',
                'type' => 'path',
                'pattern' => '/\.(git|svn|htaccess|env|config|bak|backup|sql|zip|rar|tar\.gz|php~)$/i',
                'action' => 'block',
                'enabled' => true,
                'description' => 'Blocks access to sensitive files and directories',
                'created_at' => date('Y-m-d H:i:s')
            ],
            [
                'id' => $this->generateRuleId(),
                'name' => 'Block SQL injection attempts',
                'type' => 'parameter',
                'pattern' => '/(UNION\s+SELECT|SELECT\s+.*\s+FROM|DROP\s+TABLE|--)/i',
                'action' => 'block',
                'enabled' => true,
                'description' => 'Blocks SQL injection attempts in query parameters',
                'created_at' => date('Y-m-d H:i:s')
            ],
            [
                'id' => $this->generateRuleId(),
                'name' => 'Block XSS attempts',
                'type' => 'parameter',
                'pattern' => '/(<script|javascript:|on\w+=|alert\(|eval\()/i',
                'action' => 'block',
                'enabled' => true,
                'description' => 'Blocks cross-site scripting attempts in parameters',
                'created_at' => date('Y-m-d H:i:s')
            ],
            [
                'id' => $this->generateRuleId(),
                'name' => 'Block path traversal',
                'type' => 'path',
                'pattern' => '/(\.\.|\/etc\/passwd|\/proc\/self)/i',
                'action' => 'block',
                'enabled' => true,
                'description' => 'Blocks directory traversal attempts',
                'created_at' => date('Y-m-d H:i:s')
            ],
            [
                'id' => $this->generateRuleId(),
                'name' => 'Log suspicious user agents',
                'type' => 'user_agent',
                'pattern' => '/(nmap|nikto|sqlmap|acunetix|nessus|burpsuite|metasploit)/i',
                'action' => 'log',
                'enabled' => true,
                'description' => 'Logs requests with suspicious user agents',
                'created_at' => date('Y-m-d H:i:s')
            ]
        ];
        
        // Koristimo transakciju kroz DatabaseConnection
        $this->dbConnection->beginTransaction();
        
        try {
            // Clear existing rules
            $this->dbConnection->execute("DELETE FROM `{$this->rulesTable}`");
            
            // Insert default rules
            foreach ($defaultRules as $rule) {
                $this->saveRule($rule);
            }
            
            $this->dbConnection->commit();
            $this->rules = $defaultRules;
            
            $this->logger->borna('Created default firewall rules', [
                'count' => count($defaultRules)
            ]);
        } catch (\PDOException $e) {
            $this->dbConnection->rollBack();
            $this->logger->error('Error creating default firewall rules', [
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Add a new firewall rule
     * 
     * @param array $rule Rule definition
     * @return string Rule ID
     */
    public function addRule(array $rule): string
    {
        // Validate rule
        $this->validateRule($rule);
        
        // Generate rule ID
        $rule['id'] = $this->generateRuleId();
        
        // Save to database
        if ($this->saveRule($rule)) {
            // Add to rules array
            $this->rules[] = $rule;
            
            $this->logger->borna('Added firewall rule', [
                'rule_id' => $rule['id'],
                'rule_name' => $rule['name']
            ]);
            
            return $rule['id'];
        }
        
        throw new \RuntimeException('Failed to add firewall rule');
    }
    
    /**
     * Update an existing firewall rule
     * 
     * @param string $ruleId Rule ID
     * @param array $updatedRule Updated rule definition
     * @return bool Success status
     */
    public function updateRule(string $ruleId, array $updatedRule): bool
    {
        // Find rule by ID
        $index = $this->findRuleIndexById($ruleId);
        
        if ($index === false) {
            return false;
        }
        
        // Validate updated rule
        $this->validateRule($updatedRule);
        
        // Preserve rule ID
        $updatedRule['id'] = $ruleId;
        
        // Save to database
        if ($this->saveRule($updatedRule)) {
            // Update in rules array
            $this->rules[$index] = $updatedRule;
            
            $this->logger->borna('Updated firewall rule', [
                'rule_id' => $ruleId,
                'rule_name' => $updatedRule['name']
            ]);
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Delete a firewall rule
     * 
     * @param string $ruleId Rule ID
     * @return bool Success status
     */
    public function deleteRule(string $ruleId): bool
    {
        // Find rule by ID
        $index = $this->findRuleIndexById($ruleId);
        
        if ($index === false) {
            return false;
        }
        
        // Store rule name for logging
        $ruleName = $this->rules[$index]['name'];
        
        try {
            // Delete from database
            $this->dbConnection->execute(
                "DELETE FROM `{$this->rulesTable}` WHERE `id` = :id",
                ['id' => $ruleId]
            );
            
            // Remove from rules array
            array_splice($this->rules, $index, 1);
            
            $this->logger->borna('Deleted firewall rule', [
                'rule_id' => $ruleId,
                'rule_name' => $ruleName
            ]);
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Error deleting firewall rule', [
                'error' => $e->getMessage(),
                'rule_id' => $ruleId
            ]);
            
            return false;
        }
    }
    
    /**
     * Enable or disable a firewall rule
     * 
     * @param string $ruleId Rule ID
     * @param bool $enabled Enabled status
     * @return bool Success status
     */
    public function setRuleEnabled(string $ruleId, bool $enabled): bool
    {
        // Find rule by ID
        $index = $this->findRuleIndexById($ruleId);
        
        if ($index === false) {
            return false;
        }
        
        try {
            // Update in database
            $sql = "
                UPDATE `{$this->rulesTable}` 
                SET `enabled` = :enabled, `updated_at` = :updated_at 
                WHERE `id` = :id
            ";
            
            $params = [
                'id' => $ruleId,
                'enabled' => $enabled ? 1 : 0,
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            $this->dbConnection->execute($sql, $params);
            
            // Update in rules array
            $this->rules[$index]['enabled'] = $enabled;
            
            $this->logger->borna('Changed firewall rule status', [
                'rule_id' => $ruleId,
                'rule_name' => $this->rules[$index]['name'],
                'enabled' => $enabled
            ]);
            
            return true;
        } catch (\PDOException $e) {
            $this->logger->error('Error updating firewall rule status', [
                'error' => $e->getMessage(),
                'rule_id' => $ruleId
            ]);
            
            return false;
        }
    }
    
    /**
     * Get all firewall rules
     * 
     * @return array All rules
     */
    public function getAllRules(): array
    {
        return $this->rules;
    }
    
    /**
     * Get a specific firewall rule
     * 
     * @param string $ruleId Rule ID
     * @return array|null Rule definition or null if not found
     */
    public function getRule(string $ruleId): ?array
    {
        foreach ($this->rules as $rule) {
            if ($rule['id'] === $ruleId) {
                return $rule;
            }
        }
        
        return null;
    }
    
    /**
     * Evaluate a request against firewall rules
     * 
     * @param array $requestData Request data
     * @return array Triggered rules with actions
     */
    public function evaluateRequest(array $requestData): array
    {
        $triggeredRules = [];
        
        foreach ($this->rules as $rule) {
            // Skip disabled rules
            if (!$rule['enabled']) {
                continue;
            }
            
            $triggered = false;
            
            switch ($rule['type']) {
                case 'path':
                    if (isset($requestData['path']) && preg_match($rule['pattern'], $requestData['path'])) {
                        $triggered = true;
                    }
                    break;
                    
                case 'parameter':
                    // Check query params
                    if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
                        foreach ($requestData['query_params'] as $param => $value) {
                            if (is_string($value) && preg_match($rule['pattern'], $value)) {
                                $triggered = true;
                                $triggeredRule['param'] = $param;
                                $triggeredRule['value'] = $value;
                                break;
                            }
                        }
                    }
                    
                    // Check post params if not already triggered
                    if (!$triggered && isset($requestData['post_params']) && is_array($requestData['post_params'])) {
                        foreach ($requestData['post_params'] as $param => $value) {
                            if (is_string($value) && preg_match($rule['pattern'], $value)) {
                                $triggered = true;
                                $triggeredRule['param'] = $param;
                                $triggeredRule['value'] = $value;
                                break;
                            }
                        }
                    }
                    break;
                    
                case 'user_agent':
                    if (isset($requestData['headers']['User-Agent']) && 
                        preg_match($rule['pattern'], $requestData['headers']['User-Agent'])) {
                        $triggered = true;
                    }
                    break;
                    
                case 'ip':
                    if (isset($requestData['ip']) && preg_match($rule['pattern'], $requestData['ip'])) {
                        $triggered = true;
                    }
                    break;
                    
                case 'method':
                    if (isset($requestData['method']) && preg_match($rule['pattern'], $requestData['method'])) {
                        $triggered = true;
                    }
                    break;
                    
                case 'header':
                    if (isset($requestData['headers']) && is_array($requestData['headers'])) {
                        foreach ($requestData['headers'] as $header => $value) {
                            if (is_string($value) && preg_match($rule['pattern'], $value)) {
                                $triggered = true;
                                break;
                            }
                        }
                    }
                    break;
            }
            
            if ($triggered) {
                $triggeredRules[] = [
                    'rule' => $rule,
                    'action' => $rule['action'],
                    'rule_id' => $rule['id']
                ];
                
                // Log rule trigger
                $this->logger->borna('Firewall rule triggered', [
                    'rule_id' => $rule['id'],
                    'rule_name' => $rule['name'],
                    'action' => $rule['action'],
                    'ip' => $requestData['ip'] ?? 'unknown',
                    'path' => $requestData['path'] ?? 'unknown'
                ]);
            }
        }
        
        return $triggeredRules;
    }
    
    /**
     * Generate a unique rule ID
     * 
     * @return string Unique ID
     */
    private function generateRuleId(): string
    {
        return 'rule_' . uniqid(mt_rand(), true);
    }
    
    /**
     * Find a rule index by ID
     * 
     * @param string $ruleId Rule ID
     * @return int|false Rule index or false if not found
     */
    private function findRuleIndexById(string $ruleId)
    {
        foreach ($this->rules as $index => $rule) {
            if ($rule['id'] === $ruleId) {
                return $index;
            }
        }
        
        return false;
    }
    
    /**
     * Validate a rule definition
     * 
     * @param array $rule Rule to validate
     * @throws \InvalidArgumentException If rule is invalid
     */
    private function validateRule(array $rule): void
    {
        // Required fields
        $requiredFields = ['name', 'type', 'pattern', 'action', 'enabled', 'description'];
        
        foreach ($requiredFields as $field) {
            if (!isset($rule[$field])) {
                throw new \InvalidArgumentException("Missing required field: {$field}");
            }
        }
        
        // Validate rule type
        $validTypes = ['path', 'parameter', 'user_agent', 'ip', 'method', 'header'];
        
        if (!in_array($rule['type'], $validTypes)) {
            throw new \InvalidArgumentException("Invalid rule type: {$rule['type']}");
        }
        
        // Validate rule action
        $validActions = ['block', 'log', 'redirect', 'challenge'];
        
        if (!in_array($rule['action'], $validActions)) {
            throw new \InvalidArgumentException("Invalid rule action: {$rule['action']}");
        }
        
        // Validate pattern is a valid regex
        try {
            // Test the pattern
            preg_match($rule['pattern'], 'test');
        } catch (\Exception $e) {
            throw new \InvalidArgumentException("Invalid regex pattern: {$rule['pattern']}");
        }
    }
    
    /**
     * Import rules from JSON
     * 
     * @param string $jsonRules JSON encoded rules
     * @param bool $overwrite Whether to overwrite existing rules
     * @return int Number of imported rules
     */
    public function importRules(string $jsonRules, bool $overwrite = false): int
    {
        $importedRules = json_decode($jsonRules, true);
        
        if (!is_array($importedRules)) {
            throw new \InvalidArgumentException("Invalid JSON rules format");
        }
        
        $this->dbConnection->beginTransaction();
        
        try {
            if ($overwrite) {
                // Clear existing rules
                $this->dbConnection->execute("DELETE FROM `{$this->rulesTable}`");
                $this->rules = [];
            }
            
            $importCount = 0;
            
            foreach ($importedRules as $rule) {
                try {
                    $this->validateRule($rule);
                    
                    // Generate new ID
                    $rule['id'] = $this->generateRuleId();
                    $rule['created_at'] = date('Y-m-d H:i:s');
                    
                    // Save to database
                    if ($this->saveRule($rule)) {
                        $this->rules[] = $rule;
                        $importCount++;
                    }
                } catch (\InvalidArgumentException $e) {
                    $this->logger->borna('Skipped invalid rule during import', [
                        'error' => $e->getMessage(),
                        'rule_name' => $rule['name'] ?? 'unknown'
                    ]);
                }
            }
            
            $this->dbConnection->commit();
            
            $this->logger->borna('Imported firewall rules', [
                'count' => $importCount,
                'overwrite' => $overwrite
            ]);
            
            return $importCount;
        } catch (\Throwable $e) {
            $this->dbConnection->rollBack();
            
            $this->logger->error('Error importing firewall rules', [
                'error' => $e->getMessage()
            ]);
            
            throw $e;
        }
    }
    
    /**
     * Export rules to JSON
     * 
     * @return string JSON encoded rules
     */
    public function exportRules(): string
    {
        return json_encode($this->rules, JSON_PRETTY_PRINT);
    }
}