<?php

namespace baseKRIZAN\BORNA\MachineLearning;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\BORNA\Storage\StorageInterface;

/**
 * Machine learning-based threat prediction
 * Uses statistical models to predict potential security threats
 */
class ThreatPredictor
{
    /**
     * Logger instance
     */
    private Logger $logger;
    
    /**
     * Model path
     */
    private string $modelPath;
    
    /**
     * Storage interface
     */
    private StorageInterface $storage;
    
    /**
     * Feature weights based on historical data
     */
    private array $featureWeights = [];
    
    /**
     * Prediction model parameters
     */
    private array $modelParameters = [];
    
    /**
     * Feature thresholds based on historical data
     */
    private array $featureThresholds = [];
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger,
        string $modelPath,
        StorageInterface $storage
    ) {
        $this->logger = $logger;
        $this->modelPath = $modelPath;
        $this->storage = $storage;
        
        // Load model parameters
        $this->loadModelParameters();
        
        $this->logger->borna('ThreatPredictor initialized', [
            'model_path' => $this->modelPath,
            'features_count' => count($this->featureWeights)
        ]);
    }
    
    /**
     * Load model parameters from storage or file
     */
    private function loadModelParameters(): void
    {
        // Pokušaj dohvatiti parametre iz storage-a
        $params = $this->storage->getRateLimit('ml_model_parameters');
        
        if ($params) {
            $this->modelParameters = $params;
            
            if (isset($params['feature_weights'])) {
                $this->featureWeights = $params['feature_weights'];
            }
            
            if (isset($params['feature_thresholds'])) {
                $this->featureThresholds = $params['feature_thresholds'];
            }
            
            return;
        }
        
        // Ako nema parametara u storage-u, inicijaliziraj defaultni model
        $this->initializeDefaultModel();
    }
    
    /**
     * Initialize default model parameters
     * This is a simple linear model with manually tuned weights
     * In a real implementation, this would be trained on actual data
     */
    private function initializeDefaultModel(): void
    {
        // Default feature weights za jednostavan linearni model
        $this->featureWeights = [
            'request_frequency' => 0.15,
            'parameter_count' => 0.10,
            'request_entropy' => 0.12,
            'path_depth' => 0.08,
            'unusual_parameters' => 0.20,
            
            // Client karakteristike
            'ip_reputation' => 0.25,
            'geolocation_risk' => 0.15,
            'user_agent_anomaly' => 0.18,
            'referrer_suspicious' => 0.12,
            'session_anomaly' => 0.20,
            
            // Karakteristike sadržaja
            'payload_entropy' => 0.22,
            'special_chars_ratio' => 0.15,
            'known_patterns' => 0.30,
            'parameter_length' => 0.10
        ];
        
        // Default pragovi za značajke
        $this->featureThresholds = [
            'request_frequency' => 30,   // zahtjevi po minuti
            'parameter_count' => 15,     // parametri po zahtjevu
            'request_entropy' => 4.5,    // entropija bitova
            'path_depth' => 5,           // segmenti puta
            'payload_entropy' => 5.0,    // entropija bitova
            'special_chars_ratio' => 0.3, // omjer specijalnih znakova
            'parameter_length' => 100     // prosječna duljina parametra
        ];
        
        $this->modelParameters = [
            'version' => '1.0.0',
            'type' => 'linear',
            'feature_weights' => $this->featureWeights,
            'feature_thresholds' => $this->featureThresholds,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Spremi model parametre u storage
        $this->saveModelParameters();
        
        $this->logger->borna('Inicijaliziran zadani model za predviđanje prijetnji');
    }
    
    /**
     * Save model parameters to storage and file
     */
    private function saveModelParameters(): void
    {
        // Ažuriraj parametre modela
        $this->modelParameters['feature_weights'] = $this->featureWeights;
        $this->modelParameters['feature_thresholds'] = $this->featureThresholds;
        $this->modelParameters['updated_at'] = date('Y-m-d H:i:s');
        
        // Spremi u storage
        $this->storage->storeRateLimit('ml_model_parameters', $this->modelParameters, 86400);
    }
    
    /**
     * Predict threat based on request data
     * 
     * @param array $requestData Request data
     * @return array Prediction result with score and details
     */
    public function predictThreat(array $requestData): array
    {
        // Extract features from request data
        $features = $this->extractFeatures($requestData);
        
        // Calculate threat score using linear model
        $score = $this->calculateThreatScore($features);
        
        // Normalize score to 0-100 range
        $normalizedScore = min(100, max(0, $score * 100));
        
        // Calculate confidence based on feature coverage
        $featureCoverage = count(array_filter($features)) / count($this->featureWeights);
        $confidence = max(0.5, $featureCoverage);
        
        $predictionResult = [
            'score' => (int)$normalizedScore,
            'confidence' => $confidence,
            'description' => $this->getDescriptionForScore($normalizedScore),
            'details' => [
                'model_version' => $this->modelParameters['version'] ?? '1.0.0',
                'model_type' => $this->modelParameters['type'] ?? 'linear',
                'features' => $features,
                'raw_score' => $score,
                'feature_importance' => $this->calculateFeatureImportance($features)
            ]
        ];
        
        // Log high-score predictions
        if ($normalizedScore > 70) {
            $this->logger->borna('High threat score prediction', [
                'ip' => $requestData['ip'] ?? 'unknown',
                'score' => $normalizedScore,
                'confidence' => $confidence,
                'path' => $requestData['path'] ?? 'unknown'
            ]);
        }
        
        // Record prediction for future training
        $this->recordPrediction($requestData, $features, $normalizedScore, $confidence);
        
        return $predictionResult;
    }
    
    /**
     * Extract features from request data
     * 
     * @param array $requestData Request data
     * @return array Extracted features
     */
    private function extractFeatures(array $requestData): array
    {
        $features = [];
        
        // Request characteristics
        $features['request_frequency'] = $this->extractRequestFrequency($requestData);
        $features['parameter_count'] = $this->extractParameterCount($requestData);
        $features['request_entropy'] = $this->extractRequestEntropy($requestData);
        $features['path_depth'] = $this->extractPathDepth($requestData);
        $features['unusual_parameters'] = $this->extractUnusualParameters($requestData);
        
        // Client characteristics
        $features['ip_reputation'] = $this->extractIpReputation($requestData);
        $features['geolocation_risk'] = $this->extractGeolocationRisk($requestData);
        $features['user_agent_anomaly'] = $this->extractUserAgentAnomaly($requestData);
        $features['referrer_suspicious'] = $this->extractReferrerSuspicious($requestData);
        $features['session_anomaly'] = $this->extractSessionAnomaly($requestData);
        
        // Content characteristics
        $features['payload_entropy'] = $this->extractPayloadEntropy($requestData);
        $features['special_chars_ratio'] = $this->extractSpecialCharsRatio($requestData);
        $features['known_patterns'] = $this->extractKnownPatterns($requestData);
        $features['parameter_length'] = $this->extractParameterLength($requestData);
        
        return $features;
    }
    
    /**
     * Calculate threat score using linear model
     * 
     * @param array $features Extracted features
     * @return float Threat score (0-1)
     */
    private function calculateThreatScore(array $features): float
    {
        $score = 0;
        $totalWeight = 0;
        
        foreach ($features as $feature => $value) {
            if (isset($this->featureWeights[$feature])) {
                $weight = $this->featureWeights[$feature];
                $score += $value * $weight;
                $totalWeight += $weight;
            }
        }
        
        // Normalize by total weight to get score in 0-1 range
        return $totalWeight > 0 ? $score / $totalWeight : 0;
    }
    
    /**
     * Calculate feature importance for this prediction
     * 
     * @param array $features Extracted features
     * @return array Feature importance
     */
    private function calculateFeatureImportance(array $features): array
    {
        $importance = [];
        
        foreach ($features as $feature => $value) {
            if (isset($this->featureWeights[$feature])) {
                $weight = $this->featureWeights[$feature];
                $importance[$feature] = $value * $weight;
            }
        }
        
        // Sort by importance descending
        arsort($importance);
        
        return $importance;
    }
    
    /**
     * Record prediction for future training
     * 
     * @param array $requestData Original request data
     * @param array $features Extracted features
     * @param float $score Predicted score
     * @param float $confidence Prediction confidence
     */
    private function recordPrediction(array $requestData, array $features, float $score, float $confidence): void
    {
        try {
            // Store prediction with timestamp
            $prediction = [
                'timestamp' => date('Y-m-d H:i:s'),
                'ip' => $requestData['ip'] ?? 'unknown',
                'path' => $requestData['path'] ?? 'unknown',
                'method' => $requestData['method'] ?? 'unknown',
                'features' => $features,
                'score' => $score,
                'confidence' => $confidence
            ];
            
            // Store in storage with auto-expiration
            $key = 'ml_prediction:' . uniqid();
            $this->storage->storeRateLimit($key, $prediction, 86400 * 30); // 30 days
            
            // Periodically collect predictions for model retraining
            // This would be handled by a separate training process
        } catch (\Throwable $e) {
            $this->logger->error('Failed to record prediction', [
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Get description text for threat score
     * 
     * @param float $score Threat score (0-100)
     * @return string Description
     */
    private function getDescriptionForScore(float $score): string
    {
        if ($score >= 90) {
            return 'Critical security threat predicted';
        } elseif ($score >= 75) {
            return 'High severity security threat predicted';
        } elseif ($score >= 50) {
            return 'Medium severity security threat predicted';
        } elseif ($score >= 25) {
            return 'Low severity security threat predicted';
        } else {
            return 'Minimal security threat predicted';
        }
    }
    
    /**
     * Extract request frequency feature
     */
    private function extractRequestFrequency(array $requestData): float
    {
        $ip = $requestData['ip'] ?? '';
        
        if (empty($ip)) {
            return 0;
        }
        
        // Get recent request count from storage
        $key = 'request_frequency:' . $ip;
        $data = $this->storage->getRateLimit($key);
        
        if (!$data) {
            // First request, store initial data
            $this->storage->storeRateLimit($key, [
                'count' => 1,
                'first_request' => time(),
                'last_request' => time()
            ], 60);
            
            return 0;
        }
        
        // Update request count
        $data['count'] = ($data['count'] ?? 0) + 1;
        $data['last_request'] = time();
        $this->storage->storeRateLimit($key, $data, 60);
        
        // Calculate requests per minute
        $elapsed = max(1, time() - ($data['first_request'] ?? time()));
        $requestsPerMinute = $data['count'] * 60 / $elapsed;
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['request_frequency'] ?? 30;
        return min(1, $requestsPerMinute / $threshold);
    }
    
    /**
     * Extract parameter count feature
     */
    private function extractParameterCount(array $requestData): float
    {
        $paramCount = 0;
        
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            $paramCount += count($requestData['query_params']);
        }
        
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            $paramCount += count($requestData['post_params']);
        }
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['parameter_count'] ?? 15;
        return min(1, $paramCount / $threshold);
    }
    
    /**
     * Extract request entropy feature
     */
    private function extractRequestEntropy(array $requestData): float
    {
        $path = $requestData['path'] ?? '';
        
        if (empty($path)) {
            return 0;
        }
        
        // Calculate Shannon entropy of the path
        $entropy = $this->calculateEntropy($path);
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['request_entropy'] ?? 4.5;
        return min(1, $entropy / $threshold);
    }
    
    /**
     * Extract path depth feature
     */
    private function extractPathDepth(array $requestData): float
    {
        $path = $requestData['path'] ?? '';
        
        if (empty($path)) {
            return 0;
        }
        
        // Count path segments
        $segments = explode('/', trim($path, '/'));
        $depth = count($segments);
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['path_depth'] ?? 5;
        return min(1, $depth / $threshold);
    }
    
    /**
     * Extract unusual parameters feature
     */
    private function extractUnusualParameters(array $requestData): float
    {
        $suspiciousParams = [
            'cmd', 'exec', 'code', 'eval', 'system', 'shell', 'passthru',
            'phpinfo', 'debug', 'test', 'root', 'admin', 'hack', 'passwd',
            'script', 'alert', 'cookie', 'document', 'location', 'window',
            'iframe', 'src', 'href', 'javascript', 'onload', 'onclick',
            'union', 'select', 'insert', 'update', 'delete', 'drop', 'alter'
        ];
        
        $unusualCount = 0;
        $totalParams = 0;
        
        // Check query parameters
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            foreach ($requestData['query_params'] as $param => $value) {
                $totalParams++;
                
                // Check parameter name
                if (in_array(strtolower($param), $suspiciousParams)) {
                    $unusualCount++;
                }
                
                // Check parameter value if it's a string
                if (is_string($value)) {
                    foreach ($suspiciousParams as $suspicious) {
                        if (stripos($value, $suspicious) !== false) {
                            $unusualCount++;
                            break;
                        }
                    }
                }
            }
        }
        
        // Check post parameters
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($requestData['post_params'] as $param => $value) {
                $totalParams++;
                
                // Check parameter name
                if (in_array(strtolower($param), $suspiciousParams)) {
                    $unusualCount++;
                }
                
                // Check parameter value if it's a string
                if (is_string($value)) {
                    foreach ($suspiciousParams as $suspicious) {
                        if (stripos($value, $suspicious) !== false) {
                            $unusualCount++;
                            break;
                        }
                    }
                }
            }
        }
        
        // Return ratio of unusual parameters
        return $totalParams > 0 ? min(1, $unusualCount / $totalParams) : 0;
    }
    
    /**
     * Extract IP reputation feature
     */
    private function extractIpReputation(array $requestData): float
    {
        $ip = $requestData['ip'] ?? '';
        
        if (empty($ip)) {
            return 0;
        }
        
        // Check if IP is in blocked list
        $key = 'ml_ip_reputation:' . $ip;
        $data = $this->storage->getRateLimit($key);
        
        if (!$data) {
            // No reputation data, start with neutral score
            $reputation = 0.5;
        } else {
            $reputation = $data['reputation'] ?? 0.5;
        }
        
        return $reputation;
    }
    
    /**
     * Extract geolocation risk feature
     */
    private function extractGeolocationRisk(array $requestData): float
    {
        // This would use a real geolocation service in production
        // For now, return 0 (no risk) for simplicity
        return 0;
    }
    
    /**
     * Extract user agent anomaly feature
     */
    private function extractUserAgentAnomaly(array $requestData): float
    {
        $userAgent = '';
        
        // Extract user agent from headers
        if (isset($requestData['headers']) && is_array($requestData['headers'])) {
            foreach ($requestData['headers'] as $header => $value) {
                if (strtolower($header) === 'user-agent' || strtolower($header) === 'http_user_agent') {
                    $userAgent = $value;
                    break;
                }
            }
        }
        
        if (empty($userAgent)) {
            // Missing user agent is suspicious
            return 0.8;
        }
        
        // Check for suspicious user agents
        $suspiciousPatterns = [
            'sqlmap', 'nikto', 'nmap', 'acunetix', 'burp', 'metasploit',
            'scanner', 'zgrab', 'masscan', 'curl/', 'wget/', 'python-requests',
            'go-http-client', 'ruby/', 'perl/', 'netcraft', 'harvest',
            'scan', 'vulnerability', 'security', 'exploit'
        ];
        
        foreach ($suspiciousPatterns as $pattern) {
            if (stripos($userAgent, $pattern) !== false) {
                return 1.0; // Highly suspicious
            }
        }
        
        // Check for common bot user agents that should identify themselves properly
        $botPatterns = ['bot', 'spider', 'crawler'];
        foreach ($botPatterns as $pattern) {
            if (stripos($userAgent, $pattern) !== false) {
                // Check if it's a well-known bot
                $knownBots = ['googlebot', 'bingbot', 'yandexbot', 'baiduspider', 'facebookexternalhit'];
                foreach ($knownBots as $knownBot) {
                    if (stripos($userAgent, $knownBot) !== false) {
                        return 0.1; // Known bot, low risk
                    }
                }
                
                return 0.6; // Unknown bot, moderate risk
            }
        }
        
        // Calculate entropy of user agent
        $entropy = $this->calculateEntropy($userAgent);
        
        // Very low entropy may indicate suspicious user agent
        if ($entropy < 2) {
            return 0.7;
        }
        
        // Extremely high entropy may also be suspicious
        if ($entropy > 6) {
            return 0.5;
        }
        
        return 0.2; // Normal user agent
    }
    
    /**
     * Extract referrer suspicious feature
     */
    private function extractReferrerSuspicious(array $requestData): float
    {
        $referrer = '';
        
        // Extract referrer from headers
        if (isset($requestData['headers']) && is_array($requestData['headers'])) {
            foreach ($requestData['headers'] as $header => $value) {
                if (strtolower($header) === 'referer' || strtolower($header) === 'http_referer') {
                    $referrer = $value;
                    break;
                }
            }
        }
        
        if (empty($referrer)) {
            // Missing referrer is slightly suspicious for some requests
            return 0.3;
        }
        
        // Check for suspicious referrers
        $suspiciousPatterns = [
            'localhost', '127.0.0.1', 'file://', 'data:', 'javascript:',
            '<script', 'eval(', 'alert(', 'document.cookie', 'onload=',
            'union+select', 'concat(', 'information_schema'
        ];
        
        foreach ($suspiciousPatterns as $pattern) {
            if (stripos($referrer, $pattern) !== false) {
                return 1.0; // Highly suspicious
            }
        }
        
        return 0.1; // Normal referrer
    }
    
    /**
     * Extract session anomaly feature
     */
    private function extractSessionAnomaly(array $requestData): float
    {
        $ip = $requestData['ip'] ?? '';
        $sessionId = $requestData['session_id'] ?? '';
        
        if (empty($ip) || empty($sessionId)) {
            return 0.5;
        }
        
        // Check for session changes
        $key = 'ml_session:' . $ip;
        $data = $this->storage->getRateLimit($key);
        
        if (!$data) {
            // First session for this IP
            $this->storage->storeRateLimit($key, [
                'session_id' => $sessionId,
                'change_count' => 0,
                'first_seen' => time(),
                'last_seen' => time()
            ], 3600);
            
            return 0.1;
        }
        
        // Check if session has changed
        if ($data['session_id'] !== $sessionId) {
            // Session changed, update data
            $changeCount = ($data['change_count'] ?? 0) + 1;
            $elapsed = max(1, (time() - ($data['first_seen'] ?? time())) / 3600); // hours
            
            $this->storage->storeRateLimit($key, [
                'session_id' => $sessionId,
                'change_count' => $changeCount,
                'first_seen' => $data['first_seen'] ?? time(),
                'last_seen' => time()
            ], 3600);
            
            // Calculate changes per hour
            $changesPerHour = $changeCount / $elapsed;
            
            // More than 3 changes per hour is suspicious
            return min(1, $changesPerHour / 3);
        }
        
        // Update last seen
        $data['last_seen'] = time();
        $this->storage->storeRateLimit($key, $data, 3600);
        
        return 0.1; // Normal session
    }
    
    /**
     * Extract payload entropy feature
     */
    private function extractPayloadEntropy(array $requestData): float
    {
        $payload = '';
        
        // Combine all post parameters into a single string
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($requestData['post_params'] as $param => $value) {
                if (is_string($value)) {
                    $payload .= $value;
                }
            }
        }
        
        if (empty($payload)) {
            return 0;
        }
        
        // Calculate entropy of payload
        $entropy = $this->calculateEntropy($payload);
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['payload_entropy'] ?? 5.0;
        return min(1, $entropy / $threshold);
    }
    
    /**
     * Extract special characters ratio feature
     */
    private function extractSpecialCharsRatio(array $requestData): float
    {
        $payload = '';
        
        // Combine all parameters into a single string
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            foreach ($requestData['query_params'] as $param => $value) {
                if (is_string($value)) {
                    $payload .= $value;
                }
            }
        }
        
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($requestData['post_params'] as $param => $value) {
                if (is_string($value)) {
                    $payload .= $value;
                }
            }
        }
        
        if (empty($payload)) {
            return 0;
        }
        
        // Count special characters
        $specialChars = preg_match_all('/[^a-zA-Z0-9\s]/', $payload, $matches);
        $ratio = $specialChars / strlen($payload);
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['special_chars_ratio'] ?? 0.3;
        return min(1, $ratio / $threshold);
    }
    
    /**
     * Extract known attack patterns feature
     */
    private function extractKnownPatterns(array $requestData): float
    {
        // Attack pattern categories
        $attackPatterns = [
            // SQL Injection patterns
            'sqli' => [
                '/UNION\s+SELECT/i',
                '/SELECT.+FROM.+WHERE/i',
                '/INSERT\s+INTO/i',
                '/UPDATE.+SET/i',
                '/DELETE\s+FROM/i',
                '/DROP\s+TABLE/i',
                '/ALTER\s+TABLE/i',
                '/EXEC\s+xp_/i',
                '/EXEC\s+sp_/i',
                '/DECLARE\s+@/i',
                '/WAITFOR\s+DELAY/i',
                '/CAST\(.+AS/i',
                '/CONVERT\(.+,\s*\w+\)/i',
                '/;\s*--/i',
                '/`/i'
            ],
            
            // XSS patterns
            'xss' => [
                '/<script>/i',
                '/javascript:/i',
                '/\bon\w+\s*=/i',
                '/alert\s*\(/i',
                '/eval\s*\(/i',
                '/document\./i',
                '/window\./i',
                '/String\.fromCharCode/i',
                '/eval\(/i',
                '/setTimeout\(/i',
                '/setInterval\(/i',
                '/new\s+Function\(/i',
                '/location\./i',
                '/innerHTML/i'
            ],
            
            // Path traversal patterns
            'path_traversal' => [
                '/\.\.\//i',
                '/\.\.\\\/i',
                '/%2e%2e\//i',
                '/%252e%252e\//i',
                '/\.\.%2f/i',
                '/\.\.%5c/i'
            ],
            
            // Command injection patterns
            'command_injection' => [
                '/;\s*\w+/i',
                '/\|\s*\w+/i',
                '/`\s*\w+/i',
                '/\$\(\s*\w+/i',
                '/system\s*\(/i',
                '/exec\s*\(/i',
                '/shell_exec\s*\(/i',
                '/passthru\s*\(/i',
                '/proc_open\s*\(/i',
                '/popen\s*\(/i'
            ],
            
            // LDAP injection patterns
            'ldap_injection' => [
                '/\*\)/i',
                '/\(\|\(/i',
                '/\)\|\(/i',
                '/\(\&\(/i',
                '/\)\&\(/i'
            ]
        ];
        
        $matchCount = 0;
        $totalPatterns = 0;
        
        foreach ($attackPatterns as $category => $patterns) {
            $totalPatterns += count($patterns);
            
            foreach ($patterns as $pattern) {
                $found = false;
                
                // Check query parameters
                if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
                    foreach ($requestData['query_params'] as $param => $value) {
                        if (is_string($value) && preg_match($pattern, $value)) {
                            $matchCount++;
                            $found = true;
                            break;
                        }
                    }
                }
                
                if ($found) {
                    continue;
                }
                
                // Check post parameters
                if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
                    foreach ($requestData['post_params'] as $param => $value) {
                        if (is_string($value) && preg_match($pattern, $value)) {
                            $matchCount++;
                            break;
                        }
                    }
                }
            }
        }
        
        // Calculate match ratio (matches / total patterns)
        $matchRatio = $totalPatterns > 0 ? $matchCount / $totalPatterns : 0;
        
        // Even a small number of matches is significant
        return min(1, $matchRatio * 10);
    }
    
    /**
     * Extract parameter length feature
     */
    private function extractParameterLength(array $requestData): float
    {
        $totalLength = 0;
        $paramCount = 0;
        
        // Calculate average parameter length
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            foreach ($requestData['query_params'] as $param => $value) {
                if (is_string($value)) {
                    $totalLength += strlen($value);
                    $paramCount++;
                }
            }
        }
        
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($requestData['post_params'] as $param => $value) {
                if (is_string($value)) {
                    $totalLength += strlen($value);
                    $paramCount++;
                }
            }
        }
        
        if ($paramCount === 0) {
            return 0;
        }
        
        $avgLength = $totalLength / $paramCount;
        
        // Normalize based on threshold
        $threshold = $this->featureThresholds['parameter_length'] ?? 100;
        return min(1, $avgLength / $threshold);
    }
    
    /**
     * Calculate Shannon entropy of a string
     */
    private function calculateEntropy(string $string): float
    {
        if (empty($string)) {
            return 0;
        }
        
        $length = strlen($string);
        $frequencies = [];
        
        // Count character frequencies
        for ($i = 0; $i < $length; $i++) {
            $char = $string[$i];
            if (!isset($frequencies[$char])) {
                $frequencies[$char] = 0;
            }
            $frequencies[$char]++;
        }
        
        // Calculate entropy
        $entropy = 0;
        foreach ($frequencies as $count) {
            $probability = $count / $length;
            $entropy -= $probability * log($probability, 2);
        }
        
        return $entropy;
    }
}