<?php

namespace baseKRIZAN\BORNA;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\BORNA\Storage\StorageInterface;

/**
 * Security events management for BORNA security module
 * Manages security-related events using database/redis storage
 */
class SecurityEvents
{
    private Logger $logger;
    private StorageInterface $storage;
    private int $maxEvents = 10000; // Limit the number of stored events
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger, 
        StorageInterface $storage
    ) {
        $this->logger = $logger;
        $this->storage = $storage;
        
        $this->logger->borna('SecurityEvents initialized with storage', [
            'storage_type' => get_class($storage)
        ]);
    }
    
    /**
     * Record a security event
     * 
     * @param string $eventType Type of security event
     * @param array $data Event data
     * @return bool Success status
     */
    public function recordEvent(string $eventType, array $data = []): bool
    {
        // Ensure valid event type
        if (!$this->isValidEventType($eventType)) {
            $this->logger->borna('Invalid security event type', [
                'event_type' => $eventType
            ]);
            return false;
        }
        
        // Store event using storage interface
        $stored = $this->storage->storeEvent($eventType, $data);
        
        // Log if it's a significant event
        $this->logSignificantEvent($eventType, $data);
        
        return $stored;
    }
    
    /**
     * Get security events for a time period
     * 
     * @param \DateTime $from Start time
     * @param \DateTime $to End time
     * @param string|null $eventType Optional filter by event type
     * @return array Matching events
     */
    public function getEvents(\DateTime $from, \DateTime $to, string $eventType = null): array
    {
        // Convert DateTime to timestamps
        $fromTimestamp = $from->getTimestamp();
        $toTimestamp = $to->getTimestamp();
        
        // Use storage interface to retrieve events
        return $this->storage->getEvents($fromTimestamp, $toTimestamp, $eventType);
    }
    
    /**
     * Get event counts by type
     * 
     * @param \DateTime $from Start time
     * @param \DateTime $to End time
     * @return array Event counts by type
     */
    public function getEventCounts(\DateTime $from, \DateTime $to): array
    {
        $fromTimestamp = $from->getTimestamp();
        $toTimestamp = $to->getTimestamp();
        
        $events = $this->storage->getEvents($fromTimestamp, $toTimestamp);
        $counts = [];
        
        foreach ($events as $event) {
            $type = $event['event_type'];
            
            if (!isset($counts[$type])) {
                $counts[$type] = 0;
            }
            
            $counts[$type]++;
        }
        
        return $counts;
    }
    
    /**
     * Get recent events
     * 
     * @param int $limit Maximum number of events to return
     * @param string|null $eventType Optional filter by event type
     * @return array Recent events
     */
    public function getRecentEvents(int $limit = 100, string $eventType = null): array
    {
        // Use current time as end point
        $to = new \DateTime();
        
        // Get events from last 30 days as default range
        $from = new \DateTime('-30 days');
        
        $events = $this->storage->getEvents($from->getTimestamp(), $to->getTimestamp(), $eventType);
        
        // Sort by timestamp descending and limit
        usort($events, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });
        
        return array_slice($events, 0, $limit);
    }
    
    /**
     * Get most active IPs based on security events
     * 
     * @param int $limit Maximum number of IPs to return
     * @param \DateTime|null $from Optional start time
     * @param \DateTime|null $to Optional end time
     * @return array IPs with their event counts
     */
    public function getMostActiveIPs(int $limit = 10, \DateTime $from = null, \DateTime $to = null): array
    {
        // If time range not specified, use last 30 days
        $fromTimestamp = ($from ?? new \DateTime('-30 days'))->getTimestamp();
        $toTimestamp = ($to ?? new \DateTime())->getTimestamp();
        
        $events = $this->storage->getEvents($fromTimestamp, $toTimestamp);
        
        $ipCounts = [];
        
        foreach ($events as $event) {
            if (isset($event['data']['ip'])) {
                $ip = $event['data']['ip'];
                
                if (!isset($ipCounts[$ip])) {
                    $ipCounts[$ip] = 0;
                }
                
                $ipCounts[$ip]++;
            }
        }
        
        // Sort by count descending
        arsort($ipCounts);
        
        return array_slice($ipCounts, 0, $limit, true);
    }
    
    /**
     * Get most attacked paths
     * 
     * @param int $limit Maximum number of paths to return
     * @param \DateTime|null $from Optional start time
     * @param \DateTime|null $to Optional end time
     * @return array Paths with their attack counts
     */
    public function getMostAttackedPaths(int $limit = 10, \DateTime $from = null, \DateTime $to = null): array
    {
        // If time range not specified, use last 30 days
        $fromTimestamp = ($from ?? new \DateTime('-30 days'))->getTimestamp();
        $toTimestamp = ($to ?? new \DateTime())->getTimestamp();
        
        $events = $this->storage->getEvents($fromTimestamp, $toTimestamp, 'attack_detected');
        
        $pathCounts = [];
        
        foreach ($events as $event) {
            if (isset($event['data']['path'])) {
                $path = $event['data']['path'];
                
                if (!isset($pathCounts[$path])) {
                    $pathCounts[$path] = 0;
                }
                
                $pathCounts[$path]++;
            }
        }
        
        // Sort by count descending
        arsort($pathCounts);
        
        return array_slice($pathCounts, 0, $limit, true);
    }
    
    /**
     * Export events to a file (optional, if needed)
     * 
     * @param string $format Export format ('json' or 'csv')
     * @param string|null $filename Optional custom filename
     * @return string Path to the exported file
     */
    public function exportEvents(string $format = 'json', string $filename = null): string
    {
        // Trenutni timestamp za generiranje jedinstvenog naziva datoteke
        $timestamp = date('Y-m-d_H-i-s');
        
        if ($filename === null) {
            $filename = "security_events_{$timestamp}.{$format}";
        }
        
        // Koristiti sistemski konfigurirani direktorij za privremene datoteke
        $exportPath = \baseKRIZAN\Config\Config::get('paths.logs') . '/exports';
        
        if (!is_dir($exportPath)) {
            mkdir($exportPath, 0755, true);
        }
        
        $fullPath = $exportPath . '/' . $filename;
        
        // Dohvati sve događaje - posljednjih 30 dana
        $from = new \DateTime('-30 days');
        $to = new \DateTime();
        $events = $this->getEvents($from, $to);
        
        if ($format === 'json') {
            file_put_contents($fullPath, json_encode($events, JSON_PRETTY_PRINT));
        } elseif ($format === 'csv') {
            $fp = fopen($fullPath, 'w');
            
            // Zapiši CSV zaglavlje
            fputcsv($fp, ['ID', 'Event Type', 'Timestamp', 'IP', 'Path', 'Score', 'Details']);
            
            // Zapiši podatke događaja
            foreach ($events as $event) {
                $row = [
                    $event['id'] ?? '',
                    $event['event_type'] ?? '',
                    $event['timestamp'] ?? '',
                    $event['data']['ip'] ?? '',
                    $event['data']['path'] ?? '',
                    $event['data']['score'] ?? '',
                    json_encode($event['data'])
                ];
                
                fputcsv($fp, $row);
            }
            
            fclose($fp);
        } else {
            throw new \InvalidArgumentException("Nepodržani format exporta: {$format}");
        }
        
        $this->logger->borna('Exportirani sigurnosni događaji', [
            'format' => $format,
            'file' => $fullPath,
            'count' => count($events)
        ]);
        
        return $fullPath;
    }
    
    /**
     * Check if an event type is valid
     * 
     * @param string $eventType Event type to check
     * @return bool True if valid
     */
    private function isValidEventType(string $eventType): bool
    {
        $validTypes = [
            'attack_detected', 'ip_blocked', 'ip_unblocked', 
            'login_failure', 'bruteforce_attempt', 'unauthorized_access',
            'session_hijacking', 'session_fixation', 'csrf_attempt',
            'xss_attempt', 'sqli_attempt', 'file_inclusion_attempt',
            'command_injection', 'rate_limit_exceeded', 'geolocation_blocked',
            'suspicious_activity', 'security_level_changed', 
            'firewall_rule_triggered', 'security_scan_detected',
            'admin_login', 'config_changed'
        ];
        
        return in_array($eventType, $validTypes);
    }
    
    /**
     * Log significant security events
     * 
     * @param string $eventType Event type
     * @param array $data Event data
     */
    private function logSignificantEvent(string $eventType, array $data): void
    {
        $significantEvents = [
            'attack_detected', 'ip_blocked', 'bruteforce_attempt',
            'unauthorized_access', 'session_hijacking', 'csrf_attempt',
            'xss_attempt', 'sqli_attempt', 'file_inclusion_attempt',
            'command_injection', 'geolocation_blocked'
        ];
        
        if (in_array($eventType, $significantEvents)) {
            $this->logger->borna('Sigurnosni događaj: ' . $eventType, [
                'podaci_događaja' => $data
            ]);
        }
    }



    /**
     * Get attack events for a specific time period
     * 
     * @param \DateTime $from Start time
     * @param \DateTime $to End time
     * @return array Attack events
     */
    public function getAttackEvents(\DateTime $from, \DateTime $to): array
    {
        // Get all events
        $events = $this->getEvents($from, $to);
        
        // Filter for attack events
        $attackEvents = array_filter($events, function($event) {
            return $event['event_type'] === 'attack_detected';
        });
        
        return $attackEvents;
    }
}