<?php

namespace baseKRIZAN\BORNA\Storage;

use baseKRIZAN\Error\Logger;
use baseKRIZAN\Database\DatabaseConnection;

/**
 * Database-based storage implementation for BORNA security data
 * Provides persistent, scalable storage for security telemetry
 */
class DatabaseStorage implements StorageInterface
{
    /**
     * Logger instance
     */
    private Logger $logger;
    
    /**
     * DatabaseConnection instance
     */
    private DatabaseConnection $dbConnection;
    
    /**
     * Table names
     */
    private string $blockedIpsTable = 'security_borna_blocked_ips';
    private string $eventsTable = 'security_borna_security_events';
    private string $fingerprintsTable = 'security_borna_fingerprints';
    private string $profilesTable = 'security_borna_profiles';
    private string $integrityTable = 'security_borna_integrity_hashes';
    private string $rateLimitsTable = 'security_borna_rate_limits';
    private string $anomalyTable = 'security_borna_anomaly_data';
    private string $metricsTable = 'security_borna_metrics';
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger,
        ?DatabaseConnection $dbConnection = null
    ) {
        $this->logger = $logger;
        
        // Use provided DatabaseConnection or create a new one
        if ($dbConnection !== null) {
            $this->dbConnection = $dbConnection;
        } else {
            // Auto-create DatabaseConnection
            try {
                $this->dbConnection = DatabaseConnection::getInstance();
            } catch (\Throwable $e) {
                $this->logger->error('Failed to create database connection for BORNA storage', [
                    'error' => $e->getMessage()
                ]);
                throw $e;
            }
        }
        
        // Custom table names from config if provided
        $this->loadCustomTableNames();
        
        // Ensure all required tables exist
        $this->ensureTablesExist();
        
        $this->logger->borna('BORNA DatabaseStorage initialized');
    }
    
    /**
     * Load custom table names from configuration
     */
    private function loadCustomTableNames(): void
    {
        $configPrefix = 'borna.tables.';
        
        $tableConfigs = [
            'blockedIpsTable' => $configPrefix . 'blocked_ips',
            'eventsTable' => $configPrefix . 'events',
            'fingerprintsTable' => $configPrefix . 'fingerprints',
            'profilesTable' => $configPrefix . 'profiles',
            'integrityTable' => $configPrefix . 'integrity',
            'rateLimitsTable' => $configPrefix . 'rate_limits',
            'anomalyTable' => $configPrefix . 'anomaly',
            'metricsTable' => $configPrefix . 'metrics'
        ];
        
        foreach ($tableConfigs as $property => $configKey) {
            $customName = \baseKRIZAN\Config\Config::get($configKey);
            if ($customName) {
                $this->$property = $customName;
            }
        }
    }
    
    /**
     * Ensure all required tables exist
     */
    private function ensureTablesExist(): void
    {
        $tables = [
            $this->blockedIpsTable => $this->getBlockedIpsTableSchema(),
            $this->eventsTable => $this->getEventsTableSchema(),
            $this->fingerprintsTable => $this->getFingerprintsTableSchema(),
            $this->profilesTable => $this->getProfilesTableSchema(),
            $this->integrityTable => $this->getIntegrityTableSchema(),
            $this->rateLimitsTable => $this->getRateLimitsTableSchema(),
            $this->anomalyTable => $this->getAnomalyTableSchema(),
            $this->metricsTable => $this->getMetricsTableSchema()
        ];
        
        foreach ($tables as $table => $schema) {
            $this->createTableIfNotExists($table, $schema);
        }
    }
    
    /**
     * Create a table if it doesn't exist
     */
    private function createTableIfNotExists(string $table, string $schema): void
    {
        try {
            // Validate table name
            if (!$this->isValidTableName($table)) {
                throw new \InvalidArgumentException("Invalid table name: {$table}");
            }
            
            // Check if table exists
            $tableExists = $this->dbConnection->querySingleValue(
                "SELECT 1 FROM information_schema.tables WHERE table_name = :table AND table_schema = DATABASE()",
                ['table' => $table]
            );
            
            if ($tableExists === null) {
                // Table doesn't exist, create it
                $this->dbConnection->execute($schema);
                
                $this->logger->borna("Created BORNA table: {$table}");
            }
        } catch (\Throwable $e) {
            $this->logger->error("Failed to create BORNA table: {$table}", [
                'error' => $e->getMessage(),
                'table' => $table
            ]);
        }
    }
    
    /**
     * Validate table name to prevent SQL injection
     */
    private function isValidTableName(string $tableName): bool
    {
        return preg_match('/^[a-zA-Z0-9_]+$/', $tableName) === 1;
    }
    
    /**
     * Get blocked IPs table schema
     */
    private function getBlockedIpsTableSchema(): string
    {
        return "CREATE TABLE `{$this->blockedIpsTable}` (
            `ip` VARCHAR(45) NOT NULL PRIMARY KEY,
            `reason` TEXT NOT NULL,
            `timestamp` INT UNSIGNED NOT NULL,
            `expires` INT UNSIGNED NOT NULL,
            `created_by` VARCHAR(50) NULL,
            `user_agent` TEXT NULL,
            `country_code` VARCHAR(2) NULL,
            INDEX `idx_expires` (`expires`)
        )";
    }
    
    /**
     * Get security events table schema
     */
    private function getEventsTableSchema(): string
    {
        return "CREATE TABLE `{$this->eventsTable}` (
            `id` VARCHAR(36) NOT NULL PRIMARY KEY,
            `event_type` VARCHAR(50) NOT NULL,
            `timestamp` DATETIME NOT NULL,
            `ip` VARCHAR(45) NULL,
            `path` VARCHAR(255) NULL,
            `data` JSON NULL,
            `score` INT UNSIGNED NULL,
            INDEX `idx_event_type` (`event_type`),
            INDEX `idx_timestamp` (`timestamp`),
            INDEX `idx_ip` (`ip`)
        )";
    }
    
    /**
     * Get fingerprints table schema
     */
    private function getFingerprintsTableSchema(): string
    {
        return "CREATE TABLE `{$this->fingerprintsTable}` (
            `fingerprint` VARCHAR(64) NOT NULL PRIMARY KEY,
            `ip` VARCHAR(45) NOT NULL,
            `user_agent` TEXT NULL,
            `created` INT UNSIGNED NOT NULL,
            `expires` INT UNSIGNED NOT NULL,
            `session_id` VARCHAR(64) NULL,
            `user_id` VARCHAR(36) NULL,
            `data` JSON NULL,
            INDEX `idx_expires` (`expires`),
            INDEX `idx_ip` (`ip`),
            INDEX `idx_session_id` (`session_id`),
            INDEX `idx_user_id` (`user_id`)
        )";
    }
    
    /**
     * Get profiles table schema
     */
    private function getProfilesTableSchema(): string
    {
        return "CREATE TABLE `{$this->profilesTable}` (
            `client_id_hash` VARCHAR(64) NOT NULL PRIMARY KEY,
            `client_id` VARCHAR(255) NOT NULL,
            `profile_data` JSON NOT NULL,
            `created_at` INT UNSIGNED NOT NULL,
            `updated_at` INT UNSIGNED NOT NULL,
            `last_seen` INT UNSIGNED NOT NULL,
            INDEX `idx_last_seen` (`last_seen`)
        )";
    }
    
    /**
     * Get integrity hashes table schema
     */
    private function getIntegrityTableSchema(): string
    {
        return "CREATE TABLE `{$this->integrityTable}` (
            `file_path_hash` VARCHAR(64) NOT NULL PRIMARY KEY,
            `file_path` TEXT NOT NULL,
            `hash` VARCHAR(64) NOT NULL,
            `first_seen` INT UNSIGNED NOT NULL,
            `last_verified` INT UNSIGNED NOT NULL,
            `change_count` INT UNSIGNED DEFAULT 0,
            `latest_status` ENUM('valid', 'modified', 'missing') NOT NULL DEFAULT 'valid'
        )";
    }
    
    /**
     * Get rate limits table schema
     */
    private function getRateLimitsTableSchema(): string
    {
        return "CREATE TABLE `{$this->rateLimitsTable}` (
            `key_hash` VARCHAR(64) NOT NULL PRIMARY KEY,
            `key_name` VARCHAR(255) NOT NULL,
            `data` JSON NOT NULL,
            `created_at` INT UNSIGNED NOT NULL,
            `expires` INT UNSIGNED NOT NULL,
            INDEX `idx_expires` (`expires`),
            INDEX `idx_key_name` (`key_name`)
        )";
    }
    
    /**
     * Get anomaly data table schema
     */
    private function getAnomalyTableSchema(): string
    {
        return "CREATE TABLE `{$this->anomalyTable}` (
            `id` BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `metric` VARCHAR(50) NOT NULL,
            `value` DOUBLE NOT NULL,
            `timestamp` DATETIME NOT NULL,
            `ip` VARCHAR(45) NULL,
            `path` VARCHAR(255) NULL,
            `is_anomaly` TINYINT(1) DEFAULT 0,
            `z_score` DOUBLE NULL,
            `metadata` JSON NULL,
            INDEX `idx_metric` (`metric`),
            INDEX `idx_timestamp` (`timestamp`),
            INDEX `idx_is_anomaly` (`is_anomaly`)
        )";
    }
    
    /**
     * Get metrics statistics table schema
     */
    private function getMetricsTableSchema(): string
    {
        return "CREATE TABLE `{$this->metricsTable}` (
            `metric` VARCHAR(50) NOT NULL PRIMARY KEY,
            `count` BIGINT UNSIGNED DEFAULT 0,
            `min` DOUBLE NULL,
            `max` DOUBLE NULL,
            `sum` DOUBLE DEFAULT 0,
            `sum_squares` DOUBLE DEFAULT 0,
            `mean` DOUBLE NULL,
            `variance` DOUBLE NULL,
            `stddev` DOUBLE NULL,
            `last_updated` INT UNSIGNED NOT NULL
        )";
    }
    
    /**
     * @inheritDoc
     */
    public function getBlockedIPs(): array
    {
        try {
            $rows = $this->dbConnection->queryAndFetchAllAssoc(
                "SELECT * FROM `{$this->blockedIpsTable}` WHERE `expires` > :now",
                ['now' => time()]
            );
            
            if (empty($rows)) {
                return [];
            }
            
            $blockedIPs = [];
            foreach ($rows as $row) {
                $blockedIPs[$row['ip']] = [
                    'ip' => $row['ip'],
                    'reason' => $row['reason'],
                    'timestamp' => $row['timestamp'],
                    'expires' => $row['expires']
                ];
                
                // Add additional fields if they exist
                foreach (['created_by', 'user_agent', 'country_code'] as $field) {
                    if (isset($row[$field])) {
                        $blockedIPs[$row['ip']][$field] = $row[$field];
                    }
                }
            }
            
            return $blockedIPs;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get blocked IPs from database', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
    
    /**
     * @inheritDoc
     */
    public function saveBlockedIPs(array $blockedIPs): bool
    {
        try {
            // Begin transaction for better performance
            $this->dbConnection->beginTransaction();
            
            // Clear existing blocked IPs
            $this->dbConnection->execute("DELETE FROM `{$this->blockedIpsTable}`");
            
            // Insert new blocked IPs
            $sql = "
                INSERT INTO `{$this->blockedIpsTable}` 
                (`ip`, `reason`, `timestamp`, `expires`, `created_by`, `user_agent`, `country_code`) 
                VALUES 
                (:ip, :reason, :timestamp, :expires, :created_by, :user_agent, :country_code)
            ";
            
            foreach ($blockedIPs as $ip => $data) {
                $this->dbConnection->execute($sql, [
                    'ip' => $ip,
                    'reason' => $data['reason'] ?? '',
                    'timestamp' => $data['timestamp'] ?? time(),
                    'expires' => $data['expires'] ?? (time() + 86400), // Default 24h
                    'created_by' => $data['created_by'] ?? null,
                    'user_agent' => $data['user_agent'] ?? null,
                    'country_code' => $data['country_code'] ?? null
                ]);
            }
            
            // Commit transaction
            $this->dbConnection->commit();
            
            return true;
        } catch (\Throwable $e) {
            // Rollback on error
            if ($this->dbConnection->inTransaction()) {
                $this->dbConnection->rollBack();
            }
            
            $this->logger->error('Failed to save blocked IPs to database', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeEvent(string $eventType, array $eventData): bool
    {
        try {
            // Generate UUID for event ID
            $eventId = $this->generateUuid();
            
            // Extract common fields from event data
            $ip = $eventData['ip'] ?? null;
            $path = $eventData['path'] ?? null;
            $score = $eventData['score'] ?? null;
            
            // Store event in database
            $sql = "
                INSERT INTO `{$this->eventsTable}` 
                (`id`, `event_type`, `timestamp`, `ip`, `path`, `data`, `score`) 
                VALUES 
                (:id, :event_type, :timestamp, :ip, :path, :data, :score)
            ";
            
            $this->dbConnection->execute($sql, [
                'id' => $eventId,
                'event_type' => $eventType,
                'timestamp' => date('Y-m-d H:i:s'),
                'ip' => $ip,
                'path' => $path,
                'data' => json_encode($eventData, JSON_UNESCAPED_UNICODE),
                'score' => $score
            ]);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store security event in database', [
                'error' => $e->getMessage(),
                'event_type' => $eventType
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getEvents(int $fromTimestamp, int $toTimestamp, ?string $eventType = null): array
    {
        try {
            $sql = "
                SELECT * FROM `{$this->eventsTable}` 
                WHERE `timestamp` BETWEEN :from_date AND :to_date
            ";
            
            $params = [
                'from_date' => date('Y-m-d H:i:s', $fromTimestamp),
                'to_date' => date('Y-m-d H:i:s', $toTimestamp)
            ];
            
            if ($eventType !== null) {
                $sql .= " AND `event_type` = :event_type";
                $params['event_type'] = $eventType;
            }
            
            // Order by timestamp descending
            $sql .= " ORDER BY `timestamp` DESC";
            
            $rows = $this->dbConnection->queryAndFetchAllAssoc($sql, $params);
            
            $events = [];
            foreach ($rows as $row) {
                // Parse data JSON
                $data = json_decode($row['data'], true) ?? [];
                
                $events[] = [
                    'id' => $row['id'],
                    'event_type' => $row['event_type'],
                    'timestamp' => $row['timestamp'],
                    'data' => $data
                ];
            }
            
            return $events;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get security events from database', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeClientFingerprint(string $fingerprint, array $data): bool
    {
        try {
            $sql = "
                REPLACE INTO `{$this->fingerprintsTable}`
                (`fingerprint`, `ip`, `user_agent`, `created`, `expires`, `session_id`, `user_id`, `data`)
                VALUES
                (:fingerprint, :ip, :user_agent, :created, :expires, :session_id, :user_id, :data)
            ";
            
            $this->dbConnection->execute($sql, [
                'fingerprint' => $fingerprint,
                'ip' => $data['ip'] ?? '',
                'user_agent' => $data['user_agent'] ?? null,
                'created' => $data['created'] ?? time(),
                'expires' => $data['expires'] ?? (time() + 86400), // Default 24h
                'session_id' => $data['session_id'] ?? null,
                'user_id' => $data['user_id'] ?? null,
                'data' => json_encode($data, JSON_UNESCAPED_UNICODE)
            ]);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store client fingerprint in database', [
                'error' => $e->getMessage(),
                'fingerprint' => substr($fingerprint, 0, 16) . '...'
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getClientFingerprint(string $fingerprint): ?array
    {
        try {
            $row = $this->dbConnection->queryAndFetchAssoc(
                "SELECT * FROM `{$this->fingerprintsTable}` WHERE `fingerprint` = :fingerprint AND `expires` > :now",
                [
                    'fingerprint' => $fingerprint,
                    'now' => time()
                ]
            );
            
            if (!$row) {
                return null;
            }
            
            // Parse stored JSON data
            $data = json_decode($row['data'], true) ?? [];
            
            // Add common fields
            $data['ip'] = $row['ip'];
            $data['user_agent'] = $row['user_agent'];
            $data['created'] = $row['created'];
            $data['expires'] = $row['expires'];
            $data['session_id'] = $row['session_id'];
            $data['user_id'] = $row['user_id'];
            
            return $data;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get client fingerprint from database', [
                'error' => $e->getMessage(),
                'fingerprint' => substr($fingerprint, 0, 16) . '...'
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeBehaviorProfile(string $clientId, array $profile): bool
    {
        try {
            $clientIdHash = hash('sha256', $clientId);
            $now = time();
            
            $sql = "
                INSERT INTO `{$this->profilesTable}`
                (`client_id_hash`, `client_id`, `profile_data`, `created_at`, `updated_at`, `last_seen`)
                VALUES
                (:client_id_hash, :client_id, :profile_data, :created_at, :updated_at, :last_seen)
                ON DUPLICATE KEY UPDATE
                `profile_data` = :profile_data,
                `updated_at` = :updated_at,
                `last_seen` = :last_seen
            ";
            
            $this->dbConnection->execute($sql, [
                'client_id_hash' => $clientIdHash,
                'client_id' => $clientId,
                'profile_data' => json_encode($profile, JSON_UNESCAPED_UNICODE),
                'created_at' => $now,
                'updated_at' => $now,
                'last_seen' => $now
            ]);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store behavior profile in database', [
                'error' => $e->getMessage(),
                'client_id' => substr($clientId, 0, 16) . '...'
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getBehaviorProfile(string $clientId): ?array
    {
        try {
            $clientIdHash = hash('sha256', $clientId);
            
            $row = $this->dbConnection->queryAndFetchAssoc(
                "SELECT * FROM `{$this->profilesTable}` WHERE `client_id_hash` = :client_id_hash",
                ['client_id_hash' => $clientIdHash]
            );
            
            if (!$row) {
                return null;
            }
            
            // Update last seen timestamp
            $this->dbConnection->execute(
                "UPDATE `{$this->profilesTable}` SET `last_seen` = :now WHERE `client_id_hash` = :client_id_hash",
                [
                    'now' => time(),
                    'client_id_hash' => $clientIdHash
                ]
            );
            
            // Return parsed profile data
            return json_decode($row['profile_data'], true) ?? [];
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get behavior profile from database', [
                'error' => $e->getMessage(),
                'client_id' => substr($clientId, 0, 16) . '...'
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeIntegrityHash(string $filePath, string $hash): bool
    {
        try {
            $filePathHash = hash('sha256', $filePath);
            $now = time();
            
            $sql = "
                INSERT INTO `{$this->integrityTable}`
                (`file_path_hash`, `file_path`, `hash`, `first_seen`, `last_verified`, `latest_status`)
                VALUES
                (:file_path_hash, :file_path, :hash, :first_seen, :last_verified, 'valid')
                ON DUPLICATE KEY UPDATE
                `hash` = :hash,
                `last_verified` = :last_verified,
                `change_count` = `change_count` + IF(hash != :hash, 1, 0),
                `latest_status` = IF(hash != :hash, 'modified', 'valid')
            ";
            
            $this->dbConnection->execute($sql, [
                'file_path_hash' => $filePathHash,
                'file_path' => $filePath,
                'hash' => $hash,
                'first_seen' => $now,
                'last_verified' => $now
            ]);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store integrity hash in database', [
                'error' => $e->getMessage(),
                'file_path' => $filePath
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getIntegrityHash(string $filePath): ?string
    {
        try {
            $filePathHash = hash('sha256', $filePath);
            
            $hash = $this->dbConnection->querySingleValue(
                "SELECT `hash` FROM `{$this->integrityTable}` WHERE `file_path_hash` = :file_path_hash",
                ['file_path_hash' => $filePathHash]
            );
            
            return $hash;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get integrity hash from database', [
                'error' => $e->getMessage(),
                'file_path' => $filePath
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeRateLimit(string $key, array $data, int $ttl): bool
    {
        try {
            $keyHash = hash('sha256', $key);
            $now = time();
            $expires = $now + $ttl;
            
            $sql = "
                REPLACE INTO `{$this->rateLimitsTable}`
                (`key_hash`, `key_name`, `data`, `created_at`, `expires`)
                VALUES
                (:key_hash, :key_name, :data, :created_at, :expires)
            ";
            
            $this->dbConnection->execute($sql, [
                'key_hash' => $keyHash,
                'key_name' => $key,
                'data' => json_encode($data, JSON_UNESCAPED_UNICODE),
                'created_at' => $now,
                'expires' => $expires
            ]);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store rate limit in database', [
                'error' => $e->getMessage(),
                'key' => substr($key, 0, 16) . '...'
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getRateLimit(string $key): ?array
    {
        try {
            $keyHash = hash('sha256', $key);
            $now = time();
            
            $data = $this->dbConnection->querySingleValue(
                "SELECT `data` FROM `{$this->rateLimitsTable}` WHERE `key_hash` = :key_hash AND `expires` > :now",
                [
                    'key_hash' => $keyHash,
                    'now' => $now
                ]
            );
            
            if ($data === null) {
                return null;
            }
            
            return json_decode($data, true) ?? null;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get rate limit from database', [
                'error' => $e->getMessage(),
                'key' => substr($key, 0, 16) . '...'
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function incrementCounter(string $key, int $ttl): int
    {
        try {
            $keyHash = hash('sha256', $key);
            $now = time();
            $expires = $now + $ttl;
            
            // This implementation uses a transaction to make counter increment atomic
            $this->dbConnection->beginTransaction();
            
            // Get current value
            $dataJson = $this->dbConnection->querySingleValue(
                "SELECT `data` FROM `{$this->rateLimitsTable}` WHERE `key_hash` = :key_hash FOR UPDATE",
                ['key_hash' => $keyHash]
            );
            
            $count = 1;
            $data = [];
            
            if ($dataJson !== null) {
                $data = json_decode($dataJson, true) ?? [];
                $count = ($data['count'] ?? 0) + 1;
                $data['count'] = $count;
                $data['updated_at'] = $now;
            } else {
                $data = [
                    'count' => $count,
                    'created_at' => $now,
                    'updated_at' => $now
                ];
            }
            
            // Update or insert the counter
            $sql = "
                REPLACE INTO `{$this->rateLimitsTable}`
                (`key_hash`, `key_name`, `data`, `created_at`, `expires`)
                VALUES
                (:key_hash, :key_name, :data, :created_at, :expires)
            ";
            
            $this->dbConnection->execute($sql, [
                'key_hash' => $keyHash,
                'key_name' => $key,
                'data' => json_encode($data, JSON_UNESCAPED_UNICODE),
                'created_at' => $now,
                'expires' => $expires
            ]);
            
            $this->dbConnection->commit();
            
            return $count;
        } catch (\Throwable $e) {
            if ($this->dbConnection->inTransaction()) {
                $this->dbConnection->rollBack();
            }
            
            $this->logger->error('Failed to increment counter in database', [
                'error' => $e->getMessage(),
                'key' => substr($key, 0, 16) . '...'
            ]);
            
            // Return 1 as fallback
            return 1;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeAnomalyData(string $metric, float $value, array $metadata = []): bool
    {
        try {
            // Store data point
            $sql = "
                INSERT INTO `{$this->anomalyTable}`
                (`metric`, `value`, `timestamp`, `ip`, `path`, `is_anomaly`, `z_score`, `metadata`)
                VALUES
                (:metric, :value, :timestamp, :ip, :path, :is_anomaly, :z_score, :metadata)
            ";
            
            $this->dbConnection->execute($sql, [
                'metric' => $metric,
                'value' => $value,
                'timestamp' => date('Y-m-d H:i:s'),
                'ip' => $metadata['ip'] ?? null,
                'path' => $metadata['path'] ?? null,
                'is_anomaly' => $metadata['is_anomaly'] ?? 0,
                'z_score' => $metadata['z_score'] ?? null,
                'metadata' => json_encode($metadata, JSON_UNESCAPED_UNICODE)
            ]);
            
            // Update metrics statistics
            $this->updateMetricStatistics($metric, $value);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store anomaly data in database', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getAnomalyData(string $metric, int $limit = 100): array
    {
        try {
            $sql = "
                SELECT * FROM `{$this->anomalyTable}` 
                WHERE `metric` = :metric
                ORDER BY `timestamp` DESC
                LIMIT :limit
            ";
            
            $rows = $this->dbConnection->queryAndFetchAllAssoc($sql, [
                'metric' => $metric, 
                'limit' => $limit
            ]);
            
            $results = [];
            foreach ($rows as $row) {
                // Parse metadata JSON
                $metadata = json_decode($row['metadata'], true) ?? [];
                
                $results[] = [
                    'timestamp' => $row['timestamp'],
                    'value' => (float)$row['value'],
                    'is_anomaly' => (bool)$row['is_anomaly'],
                    'z_score' => $row['z_score'] !== null ? (float)$row['z_score'] : null,
                    'metadata' => $metadata
                ];
            }
            
            return $results;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get anomaly data from database', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
            return [];
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getMetricStatistics(string $metric): array
    {
        try {
            $row = $this->dbConnection->queryAndFetchAssoc(
                "SELECT * FROM `{$this->metricsTable}` WHERE `metric` = :metric",
                ['metric' => $metric]
            );
            
            if (!$row) {
                return [
                    'count' => 0,
                    'min' => 0,
                    'max' => 0,
                    'mean' => 0,
                    'stddev' => 0
                ];
            }
            
            return [
                'count' => (int)$row['count'],
                'min' => (float)$row['min'],
                'max' => (float)$row['max'],
                'sum' => (float)$row['sum'],
                'sum_squares' => (float)$row['sum_squares'],
                'mean' => (float)$row['mean'],
                'variance' => (float)$row['variance'],
                'stddev' => (float)$row['stddev'],
                'last_updated' => (int)$row['last_updated']
            ];
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get metric statistics from database', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
            
            return [
                'count' => 0,
                'min' => 0,
                'max' => 0,
                'mean' => 0,
                'stddev' => 0
            ];
        }
    }
    
    /**
     * Update statistical values for a metric
     */
    private function updateMetricStatistics(string $metric, float $value): void
    {
        try {
            $now = time();
            
            // Koristimo transakcije za atomarno ažuriranje statistike
            $this->dbConnection->beginTransaction();
            
            // Check if metric exists
            $row = $this->dbConnection->queryAndFetchAssoc(
                "SELECT * FROM `{$this->metricsTable}` WHERE `metric` = :metric FOR UPDATE",
                ['metric' => $metric]
            );
            
            if (!$row) {
                // Initialize metrics if this is the first data point
                $sql = "
                    INSERT INTO `{$this->metricsTable}`
                    (`metric`, `count`, `min`, `max`, `sum`, `sum_squares`, `mean`, `variance`, `stddev`, `last_updated`)
                    VALUES
                    (:metric, 1, :value, :value, :value, :sum_squares, :value, 0, 0, :now)
                ";
                
                $this->dbConnection->execute($sql, [
                    'metric' => $metric,
                    'value' => $value,
                    'sum_squares' => $value * $value,
                    'now' => $now
                ]);
            } else {
                // Update existing metrics
                $count = (int)$row['count'] + 1;
                $min = min((float)$row['min'], $value);
                $max = max((float)$row['max'], $value);
                $sum = (float)$row['sum'] + $value;
                $sumSquares = (float)$row['sum_squares'] + ($value * $value);
                $mean = $sum / $count;
                
                // Calculate variance and standard deviation
                $variance = 0;
                $stddev = 0;
                
                if ($count > 1) {
                    $variance = ($sumSquares / $count) - ($mean * $mean);
                    $stddev = sqrt(max(0, $variance)); // Ensure non-negative value
                }
                
                // Update metrics
                $sql = "
                    UPDATE `{$this->metricsTable}`
                    SET `count` = :count,
                        `min` = :min,
                        `max` = :max,
                        `sum` = :sum,
                        `sum_squares` = :sum_squares,
                        `mean` = :mean,
                        `variance` = :variance,
                        `stddev` = :stddev,
                        `last_updated` = :now
                    WHERE `metric` = :metric
                ";
                
                $this->dbConnection->execute($sql, [
                    'count' => $count,
                    'min' => $min,
                    'max' => $max,
                    'sum' => $sum,
                    'sum_squares' => $sumSquares,
                    'mean' => $mean,
                    'variance' => $variance,
                    'stddev' => $stddev,
                    'now' => $now,
                    'metric' => $metric
                ]);
            }
            
            $this->dbConnection->commit();
        } catch (\Throwable $e) {
            if ($this->dbConnection->inTransaction()) {
                $this->dbConnection->rollBack();
            }
            
            $this->logger->error('Failed to update metric statistics in database', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
        }
    }
    
    /**
     * Generate a UUID v4
     * 
     * @return string UUID
     */
    private function generateUuid(): string
    {
        $data = random_bytes(16);
        
        // Set version to 0100
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
        // Set bits 6-7 to 10
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
        
        // Output the 36 character UUID
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }
    
    /**
     * Cleanup expired data
     * 
     * @return array Cleanup statistics
     */
    public function cleanup(): array
    {
        $stats = [
            'fingerprints_deleted' => 0,
            'rate_limits_deleted' => 0,
            'old_profiles_deleted' => 0,
            'expired_blocks_deleted' => 0
        ];
        
        try {
            $now = time();
            
            // Clean expired fingerprints
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->fingerprintsTable}` WHERE `expires` < :now",
                ['now' => $now]
            );
            $stats['fingerprints_deleted'] = $affectedRows;
            
            // Clean expired rate limits
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->rateLimitsTable}` WHERE `expires` < :now",
                ['now' => $now]
            );
            $stats['rate_limits_deleted'] = $affectedRows;
            
            // Clean old profiles (older than 30 days)
            $oldProfileTime = $now - (30 * 86400);
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->profilesTable}` WHERE `last_seen` < :time",
                ['time' => $oldProfileTime]
            );
            $stats['old_profiles_deleted'] = $affectedRows;
            
            // Clean expired IP blocks
            $affectedRows = $this->dbConnection->execute(
                "DELETE FROM `{$this->blockedIpsTable}` WHERE `expires` < :now",
                ['now' => $now]
            );
            $stats['expired_blocks_deleted'] = $affectedRows;
            
            if (array_sum($stats) > 0) {
                $this->logger->borna('BORNA database cleanup completed', $stats);
            }
            
            return $stats;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to perform BORNA database cleanup', [
                'error' => $e->getMessage()
            ]);
            
            return array_merge($stats, ['error' => $e->getMessage()]);
        }
    }
}