<?php

namespace baseKRIZAN\BORNA\Storage;

use baseKRIZAN\Error\Logger;
use Redis;

/**
 * Redis-based storage implementation for BORNA security data
 * Provides high-performance, distributed storage for security data
 */
class RedisStorage implements StorageInterface
{
    /**
     * Logger instance
     */
    private Logger $logger;
    
    /**
     * Redis client instance
     */
    private Redis $redis;
    
    /**
     * Key prefix for all Redis keys
     */
    private string $keyPrefix = 'borna:';
    
    /**
     * Constructor
     */
    public function __construct(
        Logger $logger,
        string $host = '127.0.0.1',
        int $port = 6379,
        ?string $password = null,
        int $database = 0
    ) {
        $this->logger = $logger;
        
        $this->initializeRedis($host, $port, $password, $database);
    }
    
    /**
     * Initialize Redis connection
     */
    private function initializeRedis(string $host, int $port, ?string $password, int $database): void
    {
        try {
            $this->redis = new Redis();
            $connected = $this->redis->connect($host, $port);
            
            if (!$connected) {
                throw new \RuntimeException("Failed to connect to Redis at {$host}:{$port}");
            }
            
            // Authenticate if password is provided
            if (!empty($password)) {
                $authenticated = $this->redis->auth($password);
                if (!$authenticated) {
                    throw new \RuntimeException("Failed to authenticate with Redis");
                }
            }
            
            // Select database
            $this->redis->select($database);
            
            $this->logger->borna('Connected to Redis for BORNA security storage', [
                'host' => $host,
                'port' => $port,
                'database' => $database
            ]);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to connect to Redis', [
                'error' => $e->getMessage(),
                'host' => $host,
                'port' => $port
            ]);
            
            // Fallback to dummy Redis client that logs errors
            $this->redis = new class($this->logger) extends Redis {
                private $logger;
                
                public function __construct($logger) {
                    $this->logger = $logger;
                }
                
                public function __call($name, $arguments) {
                    $this->logger->error("Redis operation failed: {$name}", [
                        'arguments' => $arguments,
                        'error' => 'Redis connection not available'
                    ]);
                    return null;
                }
            };
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getBlockedIPs(): array
    {
        try {
            $blockedIPs = $this->redis->get($this->keyPrefix . 'blocked_ips');
            
            if (!$blockedIPs) {
                return [];
            }
            
            $data = json_decode($blockedIPs, true);
            return is_array($data) ? $data : [];
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get blocked IPs from Redis', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
    
    /**
     * @inheritDoc
     */
    public function saveBlockedIPs(array $blockedIPs): bool
    {
        try {
            $result = $this->redis->set(
                $this->keyPrefix . 'blocked_ips',
                json_encode($blockedIPs)
            );
            
            return (bool)$result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to save blocked IPs to Redis', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeEvent(string $eventType, array $eventData): bool
    {
        try {
            $event = [
                'id' => uniqid('sec_', true),
                'event_type' => $eventType,
                'timestamp' => date('Y-m-d H:i:s'),
                'data' => $eventData
            ];
            
            // Store event in Redis list
            $key = $this->keyPrefix . 'events:' . $eventType;
            
            // Add to specific event type list
            $this->redis->lPush($key, json_encode($event));
            
            // Add to all events list
            $this->redis->lPush($this->keyPrefix . 'events:all', json_encode($event));
            
            // Trim lists to prevent unbounded growth
            $this->redis->lTrim($key, 0, 9999);
            $this->redis->lTrim($this->keyPrefix . 'events:all', 0, 9999);
            
            // Set expiration for event lists (90 days)
            $this->redis->expire($key, 60 * 60 * 24 * 90);
            
            return true;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store security event in Redis', [
                'error' => $e->getMessage(),
                'event_type' => $eventType
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getEvents(int $fromTimestamp, int $toTimestamp, ?string $eventType = null): array
    {
        try {
            $fromDate = date('Y-m-d H:i:s', $fromTimestamp);
            $toDate = date('Y-m-d H:i:s', $toTimestamp);
            
            // Determine which list to query
            $key = $this->keyPrefix . 'events:' . ($eventType ?? 'all');
            
            // Get all events from list
            $count = $this->redis->lLen($key);
            $events = [];
            
            if ($count > 0) {
                $allEvents = $this->redis->lRange($key, 0, $count - 1);
                
                foreach ($allEvents as $eventJson) {
                    $event = json_decode($eventJson, true);
                    
                    // Filter by date range
                    if ($event && $event['timestamp'] >= $fromDate && $event['timestamp'] <= $toDate) {
                        // Filter by event type if querying 'all' events
                        if ($eventType === null || $event['event_type'] === $eventType) {
                            $events[] = $event;
                        }
                    }
                }
            }
            
            // Sort by timestamp descending
            usort($events, function($a, $b) {
                return strcmp($b['timestamp'], $a['timestamp']);
            });
            
            return $events;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get security events from Redis', [
                'error' => $e->getMessage(),
                'event_type' => $eventType ?? 'all'
            ]);
            return [];
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeClientFingerprint(string $fingerprint, array $data): bool
    {
        try {
            $key = $this->keyPrefix . 'fingerprint:' . $fingerprint;
            
            // Store with expiration (24 hours by default)
            $ttl = $data['expires'] ?? (time() + 86400);
            $expiresIn = max(1, $ttl - time());
            
            $result = $this->redis->setEx(
                $key,
                $expiresIn,
                json_encode($data)
            );
            
            return (bool)$result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store client fingerprint in Redis', [
                'error' => $e->getMessage(),
                'fingerprint' => substr($fingerprint, 0, 16) . '...'
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getClientFingerprint(string $fingerprint): ?array
    {
        try {
            $key = $this->keyPrefix . 'fingerprint:' . $fingerprint;
            
            $data = $this->redis->get($key);
            
            if (!$data) {
                return null;
            }
            
            return json_decode($data, true);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get client fingerprint from Redis', [
                'error' => $e->getMessage(),
                'fingerprint' => substr($fingerprint, 0, 16) . '...'
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeBehaviorProfile(string $clientId, array $profile): bool
    {
        try {
            $hashedId = hash('sha256', $clientId);
            $key = $this->keyPrefix . 'profile:' . $hashedId;
            
            // Store with expiration (30 days)
            $expiresIn = 60 * 60 * 24 * 30;
            
            $result = $this->redis->setEx(
                $key,
                $expiresIn,
                json_encode($profile)
            );
            
            return (bool)$result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store behavior profile in Redis', [
                'error' => $e->getMessage(),
                'client_id' => substr($clientId, 0, 16) . '...'
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getBehaviorProfile(string $clientId): ?array
    {
        try {
            $hashedId = hash('sha256', $clientId);
            $key = $this->keyPrefix . 'profile:' . $hashedId;
            
            $data = $this->redis->get($key);
            
            if (!$data) {
                return null;
            }
            
            return json_decode($data, true);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get behavior profile from Redis', [
                'error' => $e->getMessage(),
                'client_id' => substr($clientId, 0, 16) . '...'
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeIntegrityHash(string $filePath, string $hash): bool
    {
        try {
            $normalizedPath = hash('sha256', $filePath);
            $key = $this->keyPrefix . 'integrity:' . $normalizedPath;
            
            // Store without expiration
            $result = $this->redis->set($key, $hash);
            
            return (bool)$result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store integrity hash in Redis', [
                'error' => $e->getMessage(),
                'file_path' => $filePath
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getIntegrityHash(string $filePath): ?string
    {
        try {
            $normalizedPath = hash('sha256', $filePath);
            $key = $this->keyPrefix . 'integrity:' . $normalizedPath;
            
            $hash = $this->redis->get($key);
            
            return $hash !== false ? $hash : null;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get integrity hash from Redis', [
                'error' => $e->getMessage(),
                'file_path' => $filePath
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeRateLimit(string $key, array $data, int $ttl): bool
    {
        try {
            $redisKey = $this->keyPrefix . 'rate:' . hash('sha256', $key);
            
            // Store with expiration
            $result = $this->redis->setEx(
                $redisKey,
                $ttl,
                json_encode($data)
            );
            
            return (bool)$result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store rate limit in Redis', [
                'error' => $e->getMessage(),
                'key' => substr($key, 0, 16) . '...'
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getRateLimit(string $key): ?array
    {
        try {
            $redisKey = $this->keyPrefix . 'rate:' . hash('sha256', $key);
            
            $data = $this->redis->get($redisKey);
            
            if (!$data) {
                return null;
            }
            
            return json_decode($data, true);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get rate limit from Redis', [
                'error' => $e->getMessage(),
                'key' => substr($key, 0, 16) . '...'
            ]);
            return null;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function incrementCounter(string $key, int $ttl): int
    {
        try {
            $redisKey = $this->keyPrefix . 'counter:' . hash('sha256', $key);
            
            // Increment counter and set expiration
            $count = $this->redis->incr($redisKey);
            $this->redis->expire($redisKey, $ttl);
            
            return (int)$count;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to increment counter in Redis', [
                'error' => $e->getMessage(),
                'key' => substr($key, 0, 16) . '...'
            ]);
            return 1; // Return 1 as fallback
        }
    }
    
    /**
     * @inheritDoc
     */
    public function storeAnomalyData(string $metric, float $value, array $metadata = []): bool
    {
        try {
            // Use sorted set for time series data
            $key = $this->keyPrefix . 'anomaly:' . $metric;
            
            // Use current timestamp as score for sorted set
            $timestamp = microtime(true);
            
            // Store data point
            $dataPoint = [
                'timestamp' => date('Y-m-d H:i:s', (int)$timestamp),
                'value' => $value,
                'metadata' => $metadata
            ];
            
            $result = $this->redis->zAdd(
                $key,
                $timestamp,
                json_encode($dataPoint)
            );
            
            // Trim set to last 5000 points
            $this->redis->zRemRangeByRank($key, 0, -5001);
            
            // Set expiration (90 days)
            $this->redis->expire($key, 60 * 60 * 24 * 90);
            
            // Update statistics
            $this->updateMetricStatistics($metric, $value);
            
            return (bool)$result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to store anomaly data in Redis', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
            return false;
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getAnomalyData(string $metric, int $limit = 100): array
    {
        try {
            $key = $this->keyPrefix . 'anomaly:' . $metric;
            
            // Get latest data points, ordered by score (timestamp) descending
            $dataPoints = $this->redis->zRevRange($key, 0, $limit - 1);
            
            if (!$dataPoints) {
                return [];
            }
            
            $result = [];
            foreach ($dataPoints as $dataPoint) {
                $data = json_decode($dataPoint, true);
                if ($data) {
                    $result[] = $data;
                }
            }
            
            return $result;
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get anomaly data from Redis', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
            return [];
        }
    }
    
    /**
     * Update statistical values for a metric
     */
    private function updateMetricStatistics(string $metric, float $value): void
    {
        try {
            $statsKey = $this->keyPrefix . 'anomaly:stats:' . $metric;
            
            // Get current stats
            $stats = $this->redis->hGetAll($statsKey);
            
            if (empty($stats)) {
                // Initialize stats
                $this->redis->hMSet($statsKey, [
                    'count' => 1,
                    'min' => $value,
                    'max' => $value,
                    'sum' => $value,
                    'sum_squares' => $value * $value,
                    'mean' => $value,
                    'variance' => 0,
                    'stddev' => 0,
                    'last_updated' => time()
                ]);
            } else {
                // Convert string values to numeric
                $count = (int)($stats['count'] ?? 0) + 1;
                $min = min((float)($stats['min'] ?? $value), $value);
                $max = max((float)($stats['max'] ?? $value), $value);
                $sum = (float)($stats['sum'] ?? 0) + $value;
                $sumSquares = (float)($stats['sum_squares'] ?? 0) + ($value * $value);
                
                // Recalculate mean
                $mean = $sum / $count;
                
                // Recalculate variance and standard deviation
                $variance = 0;
                $stddev = 0;
                
                if ($count > 1) {
                    $variance = ($sumSquares / $count) - ($mean * $mean);
                    $stddev = sqrt($variance);
                }
                
                // Update stats
                $this->redis->hMSet($statsKey, [
                    'count' => $count,
                    'min' => $min,
                    'max' => $max,
                    'sum' => $sum,
                    'sum_squares' => $sumSquares,
                    'mean' => $mean,
                    'variance' => $variance,
                    'stddev' => $stddev,
                    'last_updated' => time()
                ]);
            }
            
            // Set expiration (90 days)
            $this->redis->expire($statsKey, 60 * 60 * 24 * 90);
        } catch (\Throwable $e) {
            $this->logger->error('Failed to update metric statistics in Redis', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
        }
    }
    
    /**
     * @inheritDoc
     */
    public function getMetricStatistics(string $metric): array
    {
        try {
            $statsKey = $this->keyPrefix . 'anomaly:stats:' . $metric;
            
            $stats = $this->redis->hGetAll($statsKey);
            
            if (empty($stats)) {
                return [
                    'count' => 0,
                    'min' => 0,
                    'max' => 0,
                    'mean' => 0,
                    'stddev' => 0
                ];
            }
            
            // Convert string values to appropriate types
            return [
                'count' => (int)($stats['count'] ?? 0),
                'min' => (float)($stats['min'] ?? 0),
                'max' => (float)($stats['max'] ?? 0),
                'sum' => (float)($stats['sum'] ?? 0),
                'sum_squares' => (float)($stats['sum_squares'] ?? 0),
                'mean' => (float)($stats['mean'] ?? 0),
                'variance' => (float)($stats['variance'] ?? 0),
                'stddev' => (float)($stats['stddev'] ?? 0),
                'last_updated' => (int)($stats['last_updated'] ?? 0)
            ];
        } catch (\Throwable $e) {
            $this->logger->error('Failed to get metric statistics from Redis', [
                'error' => $e->getMessage(),
                'metric' => $metric
            ]);
            
            return [
                'count' => 0,
                'min' => 0,
                'max' => 0,
                'mean' => 0,
                'stddev' => 0
            ];
        }
    }
}