<?php

namespace baseKRIZAN\BORNA\Storage;

/**
 * Storage interface for BORNA security data
 */
interface StorageInterface
{
    /**
     * Get all blocked IPs
     * 
     * @return array Blocked IPs with their metadata
     */
    public function getBlockedIPs(): array;
    
    /**
     * Save blocked IPs
     * 
     * @param array $blockedIPs Blocked IPs with their metadata
     * @return bool Success state
     */
    public function saveBlockedIPs(array $blockedIPs): bool;
    
    /**
     * Store a security event
     * 
     * @param string $eventType Event type
     * @param array $eventData Event data
     * @return bool Success state
     */
    public function storeEvent(string $eventType, array $eventData): bool;
    
    /**
     * Get security events for a time period
     * 
     * @param int $fromTimestamp Start timestamp
     * @param int $toTimestamp End timestamp
     * @param string|null $eventType Optional event type filter
     * @return array Security events
     */
    public function getEvents(int $fromTimestamp, int $toTimestamp, ?string $eventType = null): array;
    
    /**
     * Store a client fingerprint for verification
     * 
     * @param string $fingerprint Client fingerprint hash
     * @param array $data Associated client data
     * @return bool Success state
     */
    public function storeClientFingerprint(string $fingerprint, array $data): bool;
    
    /**
     * Get client fingerprint data
     * 
     * @param string $fingerprint Client fingerprint hash
     * @return array|null Client data or null if not found
     */
    public function getClientFingerprint(string $fingerprint): ?array;
    
    /**
     * Store behavior profile for a client
     * 
     * @param string $clientId Client identifier (IP, user ID, etc.)
     * @param array $profile Behavior profile data
     * @return bool Success state
     */
    public function storeBehaviorProfile(string $clientId, array $profile): bool;
    
    /**
     * Get behavior profile for a client
     * 
     * @param string $clientId Client identifier
     * @return array|null Behavior profile data or null if not found
     */
    public function getBehaviorProfile(string $clientId): ?array;
    
    /**
     * Store integrity hash for a file
     * 
     * @param string $filePath File path
     * @param string $hash File hash
     * @return bool Success state
     */
    public function storeIntegrityHash(string $filePath, string $hash): bool;
    
    /**
     * Get integrity hash for a file
     * 
     * @param string $filePath File path
     * @return string|null Hash or null if not found
     */
    public function getIntegrityHash(string $filePath): ?string;
    
    /**
     * Store rate limit data
     * 
     * @param string $key Rate limit key
     * @param array $data Rate limit data
     * @param int $ttl Time to live in seconds
     * @return bool Success state
     */
    public function storeRateLimit(string $key, array $data, int $ttl): bool;
    
    /**
     * Get rate limit data
     * 
     * @param string $key Rate limit key
     * @return array|null Rate limit data or null if not found or expired
     */
    public function getRateLimit(string $key): ?array;
    
    /**
     * Increment a counter and get the new value atomically
     * 
     * @param string $key Counter key
     * @param int $ttl Time to live in seconds
     * @return int New counter value
     */
    public function incrementCounter(string $key, int $ttl): int;
    
    /**
     * Store anomaly data
     * 
     * @param string $metric Metric name
     * @param float $value Metric value
     * @param array $metadata Additional metadata
     * @return bool Success state
     */
    public function storeAnomalyData(string $metric, float $value, array $metadata = []): bool;
    
    /**
     * Get anomaly data for a metric
     * 
     * @param string $metric Metric name
     * @param int $limit Maximum number of data points to retrieve
     * @return array Data points
     */
    public function getAnomalyData(string $metric, int $limit = 100): array;
    
    /**
     * Get statistical aggregates for a metric
     * 
     * @param string $metric Metric name
     * @return array Statistical values (min, max, avg, stddev, etc.)
     */
    public function getMetricStatistics(string $metric): array;
}