<?php

namespace baseKRIZAN\BORNA;

use baseKRIZAN\Error\Logger;

/**
 * Threat detection engine for BORNA security module
 * Analyzes requests and identifies potential security threats
 */
class ThreatDetector
{
    private Logger $logger;
    private array $config;
    private array $attackSignatures = [];
    private array $suspiciousPatterns = [];
    private array $rateLimit = [];
    
    /**
     * Constructor
     */
    public function __construct(Logger $logger, array $config)
    {
        $this->logger = $logger;
        $this->config = $config;
        
        // Load attack signatures and suspicious patterns
        $this->loadSignatures();
    }
    
    /**
     * Load attack signatures from definitions
     */
    private function loadSignatures(): void
    {
        // XSS attack patterns
        $this->attackSignatures['xss'] = [
            // Basic script tag injection attempts
            '/(<script[^>]*>.*?<\/script>)/i',
            '/(javascript\s*:)/i',
            '/(\bon\w+\s*=)/i',
            '/(document\.cookie)/i',
            '/(document\.location)/i',
            '/(\.[a-zA-Z]+\.innerHTML)/i',
            '/(eval\s*\()/i',
            '/(String\.fromCharCode)/i',
            
            // Encoded JavaScript
            '/(&#x[0-9a-f]+;)/i',
            '/(&#[0-9]+;)/i',
            
            // Event handlers
            '/(\bon[a-z]+\s*=)/i',
            
            // CSS expressions (for older IE)
            '/(expression\s*\()/i',
            
            // VBScript
            '/(vbscript\s*:)/i',
            
            // Breaking out of attributes
            '/(["\']\s*[^"\']*\s*[<>]\s*[^"\']*\s*["\']\s*)/i',
            
            // Data URI with script
            '/(data\s*:\s*text\/html)/i'
        ];
        
        // SQL Injection attack patterns
        $this->attackSignatures['sqli'] = [
            // Common SQL commands
            '/(UNION\s+SELECT)/i',
            '/(SELECT\s+.*\s+FROM)/i',
            '/(INSERT\s+INTO)/i',
            '/(DELETE\s+FROM)/i',
            '/(DROP\s+TABLE)/i',
            '/(ALTER\s+TABLE)/i',
            '/(EXEC\s+xp_cmdshell)/i',
            
            // SQL comments
            '/(--|#|\/\*|\*\/)/i',
            
            // Common SQL functions
            '/(CHAR\(|CONCAT\(|ASCII\()/i',
            
            // SQL authentication bypass
            "/(' OR 1=1)/i",
            "/(\" OR 1=1)/i",
            "/(' OR '1'='1)/i",
            '/(" OR "1"="1)/i',
            
            // Stack query operators
            '/(;.*;)/i',
            
            // Hex encoding
            '/(0x[0-9a-f]+)/i'
        ];
        
        // Common file inclusion attack patterns
        $this->attackSignatures['rfi_lfi'] = [
            // Remote file inclusion
            '/(https?:\/\/\S+\?)/i',
            '/(ftp:\/\/\S+\?)/i',
            
            // Local file inclusion
            '/(\.\.\/)/i',
            '/(\/etc\/passwd)/i',
            '/(\/etc\/shadow)/i',
            '/(\/proc\/self)/i',
            '/(\/var\/log)/i',
            '/(\/boot.ini)/i',
            '/(php:\/\/)/i'
        ];
        
        // CSRF attack patterns
        $this->attackSignatures['csrf'] = [
            // Looking for hidden form elements with malicious intents
            '/(<input[^>]*type\s*=\s*[\'"]hidden[\'"][^>]*value\s*=\s*[\'"][^"\']*(?:http|www|ftp)[^"\']*[\'"][^>]*>)/i'
        ];
        
        // Command injection patterns
        $this->attackSignatures['command_injection'] = [
            // Basic command operators
            '/([;&|`])/i',
            
            // Common shell commands
            '/(ping\s+-c|wget\s+|curl\s+|nc\s+|netcat\s+|bash\s+|python\s+|perl\s+)/i',
            
            // Process manipulation
            '/(\/bin\/|\/usr\/bin)/i',
            
            // Character encoding bypass attempts
            '/(\\\\x[0-9a-f]{2})/i'
        ];
        
        // Path traversal patterns
        $this->attackSignatures['path_traversal'] = [
            '/(\.\.\/|\.\.\\\\)/i',
            '/(\/\.\.\/|\\\\\.\.\\\\)/i',
            '/(\.\.%2f|\.\.%5c)/i',
            '/(\.\.%252f|\.\.%255c)/i'
        ];
        
        // Suspicious patterns (lower severity)
        $this->suspiciousPatterns = [
            // Potentially malicious parameter names
            '/^(cmd|exec|command|shell|passthru|system)$/',
            
            // Base64 encoded payloads (could be legitimate but worth checking)
            '/^[a-zA-Z0-9+\/]{20,}={0,2}$/',
            
            // Excessive data length
            '/^.{1000,}$/',
            
            // Common scanning tools user agents
            '/(nmap|nikto|sqlmap|burpsuite|nessus|acunetix|metasploit)/i',
            
            // Suspicious file extensions in URLs
            '/(\.php\d+|\.phtml|\.php~|\.php-|\.php_)/i',
            
            // Uncommon HTTP methods
            '/(TRACE|TRACK|DEBUG|PUT|DELETE)/i'
        ];
    }
    
    /**
     * Check request against known attack signatures
     * 
     * @param array $requestData Request data to check
     * @return array Detected threats with scores
     */
    public function detectThreats(array $requestData): array
    {
        $threats = [];
        
        // Check GET parameters
        if (isset($requestData['query_params']) && is_array($requestData['query_params'])) {
            foreach ($requestData['query_params'] as $param => $value) {
                $result = $this->checkParameter($param, $value);
                if (!empty($result)) {
                    $threats[] = [
                        'type' => 'query_param',
                        'param' => $param,
                        'patterns' => $result
                    ];
                }
            }
        }
        
        // Check POST parameters
        if (isset($requestData['post_params']) && is_array($requestData['post_params'])) {
            foreach ($requestData['post_params'] as $param => $value) {
                $result = $this->checkParameter($param, $value);
                if (!empty($result)) {
                    $threats[] = [
                        'type' => 'post_param',
                        'param' => $param,
                        'patterns' => $result
                    ];
                }
            }
        }
        
        // Check cookies
        if (isset($requestData['cookies']) && is_array($requestData['cookies'])) {
            foreach ($requestData['cookies'] as $name => $value) {
                $result = $this->checkParameter($name, $value);
                if (!empty($result)) {
                    $threats[] = [
                        'type' => 'cookie',
                        'name' => $name,
                        'patterns' => $result
                    ];
                }
            }
        }
        
        // Check headers
        if (isset($requestData['headers']) && is_array($requestData['headers'])) {
            foreach ($requestData['headers'] as $header => $value) {
                $result = $this->checkParameter($header, $value);
                if (!empty($result)) {
                    $threats[] = [
                        'type' => 'header',
                        'header' => $header,
                        'patterns' => $result
                    ];
                }
            }
        }
        
        // Check URL path for path traversal and other attacks
        if (isset($requestData['path'])) {
            $result = $this->checkParameter('path', $requestData['path']);
            if (!empty($result)) {
                $threats[] = [
                    'type' => 'path',
                    'patterns' => $result
                ];
            }
        }
        
        // Check request method - higher priority for uncommon methods
        if (isset($requestData['method']) && !in_array($requestData['method'], ['GET', 'POST'])) {
            $threats[] = [
                'type' => 'method',
                'method' => $requestData['method'],
                'patterns' => ['uncommon_method']
            ];
        }
        
        // Rate limiting
        if (isset($requestData['ip'])) {
            $ip = $requestData['ip'];
            $now = time();
            
            // Initialize rate limiting for this IP
            if (!isset($this->rateLimit[$ip])) {
                $this->rateLimit[$ip] = [
                    'count' => 0,
                    'first_request' => $now,
                    'last_request' => $now
                ];
            }
            
            // Update rate limit data
            $this->rateLimit[$ip]['count']++;
            $this->rateLimit[$ip]['last_request'] = $now;
            
            // Check if rate limit is exceeded
            $elapsed = $now - $this->rateLimit[$ip]['first_request'];
            $rateLimit = $this->config['rate_limit_requests'] ?? 60;
            $ratePeriod = $this->config['rate_limit_period'] ?? 60;
            
            if ($elapsed <= $ratePeriod && $this->rateLimit[$ip]['count'] > $rateLimit) {
                $threats[] = [
                    'type' => 'rate_limit',
                    'ip' => $ip,
                    'count' => $this->rateLimit[$ip]['count'],
                    'period' => $elapsed,
                    'limit' => $rateLimit
                ];
            }
            
            // Reset rate limiting if period has passed
            if ($elapsed > $ratePeriod) {
                $this->rateLimit[$ip] = [
                    'count' => 1,
                    'first_request' => $now,
                    'last_request' => $now
                ];
            }
        }
        
        return $threats;
    }
    
    /**
     * Calculate threat score based on detected threats
     * 
     * @param array $threats Detected threats
     * @return int Threat score (0-100)
     */
    public function calculateThreatScore(array $threats): int
    {
        if (empty($threats)) {
            return 0;
        }
        
        $score = 0;
        $weights = [
            'xss' => 25,
            'sqli' => 30,
            'rfi_lfi' => 35,
            'csrf' => 20,
            'command_injection' => 40,
            'path_traversal' => 30,
            'rate_limit' => 15,
            'suspicious' => 10,
            'method' => 15
        ];
        
        // Process each threat
        foreach ($threats as $threat) {
            switch ($threat['type']) {
                case 'query_param':
                case 'post_param':
                case 'cookie':
                case 'header':
                case 'path':
                    foreach ($threat['patterns'] as $category => $matches) {
                        $matchCount = count($matches);
                        if (isset($weights[$category])) {
                            $score += $weights[$category] * min(1, $matchCount * 0.3);
                        }
                    }
                    break;
                    
                case 'rate_limit':
                    // Higher score for more requests over the limit
                    $exceedFactor = ($threat['count'] - $threat['limit']) / $threat['limit'];
                    $score += $weights['rate_limit'] * min(1, $exceedFactor);
                    break;
                
                case 'method':
                    $score += $weights['method'];
                    break;
            }
        }
        
        // Cap score at 100
        return min(100, $score);
    }
    
    /**
     * Check a parameter against attack signatures
     * 
     * @param string $name Parameter name
     * @param string $value Parameter value
     * @return array Matched patterns by category
     */
    private function checkParameter(string $name, $value): array
    {
        if (!is_string($value)) {
            return [];
        }
        
        $matches = [];
        
        // Check parameter name against suspicious patterns
        foreach ($this->suspiciousPatterns as $pattern) {
            if (preg_match($pattern, $name)) {
                $matches['suspicious'][] = [
                    'pattern' => $pattern,
                    'matched' => $name
                ];
            }
        }
        
        // Check parameter value against attack signatures
        foreach ($this->attackSignatures as $category => $patterns) {
            foreach ($patterns as $pattern) {
                if (preg_match($pattern, $value, $patternMatches)) {
                    $matches[$category][] = [
                        'pattern' => $pattern,
                        'matched' => $patternMatches[0]
                    ];
                }
            }
        }
        
        // Check parameter value against suspicious patterns
        foreach ($this->suspiciousPatterns as $pattern) {
            if (preg_match($pattern, $value, $patternMatches)) {
                $matches['suspicious'][] = [
                    'pattern' => $pattern,
                    'matched' => $patternMatches[0]
                ];
            }
        }
        
        return $matches;
    }
    
    /**
     * Analyze a request and return threat analysis
     * 
     * @param array $requestData Request data including path, method, ip, params, etc.
     * @return array Analysis results with threat score and details
     */
    public function analyzeRequest(array $requestData): array
    {
        // Detect threats
        $threats = $this->detectThreats($requestData);
        
        // Calculate threat score
        $score = $this->calculateThreatScore($threats);
        
        // Log high-severity threats
        if ($score > 50) {
            $this->logger->borna('High-severity threats detected', [
                'ip' => $requestData['ip'] ?? 'unknown',
                'path' => $requestData['path'] ?? 'unknown',
                'threat_score' => $score,
                'threats_count' => count($threats)
            ]);
        }
        
        return [
            'score' => $score,
            'threats' => $threats
        ];
    }
}