<?php
// app/baseKRIZAN/Bootstrap/ApplicationFactory.php

namespace baseKRIZAN\Bootstrap;

use baseKRIZAN\Services\Container;
use baseKRIZAN\Bootstrap\Bootstrap;
use baseKRIZAN\Error\UnifiedErrorHandler;
use baseKRIZAN\Http\Request;
use baseKRIZAN\Http\RequestClassifier;

/**
 * Simplified Factory class for creating and running Bootstrap instances.
 */
class ApplicationFactory
{
    /**
     * Shared container instance across requests
     * 
     * @var Container|null
     */
    private static ?Container $sharedContainer = null;
    
    /**
     * Bootstrap configuration options
     * 
     * @var array
     */
    private static array $config = [
        'default_mode' => 'full',
        'shared_container' => true,
        'auto_shutdown' => true,
        'error_log_path' => '/storage/logs/app_errors.log'
    ];
    
    /**
     * Set configuration options for ApplicationFactory
     * 
     * @param array $config New configuration options
     * @return void
     */
    public static function configure(array $config): void
    {
        self::$config = array_merge(self::$config, $config);
    }
    
    /**
     * Creates and initializes a new Bootstrap instance.
     * 
     * @param string|null $bootstrapMode Bootstrap mode ('full', 'minimal', 'api', 'console')
     * @return Bootstrap Initialized application
     */
    public static function create(?string $bootstrapMode = null): Bootstrap
    {
        // Use configuration or passed parameters
        $useSharedContainer = self::$config['shared_container'];
        $bootstrapMode = $bootstrapMode ?? self::$config['default_mode'];
        
        // Get or create container
        $container = $useSharedContainer
            ? self::getSharedContainer()
            : self::createContainer();
        
        // Register bootstrap configuration in container
        $container->register('bootstrap.config', self::$config);
        
        // Get bootstrap instance and initialize it
        $app = Bootstrap::getInstance($container);
        $app->bootstrap($bootstrapMode);
        
        return $app;
    }
    
    /**
     * Gets or creates a shared container
     * 
     * @return Container Container instance
     */
    private static function getSharedContainer(): Container
    {
        if (self::$sharedContainer === null) {
            self::$sharedContainer = self::createContainer();
        }
        
        return self::$sharedContainer;
    }
    
    /**
     * Creates a new container instance
     * 
     * @return Container New container instance
     */
    private static function createContainer(): Container
    {
        $container = new Container();
        $container->register('applicationFactory', static::class);
        return $container;
    }
    
    /**
     * Runs the application with the given path and request method.
     * 
     * @param string $path Request path
     * @param string $method HTTP request method
     * @param string|null $bootstrapMode Bootstrap mode (if null, auto-detected)
     * @return void
     */
    public static function run(
        string $path, 
        string $method, 
        ?string $bootstrapMode = null
    ): void {
        try {
            // Initialize RequestClassifier for request type detection
            $classifier = self::initializeRequestClassifier();
            
            // Auto-detect bootstrap mode if not explicitly given
            if ($bootstrapMode === null) {
                $bootstrapMode = $classifier->determineBootstrapType($path);
            }
            
            // Create application with appropriate mode
            $app = self::create($bootstrapMode);
            
            // Register RequestClassifier in container
            $container = $app->getContainer();
            if (!$container->has('requestClassifier')) {
                $container->register('requestClassifier', $classifier);
            }
            
            // Create and register request object
            $request = new Request();
            $request->setPath($path);
            
            // Set request type and register it in the app
            $requestType = $classifier->getRequestCategory($request);
            $request->setAttribute('request_type', $requestType);
            $app->setCurrentRequest($request);
            
            // Process request
            $app->handleRequest($path, $method);
            
            // Execute shutdown routine if configured
            if (self::$config['auto_shutdown']) {
                $app->shutdown();
            }
            
        } catch (\Throwable $e) {
            self::handleException($e);
        }
    }
    
    /**
     * Initializes and configures RequestClassifier
     * 
     * @return RequestClassifier
     */
    private static function initializeRequestClassifier(): RequestClassifier
    {
        $logger = null;
        
        // Use logger from container if available
        if (self::$sharedContainer && self::$sharedContainer->has('logger')) {
            $logger = self::$sharedContainer->get('logger');
        }
        
        return RequestClassifier::getInstance($logger);
    }
    
    /**
     * Handles application-level errors.
     * 
     * @param \Throwable $e Caught error
     * @return never
     */
    public static function handleException(\Throwable $e): never
    {
        try {
            // Get global error handler
            $handler = UnifiedErrorHandler::getInstance();
            
            // Determine response format based on request
            $responseFormat = $handler->determineResponseFormat();
            
            // Get current request if available
            $request = null;
            if (class_exists('\\baseKRIZAN\\Bootstrap\\Bootstrap')) {
                try {
                    $app = Bootstrap::getInstance();
                    $container = $app->getContainer();
                    
                    if ($container && $container->has('request')) {
                        $request = $container->get('request');
                    }
                    
                    // Ensure logger is available and log the error
                    if ($container && $container->has('logger')) {
                        $logger = $container->get('logger');
                        $logger->error('Application exception caught', [
                            'message' => $e->getMessage(),
                            'file' => $e->getFile(),
                            'line' => $e->getLine(),
                            'trace' => $e->getTraceAsString()
                        ]);
                    }
                } catch (\Throwable $ex) {
                    // Ignore if we can't get the request or log
                }
            }
            
            // Handle the exception through the unified handler
            $errorResponse = $handler->handleException($e, $responseFormat, $request);
            
            if ($errorResponse instanceof \baseKRIZAN\Http\Response) {
                $errorResponse->send();
            } else {
                (new \baseKRIZAN\Http\Response($errorResponse, 500))->send();
            }
        } catch (\Throwable $handlerException) {
            // Fallback error handling if the error handler itself fails
            $errorMessage = "Fatal error: " . $e->getMessage();
            if (PHP_SAPI === 'cli') {
                // CLI response
                echo $errorMessage . PHP_EOL;
                echo "Original exception: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine() . PHP_EOL;
                echo "Error handler exception: " . $handlerException->getMessage() . PHP_EOL;
            } else {
                // HTTP response
                header('HTTP/1.1 500 Internal Server Error');
                echo '<h1>500 Internal Server Error</h1>';
                if (defined('DEVELOPMENT_MODE') && DEVELOPMENT_MODE) {
                    echo '<p>' . htmlspecialchars($errorMessage) . '</p>';
                    echo '<p>Error handler failed: ' . htmlspecialchars($handlerException->getMessage()) . '</p>';
                } else {
                    echo '<p>An internal error occurred. Please try again later.</p>';
                }
            }
        }
        
        exit(1);
    }
    
    /**
     * Resets ApplicationFactory to initial state.
     * Used for testing and debugging.
     * 
     * @return void
     */
    public static function reset(): void
    {
        self::$sharedContainer = null;
        self::$config = [
            'default_mode' => 'full',
            'shared_container' => true,
            'auto_shutdown' => true,
            'error_log_path' => '/storage/logs/app_errors.log'
        ];
    }
}